/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "SharpLCD.hpp"
#include "BLEDevice.h"
#include "icon.h"
//#include "font.h"
#include "dotfont.h"

Serial pc (USBTX,USBRX);

DigitalOut led1(P0_12);
DigitalOut motor(P0_1);
DigitalOut screen(P0_21);
DigitalIn  button(P0_16);
//a50,a60,a70,a80,a90,a100,a110,a120,a130,a140,a150,a160,a170

char strn[50]; // incoming tx data
char user[50]; // username to display
uint8_t* rings[2] = {a0,a10};
int set=0;
int set2=0;
int pairing =1;
int anim=0;
int elip=0;

int timeUpdate = 1;
time_t unixTime;
char timeStr[50]; // time to display
char dayStr[50]; // day to display
char dateStr[50]; // date to display
Timer mClock;
int clockOffset = 0;
int lastms = 0;

//SharpLCD(PinName enable, PinName cs, PinName mosi, PinName miso_unused, PinName sclk, PinName _unused = NC)
SharpLCD lcd(P0_0, P0_24, P0_20, P0_22, P0_25, P0_27);
//SharpLCD lcd(P0_25, P0_24, P0_23, P0_13, P0_22, P0_27);
uint8_t framebuffer[SharpLCD::SIZEOF_FRAMEBUFFER_FOR_ALLOC];
SharpLCD::FrameBuffer fb(framebuffer);
BLEDevice ble;
//UARTService uart(ble);

// The Nordic UART Service
static const uint8_t uart_base_uuid[] = {0x6e, 0x40, 0x00, 0x01, 0xb5, 0xa3, 0xf3, 0x93, 0xe0, 0xa9, 0xe5,0x0e, 0x24, 0xdc, 0xca, 0x9e};
static const uint8_t uart_tx_uuid[]   = {0x6e, 0x40, 0x00, 0x02, 0xb5, 0xa3, 0xf3, 0x93, 0xe0, 0xa9, 0xe5,0x0e, 0x24, 0xdc, 0xca, 0x9e};
static const uint8_t uart_rx_uuid[]   = {0x6e, 0x40, 0x00, 0x03, 0xb5, 0xa3, 0xf3, 0x93, 0xe0, 0xa9, 0xe5,0x0e, 0x24, 0xdc, 0xca, 0x9e};
static const uint8_t uart_base_uuid_rev[] = {0x9e, 0xca, 0xdc, 0x24, 0x0e, 0xe5, 0xa9, 0xe0, 0x93, 0xf3, 0xa3, 0xb5, 0x01, 0x00, 0x40, 0x6e};
uint8_t txPayload[50] = {0};
uint8_t rxPayload[50] = {0};
GattCharacteristic  txCharacteristic (uart_tx_uuid, txPayload, 1, 50,
                                      GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE);
GattCharacteristic  rxCharacteristic (uart_rx_uuid, rxPayload, 1, 50,
                                      GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);
GattCharacteristic *uartChars[] = {&txCharacteristic, &rxCharacteristic};
GattService         uartService(uart_base_uuid, uartChars, sizeof(uartChars) / sizeof(GattCharacteristic *));

const FONT_INFO* mainFont;
const FONT_INFO* exFont;
const FONT_INFO* subFont;

void screenSetup()
{
    screen=1;
    wait_ms(200);
    lcd.enableDisplay();
    fb.clear();
    lcd.clear();

    mainFont = searchFontFace("Square Head Black", 20);
    subFont = searchFontFace("Century Gothic Black", 9);
    exFont = searchFontFace("Lucida", 8);
}

void showPairing()
{
    fb.clear();
    fb.bitBlit(sScreen, 96, 96, 0, 0);
    fb.bitBlit(rings[anim], 32, 32, 2, 57); 
    char* dots[4] = {"", "..", "...", "..."};
    char str[15] = "Pairing";
    strcat(str, dots[elip]);
    
    fb.printString(exFont, 38, 80, str);
    
    lcd.drawFrameBuffer(fb);
    elip++;
    anim++;
    wait(0.15);
    
    if (anim>=2) {
        anim = 0;   
    }
    
    if (elip>=4) {
        elip = 0;   
    }
}

void disconnectionCallback(Gap::Handle_t handle, Gap::DisconnectionReason_t reason)
{
    ble.startAdvertising(); // restart advertising
}

void onDataWritten(const GattCharacteristicWriteCBParams *params)
{
    uint16_t txHandle = txCharacteristic.getValueAttribute().getHandle();
    uint16_t bytesRead;
    
    if (params->charHandle == txHandle) {
        memset(strn, 0, sizeof(strn));
        ble.readCharacteristicValue(txHandle, txPayload, &bytesRead);
        strncpy(strn,(const char*)txPayload,bytesRead);
        set=1;
        pc.printf ("\n\r Payload length: %d", bytesRead);
        /*pc.printf("\n\r Payload: ");
        for (int i=0;i<10;i++) {
            pc.printf("%c", (char)txPayload[i]);
        }*/
        pc.printf ("\n\r Received: %s", strn);
    }
}

void bleSetup()
{
    //Init ble
    ble.init();
    //and handlers...
    ble.onDisconnection((Gap::DisconnectionEventCallback_t)&disconnectionCallback);
    ble.onDataWritten(onDataWritten);
    // setup advertising
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                     (const uint8_t *)"YO!", sizeof("YO!") - 1);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                     (const uint8_t *)uart_base_uuid_rev, sizeof(uart_base_uuid));
    ble.setAdvertisingInterval(160); // 100ms; in multiples of 0.625ms.
    ble.startAdvertising();
    ble.addService(uartService);
}

void setTime(int newTime) {
    unixTime = newTime;
    struct tm* timeinfo = localtime (&unixTime);
    strftime (timeStr, 20, "%R", timeinfo);
    strftime (dayStr, 20, "%a", timeinfo);
    strftime (dateStr, 20, "%b %e", timeinfo);
}

// Main LCD display function, add all framebuffer updates here ONLY
void updateLCD() {
    fb.clear();
    fb.bitBlit(img, 96, 96, 0,0);
    
    fb.bitBlit(battery, 16, 16, 75, 0);
    
    fb.bitBlit(BLE, 16, 16, 0, 0);

    fb.printString(mainFont, 60, 65, user);
    
    fb.printString(mainFont, 5, 61, timeStr);
                   
    fb.printString(subFont, 14, 78, dayStr);
    
    fb.printString(subFont, 43, 78, dateStr);
                   
    lcd.drawFrameBuffer(fb);
}

int main(void) {

    screenSetup();
    bleSetup();
    
    led1 = 0;
    set2 = 0;

    uint16_t rxHandle = rxCharacteristic.getValueAttribute().getHandle();
    
    // Initial time update
    pc.printf("\n\r Updating time");
    char cmd[5] = "time";
    ble.updateCharacteristicValue(rxHandle, (const uint8_t*)cmd, 4);

    while (true) {
        
        if (pairing){
            showPairing();
        }
    
        if(set){
            if (timeUpdate) {
                setTime(atoi(strn));
                pc.printf("\n\r New time: %s", timeStr);
                timeUpdate = 0;
                pairing=0;
                mClock.start();
            } else {
                strncpy(user, strn, sizeof(strn));
            }
            set = 0;
        }
        
        if(!button && !set2){
            set2 = 1;
            uint16_t bytesToSend = strlen(user);
            memcpy(rxPayload, user, bytesToSend);
            ble.updateCharacteristicValue(rxHandle, rxPayload, bytesToSend);
            pc.printf ("\n\r Sending %s", user);
        } else if (button){
            set2 = 0;
        }

        clockOffset = mClock.read_ms();
        if (clockOffset >= 1000) {
            clockOffset += lastms;
            int addTime = unixTime + (clockOffset/1000);
            lastms = clockOffset%1000; // e.g. 2564 - 2000 = 564
            mClock.reset();
            setTime(addTime);
            
            // do all the framebuffer updating here every second
            updateLCD();
        }
    }
}