#include "mbed.h"
#include "DebouncedInterrupt.h"
#include "font.h"
#include "SharpLCD.hpp"
#include "icon.h"

//I/O Initialisation
DigitalOut myled(LED1);
DigitalOut motor(P0_23);
#define BUTTON_INTERRUPT_DEBOUNCE_TIME_MS 150
DebouncedInterrupt buttonOne (P0_16);
DebouncedInterrupt buttonTwo (P0_17);

SharpLCD lcd(P0_0, P0_24, P0_20, P0_22, P0_25, P0_27);

uint8_t framebuffer[SharpLCD::SIZEOF_FRAMEBUFFER_FOR_ALLOC];
SharpLCD::FrameBuffer fb(framebuffer);

//Notification Data struct
#define NOTIFICATION_TEXT_MAX_LENGTH 20
#define NOTIFICATION_TYPE_ALARM    0x01
#define NOTIFICATION_TYPE_CALL     0x02
#define NOTIFICATION_TYPE_SMS      0x03
#define NOTIFICATION_TYPE_EMAIL    0x04
#define NOTIFICATION_TYPE_EVENT    0x05
#define NOTIFICATION_STATE_DELETED 0x00
#define NOTIFICATION_STATE_UNREAD  0x01
#define NOTIFICATION_STATE_READ    0x02
struct Notification{
    uint8_t type;
    uint8_t state;
    char primary_text[NOTIFICATION_TEXT_MAX_LENGTH];
    char secondary_text[NOTIFICATION_TEXT_MAX_LENGTH];
    };

//notification storage
#define MAX_NOTIFICATIONS 20
Notification notifications[MAX_NOTIFICATIONS];
uint8_t notification_count = 0;
int visible_notification_index = -1;

//States
uint8_t needs_display_update = 0;
uint8_t has_unread = 0;

//For LED
#define LED_BLINK_TIME_MS 500
uint8_t led_on = 0;
Timer led_timer;

//For Buzzer
#define BUZZ_TIME_MS 200
#define MAX_BUZZ_COUNT 3
uint8_t buzz_count = MAX_BUZZ_COUNT;
Timer buzz_timer;

void setType(Notification* note, uint8_t type) {
    note->type = type;
}

uint8_t isCall(Notification note) {
    return note.type == NOTIFICATION_TYPE_CALL;
}

uint8_t isSMS(Notification note) {
    return note.type == NOTIFICATION_TYPE_SMS;
}

uint8_t isEmail(Notification note) {
    return note.type == NOTIFICATION_TYPE_EMAIL;
}

uint8_t isEvent(Notification note) {
    return note.type == NOTIFICATION_TYPE_EVENT;
}

void setState(Notification* note, uint8_t state) {
    note->state = state;
}

uint8_t isDeleted(Notification note) {
    return note.state == NOTIFICATION_STATE_DELETED;
}

uint8_t isUnread(Notification note) {
    return note.state == NOTIFICATION_STATE_UNREAD;
}

uint8_t isRead(Notification note) {
    return note.state == NOTIFICATION_STATE_READ;
}

//Layout Coordinates
#define TYPE_ICON_X         10 //5
#define TYPE_ICON_Y         60 //10
#define UNREAD_ICON_X       60 //96
#define UNREAD_ICON_Y       60 //14
#define PRIMARY_TEXT_X      10 //5
#define PRIMARY_TEXT_Y      30 //64
#define SECONDARY_TEXT_X    10 //5
#define SECONDARY_TEXT_Y    50 //82
//Update Display to show current notification...

//Serial  pc(USBTX, USBRX);

void doDisplayUpdate() {
    //pc.printf("display update"); //Debug
    lcd.clear();
    if(visible_notification_index >= 0) {
        //Write current notification...        
        //Draw type icon...
        switch(notifications[visible_notification_index].type) {
            case NOTIFICATION_TYPE_ALARM:
                fb.bitBlit(Alarm, 32, 32, TYPE_ICON_X, TYPE_ICON_Y);
                break;
                
            case NOTIFICATION_TYPE_CALL:
                fb.bitBlit(Call, 32, 32, TYPE_ICON_X, TYPE_ICON_Y);
                break;
                
            case NOTIFICATION_TYPE_SMS:
                fb.bitBlit(SMS, 32, 32, TYPE_ICON_X, TYPE_ICON_Y);
                break;
                
            case NOTIFICATION_TYPE_EMAIL:
                fb.bitBlit(Email, 32, 32, TYPE_ICON_X, TYPE_ICON_Y);
                break;
                
            case NOTIFICATION_TYPE_EVENT:
                fb.bitBlit(Calendar, 32, 32, TYPE_ICON_X, TYPE_ICON_Y);
                break;
        }
                
        //Unread notification
        if(isUnread(notifications[visible_notification_index])){
            fb.bitBlit(Unread, 16, 24, UNREAD_ICON_X, UNREAD_ICON_Y);
        }
        //Text fields...
        fb.printString(lookupFontFace("Lucida 8pt", 8), PRIMARY_TEXT_X, PRIMARY_TEXT_Y, BLACK, notifications[visible_notification_index].primary_text);
        fb.printString(lookupFontFace("Lucida 8pt", 8), SECONDARY_TEXT_X, SECONDARY_TEXT_Y, BLACK, notifications[visible_notification_index].secondary_text);
    } else {
        //no notifications...
        //Write current notification...
        //fb.printString(lookupFontFace("Lucida 8pt", 8), PRIMARY_TEXT_X, PRIMARY_TEXT_Y, BLACK, "No Notifications");
    }
    //epaper.write_disp();
    lcd.drawFrameBuffer(fb);
    needs_display_update = 0;
}

//Request a display update..
void requestDisplayUpdate() {
    needs_display_update = 1;
}

//Starts buzz pattern...
void startBuzz() {
    //Buzz
    buzz_count = 0;
    buzz_timer.reset();
    buzz_timer.start();
}


//Add Notification...
int addNotification(Notification note) {
    //Find insertion point...
    uint8_t index = 0;
    for(index = 0; index < MAX_NOTIFICATIONS; index++) {
        if(isDeleted(notifications[index])) {
            //Here...
            break;
        }
    }
    //If here, didn't find insertion point... wrap to beginning.
    if(index >= MAX_NOTIFICATIONS) index = 0;
    notifications[index] = note;
    //Set buzzer
    startBuzz();
    //Set unread
    if(isUnread(note)) has_unread = 1;
    return index;
}

void deleteNotification(int index) {
    setState(&notifications[index], NOTIFICATION_STATE_DELETED);
    //Shift array elements left...
    int i = 0;
    for (i = index + 1; i < MAX_NOTIFICATIONS; i++) {
        notifications[i - 1] = notifications[i];
    }
}

void checkUnread(){
    uint8_t i = 0;
    led_timer.stop();
    has_unread = 0;
    for(i = 0; i < MAX_NOTIFICATIONS; i++) {
        if(isUnread(notifications[i])) {
            has_unread = 1;
            led_timer.reset();
            led_timer.start();
            break;
        }
    }
}

//Button One Handler
void buttonOnePressed(){
    if(visible_notification_index >= 0) {
        //Increment index, wrap to beginning if last.
        visible_notification_index++;
        if(visible_notification_index >= MAX_NOTIFICATIONS || isDeleted(notifications[visible_notification_index])) {
            visible_notification_index = 0;
            if(isDeleted(notifications[visible_notification_index])) {
                //Still deleted... none...
                visible_notification_index = -1;
            }
        }
        //Trigger display update...
        requestDisplayUpdate();
    }
}

//Button Two handler
void buttonTwoPressed(){
    if(visible_notification_index >= 0) {
        if(!isDeleted(notifications[visible_notification_index])) {
            //Exists.
            if(isUnread(notifications[visible_notification_index])) {
                //Toggle to 'read'
                setState(&notifications[visible_notification_index], NOTIFICATION_STATE_READ);
                checkUnread();
            } else if (isRead(notifications[visible_notification_index])) {
                //Already 'read'... delete, this also shifts remaining notifications down...
                deleteNotification(visible_notification_index);
                //If current is deleted...
                if(isDeleted(notifications[visible_notification_index])) {
                    //We're at end, so wrap...
                    visible_notification_index = 0;
                    if(isDeleted(notifications[visible_notification_index])) {
                        //Still deleted... so there are none...
                        visible_notification_index = -1;
                    }
                }
                //Otherwise, we've got one, so we should be ok...
            }
            
            //Trigger display update...
            requestDisplayUpdate();
        }
    }
}

//Initialise notification data...
void initNotificationData() {
    //For debug purposes...
    
    notifications[0].type = NOTIFICATION_TYPE_ALARM;
    notifications[0].state = NOTIFICATION_STATE_UNREAD;
    //strcpy(notifications[0].primary_text,"Wake Up");
    //strcpy(notifications[0].secondary_text, "07:00");
    strcpy(notifications[0].primary_text,"Test");
    strcpy(notifications[0].secondary_text, "1");
    
    notifications[1].type = NOTIFICATION_TYPE_CALL;
    notifications[1].state = NOTIFICATION_STATE_UNREAD;
    //strcpy(notifications[1].primary_text,"Eric Gowland");
    //strcpy(notifications[1].secondary_text, "07770909177");
    strcpy(notifications[1].primary_text,"Test");
    strcpy(notifications[1].secondary_text, "2");
    
    notifications[2].type = NOTIFICATION_TYPE_SMS;
    notifications[2].state = NOTIFICATION_STATE_UNREAD;
    //strcpy(notifications[2].primary_text,"Hi, txt me...");
    //strcpy(notifications[2].secondary_text, "07770909177");
    strcpy(notifications[2].primary_text,"Test");
    strcpy(notifications[2].secondary_text, "3");
    
    notifications[3].type = NOTIFICATION_TYPE_EMAIL;
    notifications[3].state = NOTIFICATION_STATE_UNREAD;
    //strcpy(notifications[3].primary_text, "Dear Sir I have $US");
    //strcpy(notifications[3].secondary_text, "not@scam.net");
    strcpy(notifications[3].primary_text, "Test");
    strcpy(notifications[3].secondary_text, "4");
    
    notifications[4].type = NOTIFICATION_TYPE_EVENT;
    notifications[4].state = NOTIFICATION_STATE_UNREAD;
    //strcpy(notifications[4].primary_text,"Review Meeting");
    //strcpy(notifications[4].secondary_text, "10:00 - 10:30");
    strcpy(notifications[4].primary_text,"Test");
    strcpy(notifications[4].secondary_text, "5");
    
    //Set location, etc.
    visible_notification_index = 0;
    checkUnread();
    requestDisplayUpdate();
    startBuzz();
}

//Main Program Function
int main() {
    lcd.enableDisplay();
    //Init Data
    initNotificationData();
    //Attach interrupt handlers...
    buttonOne.attach(buttonOnePressed, IRQ_RISE, BUTTON_INTERRUPT_DEBOUNCE_TIME_MS);
    buttonTwo.attach(buttonTwoPressed, IRQ_RISE, BUTTON_INTERRUPT_DEBOUNCE_TIME_MS);
    //Request display update...
    requestDisplayUpdate();
    while(1) {
        if(needs_display_update){
            doDisplayUpdate();
        } else {
            //If state hasn't changed, just action timers...
            //Unread LED
            if(has_unread) {
                //LED flashing...
                if(led_timer.read_ms() > LED_BLINK_TIME_MS) {
                    myled = !myled;
                    led_timer.reset();
                }
            } else {
                myled = 0;
                led_timer.stop();
            }
            
            //Buzz
            if(buzz_count < MAX_BUZZ_COUNT) {
                //Buzzing...
                if(buzz_timer.read_ms() > BUZZ_TIME_MS) {
                    motor = !motor;
                    if(!motor) buzz_count++;
                    buzz_timer.reset();
                }
            } else {
                motor = 0;
                buzz_timer.stop();
            }
        }
    }
}