
#include "mbed.h"
#include "WS2812.h"
#include "PixelArray.h"

Serial pc(USBTX, USBRX);
DigitalOut led_green(LED_GREEN, 1);
AnalogIn sensor1(PTB2);
AnalogIn sensor2(A0);


#define WS2812_BUF 112
#define NUM_COLORS 6
#define NUM_LEDS_PER_COLOR 1

//-------- Colours -----------
#define RED 0x2f0000
#define YELLOW 0x2f2f00
#define GREEN 0x002f00 // Player has conceded a goal
#define LIGHTBLUE 0x002f2f
#define DARKBLUE 0x00002f // Player scored a goal
#define PINK 0x2f002f
#define OFF 0x000000
#define WHITE 0xffffff
#define ARMBLUE 0x128BAB

// Parameters
const int READ_FREQUENCY = 20;
const int BUFFER_SIZE = 2*READ_FREQUENCY; // in seconds
const float VOLTAGE_REF = 5.000f; // V
const int DEVIATION = 100; // mV
const int DEFAULT_DATA = -1;

bool seg1A, seg1B, seg1C, seg1D, seg1E, seg1F, seg1G;
int seg1Array[112];
int MainArray[11][122];

//void write_8Seg(int num);
PixelArray px(WS2812_BUF);

// See the program page for information on the timing numbers
// The given numbers are for the K64F
WS2812 ws(D9, WS2812_BUF, 0, 5, 5, 0);
WS2812 ws2(D8,WS2812_BUF, 0, 5, 5, 0);
WS2812 ws3(D7,WS2812_BUF, 0, 5, 5, 0);
WS2812 ws4(D6,WS2812_BUF, 0, 5, 5, 0);


// Score counters
int p1Score;
int p2Score;
int ScoreLimit = 5;
bool FinishedGame = false;
int EndFlashes = 0;
int NumFlashes;

// Create timers
Ticker pollTicker1;
Ticker pollTicker2;
Timer t;

int readings[BUFFER_SIZE];          // array of readings
int reading = 0;                    // index into above
int min = DEFAULT_DATA;
int max = DEFAULT_DATA;
int average = DEFAULT_DATA;

void readSensor();
void handleGoal();
void reset();

void setup()
{
    // Turn on green LED
    led_green = 1;

    // Fills 2D array with data
    for(int i=0; i<10; i++) {
        write_7Seg(i);
    }

    p1Score = 0;
    p2Score = 0;
    NumFlashes = 0;

    // Set LEDS to start values
    write_pbbuf(p1Score);
    write_pbbuf2(p2Score);
    write_goal1(WHITE);
    write_goal2(WHITE);
    wait_ms(1000);
    write_goal1(OFF);
    write_goal2(OFF);

    // Fill buffer with default data
    for (int i=0; i<BUFFER_SIZE; i++) {
        readings[i] = DEFAULT_DATA;
    }

    pollTicker1.attach(&readSensor1,(float)1/READ_FREQUENCY);
    pollTicker2.attach(&readSensor2,(float)1/READ_FREQUENCY);

    // Turn off green LED
    led_green = 0;
}

// Set segment variables
void write_7Seg(int num)
{
    switch (num) {
        case 0 :
            // Off
            seg1A = 1;
            seg1B = 1;
            seg1C = 1;
            seg1D = 1;
            seg1E = 1;
            seg1F = 1;
            seg1G = 0;

            setLEDArray(0);

            break;

        case 1 :
            // 1
            seg1A = 0;
            seg1B = 1;
            seg1C = 1;
            seg1D = 0;
            seg1E = 0;
            seg1F = 0;
            seg1G = 0;

            setLEDArray(1);
            break;

        case 2 :
            // 2
            seg1A = 1;
            seg1B = 1;
            seg1C = 0;
            seg1D = 1;
            seg1E = 1;
            seg1F = 0;
            seg1G = 1;

            setLEDArray(2);

            break;

        case 3 :
            // 3
            seg1A = 1;
            seg1B = 1;
            seg1C = 1;
            seg1D = 1;
            seg1E = 0;
            seg1F = 0;
            seg1G = 1;

            setLEDArray(3);
            break;

        case 4:
            // 4
            seg1A = 0;
            seg1B = 1;
            seg1C = 1;
            seg1D = 0;
            seg1E = 0;
            seg1F = 1;
            seg1G = 1;

            setLEDArray(4);

            break;

        case 5:
            // 5
            seg1A = 1;
            seg1B = 0;
            seg1C = 1;
            seg1D = 1;
            seg1E = 0;
            seg1F = 1;
            seg1G = 1;

            setLEDArray(5);

            break;

        case 6:
            // 6
            seg1A = 1;
            seg1B = 0;
            seg1C = 1;
            seg1D = 1;
            seg1E = 1;
            seg1F = 1;
            seg1G = 1;

            setLEDArray(6);

            break;

        case 7:
            // 7
            seg1A = 1;
            seg1B = 1;
            seg1C = 1;
            seg1D = 0;
            seg1E = 0;
            seg1F = 0;
            seg1G = 0;

            setLEDArray(7);

            break;

        case 8:
            // 8
            seg1A = 1;
            seg1B = 1;
            seg1C = 1;
            seg1D = 1;
            seg1E = 1;
            seg1F = 1;
            seg1G = 1;

            setLEDArray(8);
            break;

        case 9:
            // 9
            seg1A = 1;
            seg1B = 1;
            seg1C = 1;
            seg1D = 0;
            seg1E = 0;
            seg1F = 1;
            seg1G = 1;

            setLEDArray(9);
            break;

        case 10:
            // OFF
            seg1A = 0;
            seg1B = 0;
            seg1C = 0;
            seg1D = 0;
            seg1E = 0;
            seg1F = 0;
            seg1G = 0;

            setLEDArray(10);
            break;

        default :

            break;

    }
}

// Write segment config to main array
void setLEDArray(int x)
{
    for (int i = 0; i < WS2812_BUF; i++) {
        if (i < 16) {
            MainArray[x][i] = seg1A;
        }

        if ( i >= 16 && i < 32) {
            MainArray[x][i] = seg1B;
        }

        if (i >= 32 && i < 48) {
            MainArray[x][i] = seg1C;
        }

        if (i >= 48 && i < 64) {
            MainArray[x][i]= seg1D;
        }

        if ( i >= 64 && i < 80) {
            MainArray[x][i] = seg1E;
        }

        if (i >= 80 && i < 96) {
            MainArray[x][i] = seg1F;
        }

        if ( i >= 96 && i < 112) {
            MainArray[x][i] = seg1G;
        }
    }// FOR LOOP
}

// Write to player 1 LED (ROBOT)
void write_pbbuf(int line_num)
{
    write_px(line_num);
    ws.write(px.getBuf());
}

// Write to player 2 LED (HUMAN)
void write_pbbuf2(int line_num)
{
    write_px(line_num);
    ws2.write(px2.getBuf());
}

// Write pixel array
void write_px(int line_num)
{
    for (int i = 0; i < WS2812_BUF; i++) {
        if (MainArray[line_num][i] == 0) {
            px.Set(i,OFF);
        }

        if (MainArray[line_num][i] == 1) {
            px.Set(i,LIGHTBLUE);
        }
    }
}

// Read goal sensor 1 (ROBOT)
void readSensor1()
{
    // Convert readings to mV
    readings[reading] = (int)(sensor1.read() * 1000 * VOLTAGE_REF);
    analyseReadings(0);
}

// Read goal sensor 2 (HUMAN)
void readSensor2()
{
    // Convert readings to mV
    readings[reading] = (int)(sensor2.read() * 1000 * VOLTAGE_REF);
    analyseReadings(1);
}

// Analyse sensor readings
void analyseReadings(bool player)
{
    min = DEFAULT_DATA;
    max = DEFAULT_DATA;
    int sum = 0;
    int count = 0;

    for (int i=0; i<BUFFER_SIZE; i++) {
        if (readings[i] == DEFAULT_DATA) {
            // ignore default values
            break;
        }

        if ((min == DEFAULT_DATA) || (readings[i] < min)) {
            min = readings[i];
        }

        if ((max == DEFAULT_DATA) || (readings[i] > max)) {
            max = readings[i];
        }

        sum += readings[i];
        count++;
    }

    if (count == 0) {
        // No data yet
        return;
    }

    average = sum/count;

    // Goal detected if current reading deviates from average by more than 'deviation'
    if ((readings[reading] < (average - DEVIATION)) || (readings[reading] > (average + DEVIATION))) {
        incrementScore(!player); // Increase player score opposite the player whose goal's sensor reported a goal
    }

    reading = (reading + 1) % BUFFER_SIZE;
}

// Increment score
void incrementScore(bool isPlayer1)
{
    // Player 1 sensor reported goal, increase player 2 score
    if(isPlayer1 == 1) {
        if(p2Score < ScoreLimit) {
            p2Score++;
            write_goal(GREEN, isPlayer1);
            write_goal(DARKBLUE, !isPlayer1);
            t.start();
        } else if(Score == ScoreLimit) {
            FinishedGame = true;
        }

        write_pbbuf2(p2Score);

    }

    // Player 2 sensor reported goal, increase player 1 score
    if(isPlayer1 == 0) {
        if(p1Score < ScoreLimit) {
            p1Score++;
            write_goal(DARKBLUE,isPlayer1);
            write_goal(GREEN,!isPlayer1);
            t.start();
        } else if(p1Score == ScoreLimit) {
            FinishedGame = true;
        }

        write_pbbuf(p1Score);
    }
    t.start();
}

// Write goal LED
void write_goal(unsigned int Colour, bool player1Goal)
{
    for (int i = 0; i < WS2812_BUF; i++) {
        px.Set(i,Colour);
    }

    if(player1Goal == 1) {
        ws4.write(px.getBuf());
    } else if (player1Goal == 0) {
        ws3.write(px.getBuf());
    }

}

// Turn off goal LEDS
void turn_off_goal(unsigned int Colour)
{
    for (int i = 0; i < WS2812_BUF; i++) {
        px.Set(i,Colour);
    }
    ws4.write(px.getBuf());

    for (int i = 0; i < WS2812_BUF; i++) {
        px.Set(i,Colour);
    }
    ws4.write(px.getBuf());
}


int main()
{
    // Set brightness of the 4 LED strips
    ws.setII(0xFF);
    ws.useII(WS2812::GLOBAL);
    ws2.setII(0xFF);
    ws2.useII(WS2812::GLOBAL);
    ws3.setII(0xFF);
    ws3.useII(WS2812::GLOBAL);
    ws4.setII(0xFF);
    ws4.useII(WS2812::GLOBAL);
    setup();

    while(1) {
        wait(1.0f);
        // Check timer to see if 3s or 5s has elapsed
        if(t.read() >= 3.0f && FinishedGame == false) {
            turn_off_goal(OFF);
            t.stop();
            t.reset();
        } else if(t.read() >= 5.0f && FinishedGame == true) {
            /* Some finished game code here */
            t.stop();
            t.reset();
        }
    }
}





