/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "mbed.h"
#include "ble/BLE.h"
#include "ButtonService.h"

Serial pc(USBTX, USBRX);
DigitalOut  led1(LED1);     //sets led1 to digital out for connection status indicator
DigitalOut  led2(LED2);     //sets led1 to digital out for seat status indicator
InterruptIn button(P0_7);   //sets Pin 7 to an interrupt enabled input for chair status updating

const static char     DEVICE_NAME[] = "Seat Sensor";        //specifies name of the device to be found
static const uint16_t uuid16_list[] = {ButtonService::BUTTON_SERVICE_UUID};

Ticker ticker;      //defines a ticker for periodic callback for connection indicator
Timer debounceTimer;        //defines a timer for the debouncing loop
int pin = P0_7;
BLE ble;

enum {      //an enumerator defining the three possible output states. Chair empty(RELEASED) =0. Chair OCCUPIED(PRESSED) = 1. Chair default(IDLE) = 2.
    RELEASED = 0,
    PRESSED,
    IDLE
};

static uint8_t buttonState = IDLE;      //initialised buttonState(Chair state) to the default(IDLE)
static ButtonService *buttonServicePtr;

/*function is called from the ISR when the chair becomes occupied, resetting the debounce timer and changing button state*/
void buttonPressedCallback(void)
{

    debounceTimer.reset();
    buttonState = PRESSED;

}
/*function is called from the ISR when the chair becomes empty, resetting the debounce timer and changing button state*/
void buttonReleasedCallback(void)
{

    debounceTimer.reset();
    buttonState = RELEASED;

}


/* upon disconnecting advertising process is repeated */
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{

    BLE::Instance().gap().startAdvertising();

}

/* on connection state is outputted updating firefly */
void connectionCallback(const Gap::ConnectionCallbackParams_t *params)
{
    
       // buttonServicePtr->updateButtonState(button.read());
        
    
}

/* periodic callback for indicating state of the board */
void periodicCallback(void)
{
    if(ble.getGapState().connected == 0) {
        led1=!led1;
    } else {
        led1= 1;
    }

}

/**
 * This function is called when the ble initialization process has failled
 */
void onBleInitError(BLE &ble, ble_error_t error)
{
    /* Initialization error handling should go here */
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }


    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);
    ble.gap().onConnection(connectionCallback);

    /* Setup primary service */
    buttonServicePtr = new ButtonService(ble, false /* initial value for button pressed */);

    /* setup advertising */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(1000); /* 1000ms. */
    ble.gap().startAdvertising();

}

int main(void)
{

    debounceTimer.start();      //initialises the debounce timer
    ticker.attach(periodicCallback, 1); // periodic ticker indicating the board it setup and running
    button.fall(buttonPressedCallback); //interrupt call for chair occupied
    button.rise(buttonReleasedCallback);//interrupt call for chair empty

    BLE &ble = BLE::Instance();
    ble.init(bleInitComplete);

     
    while (ble.hasInitialized()  == false) 
    {
        /* spin loop */
    }
    
    /* updates state to the firefly upon mbed board booting */
    if (button.read() == 1) {
        buttonServicePtr->updateButtonState(1);
        led2 = !button.read();
    } else {
        buttonServicePtr->updateButtonState(0);
        led2 = !button.read();
    }

    /*main while loop*/
    while (true) {
        
        /*this is the debouncing loop, debounce only occurs upon state change from occupied 
        to empty so as to detect and ignore temporary shifting in the occupants position causing the system to believe the state has changed */
        
        while(buttonState != IDLE) 
        {
            if((buttonState == PRESSED)&&(debounceTimer.read_ms() > 200))
            {

                led2=buttonState;
                buttonServicePtr->updateButtonState(buttonState);
                buttonState = IDLE;

            }

            else if ((buttonState == RELEASED) && (debounceTimer.read_ms()>2000)) //debounce timer set to 2 seconds, can be adjusted for more realtime or reliable updates
            {

                led2=buttonState;
                buttonServicePtr->updateButtonState(buttonState);
                buttonState = IDLE;

            }
        }

        ble.waitForEvent();

    }
}


