#include "main.h"

int main()
{
    Setup();
    
    pc.printf("\nSETUP(IDLE)");

    while(1) {
        
        if (idle_flag == 0) { // IDLE
            
            IdleStateProtocol();
            
        }
        
        if (idle_flag == 1) { // PLAY
            
            PlayStateProtocol();
        
        }
        
        wait(0.02);
        
    }
    
}

void Setup()
{
    // Turn on green LED
    led_green = 0;
    
    // enable the internal pull-down resistor on idleButton
    idleButton.mode(PullDown);
    
    idleButton.attach_asserted(&idleButtonISR);

    idleButton.setSampleFrequency( 10000 );

    // Set brightness of the 4 LED strips
    robotScoreLED.setII(0x30);
    robotScoreLED.useII(WS2812::GLOBAL);
    playerScoreLED.setII(0x30);
    playerScoreLED.useII(WS2812::GLOBAL);
    
    // Set scores to 0
    robotScore = 0;
    playerScore = 0;
    numFlashes = 0;
    
    memset(mainArray, 0, sizeof mainArray);
        
    // Turn off green LED
    led_green = 1;
    
}

void IdleStateProtocol()
{
    if (idle_button_pressed == 1) { // button just pressed

        pc.printf("\nIDLE");

        PlayToIdleTransition();

    }

    FigureOf8Animation(true,true,true,true,1);
    CircleAnimation(true,true,true,true,1);
    
}

void PlayStateProtocol()
{
    if (idle_button_pressed == 1) { // button just pressed

        pc.printf("\nPLAY");
        
        if (previous_state == 0) {
            
            IdleToPlayTransition();
            
        }
        else {
        
            idle_button_pressed = 0;

            DrainAnimation(true,false,true,false);

            wait(1);

            robotScore = 0;

            playerScore = 0;

            WriteScores();
        
        }

    }
    
    double rbbValue = robotBreakBeam; // Read Robot Break beam
    double pbbValue = playerBreakBeam; // Read Player Break beam

    // IF PLAYER HAS SCORED A GOAL
    if (((prevRbbValue - rbbValue) > 0.03) || (PB1==0)) {
        pc.printf("Player has scored a goal \n\r");
        HandleGoal(false);
    }

    // IF ROBOT HAS SCORED A GOAL
    if (((prevPbbValue - pbbValue) > 0.03) || (PB2==0)) {
        pc.printf("Robot has scored a goal \n\r");
        HandleGoal(true);
    }

    prevRbbValue = rbbValue;
    prevPbbValue = pbbValue;

    pc.printf("PlayerGoal: %f, RobotGoal: %f \r\n",pbbValue,rbbValue);
    pc.printf("Player: %i v %i : Robot \r\n",playerScore,robotScore);
    
}

void IdleToPlayTransition()
{
    // reset scores
    robotScore = 0;
    playerScore = 0;
    
    // reset array
    memset(mainArray, 0, sizeof mainArray);
    
    // form closed loop of LEDs around exterior of display in opposite directions for robot and player scores
    for (int i = 9; i < 115; i++) { 

        if (i < 104) {
            mainArray[0][i] = 1;
        }
        if (i > 103) {
            mainArray[0][i-104] = 1;
        }
        if (i < 19) {
            mainArray[1][18-i] = 1;
        }
        if (i > 18) {
            mainArray[1][122-i] = 1;
        }

        wait_ms(10);

        WritePxAnimation(0,false,true);
        playerScoreLED.write(playerScorePx.getBuf());
        WritePxAnimation(1,true,true);
        robotScoreLED.write(robotScorePx.getBuf());

    }
    
    // drain closed loop away
    DrainAnimation(true,true,true,true);
    
    wait(1);
    
    // flash 0-0 on display
    WriteScores();
    
    idle_button_pressed = 0;

}

void PlayToIdleTransition()
{
    // reset scores and main array
    robotScore = 0;
    playerScore = 0;
    memset(mainArray, 0, sizeof mainArray);
    
    FigureOf8Animation(true,true,true,true,1);
        
    idle_button_pressed = 0;
    
}

// set segment number patterns
void SetNumberPatterns()
{
    for (int num = 0; num < 11; num++) {

        switch (num) {
            case 0 :
                // 0
                seg1A = 1;
                seg1B = 1;
                seg1C = 1;
                seg1D = 1;
                seg1E = 1;
                seg1F = 1;
                seg1G = 0;

                SetLEDArray(0);
                break;

            case 1 :
                // 1
                seg1A = 0;
                seg1B = 1;
                seg1C = 1;
                seg1D = 0;
                seg1E = 0;
                seg1F = 0;
                seg1G = 0;

                SetLEDArray(1);
                break;

            case 2 :
                // 2
                seg1A = 1;
                seg1B = 1;
                seg1C = 0;
                seg1D = 1;
                seg1E = 1;
                seg1F = 0;
                seg1G = 1;

                SetLEDArray(2);
                break;

            case 3 :
                // 3
                seg1A = 1;
                seg1B = 1;
                seg1C = 1;
                seg1D = 1;
                seg1E = 0;
                seg1F = 0;
                seg1G = 1;

                SetLEDArray(3);
                break;

            case 4:
                // 4
                seg1A = 0;
                seg1B = 1;
                seg1C = 1;
                seg1D = 0;
                seg1E = 0;
                seg1F = 1;
                seg1G = 1;

                SetLEDArray(4);
                break;

            case 5:
                // 5
                seg1A = 1;
                seg1B = 0;
                seg1C = 1;
                seg1D = 1;
                seg1E = 0;
                seg1F = 1;
                seg1G = 1;

                SetLEDArray(5);
                break;

            case 6:
                // 6
                seg1A = 1;
                seg1B = 0;
                seg1C = 1;
                seg1D = 1;
                seg1E = 1;
                seg1F = 1;
                seg1G = 1;

                SetLEDArray(6);
                break;

            case 7:
                // 7
                seg1A = 1;
                seg1B = 1;
                seg1C = 1;
                seg1D = 0;
                seg1E = 0;
                seg1F = 0;
                seg1G = 0;

                SetLEDArray(7);
                break;

            case 8:
                // 8
                seg1A = 1;
                seg1B = 1;
                seg1C = 1;
                seg1D = 1;
                seg1E = 1;
                seg1F = 1;
                seg1G = 1;

                SetLEDArray(8);
                break;

            case 9:
                // 9
                seg1A = 1;
                seg1B = 1;
                seg1C = 1;
                seg1D = 0;
                seg1E = 0;
                seg1F = 1;
                seg1G = 1;

                SetLEDArray(9);
                break;

            case 10:
                // OFF
                seg1A = 0;
                seg1B = 0;
                seg1C = 0;
                seg1D = 0;
                seg1E = 0;
                seg1F = 0;
                seg1G = 0;

                SetLEDArray(10);
                break;

            default :

                break;

        }

    }

}

// Write segment config to main array
void SetLEDArray(int x)
{
    for (int i = 0; i < WS2812_BUF; i++) {
        if (i < 18) {
            mainArray[x][i] = seg1A; 
        }

        if ( i >= 18 && i < 35) {
            mainArray[x][i] = seg1B;
        }

        if (i >= 35 && i < 52) {
            mainArray[x][i] = seg1C;
        }

        if (i >= 52 && i < 70) {
            mainArray[x][i]= seg1D;
        }

        if ( i >= 70 && i < 87) {
            mainArray[x][i] = seg1E;
        }

        if (i >= 87 && i < 104) {
            mainArray[x][i] = seg1F;
        }

        if ( i >= 104 && i < 122) {
            mainArray[x][i] = seg1G;
        }
    }
}

// Update both Score LEDs with the current score
void WriteScores()
{
    SetNumberPatterns();
    WritePxAnimation(playerScore,false,false);
    playerScoreLED.write(playerScorePx.getBuf());
    wait(0.01);
    WritePxAnimation(robotScore,true,false);
    robotScoreLED.write(robotScorePx.getBuf());
}

// Write Pixel array with either random colour pattern or in standard blue
void WritePxAnimation(int line_num,bool isRobot,bool colour)
{
    if (colour == true) { // random colours
        if(isRobot == true) {

            for (int i = 0; i < WS2812_BUF; i++) {
                if (mainArray[line_num][i] == 0) {
                    robotScorePx.Set(i,OFF);
                }

                if (mainArray[line_num][i] == 1) {
                    robotScorePx.Set(i,rand_colors[rand() % 10]);
                }
            }
        } else {
            for (int i = 0; i < WS2812_BUF; i++) { 
                if (mainArray[line_num][i] == 0) {
                    playerScorePx.Set(i,OFF);
                }

                if (mainArray[line_num][i] == 1) {
                    playerScorePx.Set(i,rand_colors[rand() % 10]);
                }
            }
        }
    } else { // blue
        if(isRobot == true) {

            for (int i = 0; i < WS2812_BUF; i++) {
                if (mainArray[line_num][i] == 0) {
                    robotScorePx.Set(i,OFF);
                }

                if (mainArray[line_num][i] == 1) {
                    robotScorePx.Set(i,BLUE);
                }
            }
        } else {
            for (int i = 0; i < WS2812_BUF; i++) {
                if (mainArray[line_num][i] == 0) {
                    playerScorePx.Set(i,OFF);
                }

                if (mainArray[line_num][i] == 1) {
                    playerScorePx.Set(i,BLUE);
                }
            }
        }

    }
}    

// Handle behaviour for the goal scored event
void HandleGoal(bool hasRobotScored)
{
    GoalAnimation(hasRobotScored);

    // If either player reaches the score limit, then call HandleWin()
    if(robotScore == scoreLimit || playerScore == scoreLimit) {
        HandleWin();
    } 
}

// Handle behaviour when either the player or robot has won a game. 
void HandleWin()
{
    // Initalise variable as true
    bool isRobotWinner = true;
    
    // Decide if the robot is the winner. If not, then change isRobotWinner to false
    if(playerScore == scoreLimit) {
        
        isRobotWinner = false;
        pc.printf("Player won!");
        WinAnimation(isRobotWinner);

    } else {
        pc.printf("Robot won!");
        WinAnimation(isRobotWinner);

    }

    // Reset scores then write to LEDs
    robotScore = 0;
    playerScore = 0;
    
    // IDLE state
    idle_flag = 0;
    
    idle_button_pressed = 0; // allow interrupt to fire
    
    memset(mainArray, 0, sizeof mainArray);
    
}

// animation for when a goal is scored
void GoalAnimation(bool hasRobotScored)
{
    if(hasRobotScored == true) {
        
        // clear current score 
        switch (robotScore) {
            case 0:   
                ZeroOutAnimation(true,false,false,false);
                break;
            case 1: 
                OneOutAnimation(true,false,false,false);
                break;
            case 2: 
                TwoOutAnimation(true,false,false,false);
                break;
            case 3: 
                ThreeOutAnimation(true,false,false,false);
            default : 
                break;   
        }
        
        // increment score
        robotScore++;
        wait_ms(250);
        
        // write new score
        switch (robotScore) {
            case 0: 
                ZeroInAnimation(true,false,false,false);
                break;
            case 1: 
                OneInAnimation(true,false,false,false);
                break;
            case 2: 
                TwoInAnimation(true,false,false,false);
                break;
            case 3: 
                ThreeInAnimation(true,true,false,false); // with random colour pattern
            default : 
                break;   
        }
        
    } else {
        
        // clear current score 
        switch (playerScore) {
            case 0: 
                ZeroOutAnimation(false,false,true,false);
                break;
            case 1: 
                OneOutAnimation(false,false,true,false);
                break;
            case 2: 
                TwoOutAnimation(false,false,true,false);
                break;
            case 3: 
                ThreeOutAnimation(false,false ,true,false);
            default : 
                break;   
        }
        
        // increment score
        playerScore++;
        wait_ms(250);
        
        // write new score
        switch (playerScore) {
            case 0: 
                ZeroInAnimation(false,false,true,false);
                break;
            case 1: 
                OneInAnimation(false,false,true,false);
                break;
            case 2: 
                TwoInAnimation(false,false,true,false);
                break;
            case 3: 
                ThreeInAnimation(false,false,true,true); // with random colour pattern
            default : 
                break;   
        }
    }
    
}

// animation when player or robot win the game
void WinAnimation(bool isRobotWinner)
{
    if(isRobotWinner == false) {
        
        // dim brightness of the robot score because it lost
        robotScoreLED.setII(0x10);
        robotScoreLED.useII(WS2812::GLOBAL);
        robotScoreLED.write(robotScorePx.getBuf()); // rewrite buffer to change brightness
                
        for (int i = 0; i < 80; i++) { // flash player score with changing random colours
            WritePxAnimation(playerScore,false,true);
            playerScoreLED.write(playerScorePx.getBuf());
            wait_ms(50);
        }     
        
        DrainAnimation(true,false,true,true); // drain both scores away with the player score in random colours
        
        // reset robot score brightness
        robotScoreLED.setII(0x30);
        robotScoreLED.useII(WS2812::GLOBAL);
    
    } else {
        
        // dim brightness of the player score because they lost
        playerScoreLED.setII(0x10);
        playerScoreLED.useII(WS2812::GLOBAL);
        playerScoreLED.write(playerScorePx.getBuf()); // rewrite buffer to change brightness
        
        for (int i = 0; i < 80; i++) { // flash robot score with changing random colours
            WritePxAnimation(robotScore,true,true);
            robotScoreLED.write(robotScorePx.getBuf());
            wait_ms(50);                 
        }
        
        DrainAnimation(true,true,true,false); // drain both scores away with the robot score in random colours
        
        // reset player score brightness
        playerScoreLED.setII(0x30);
        playerScoreLED.useII(WS2812::GLOBAL);
    }
    
    wait(3);
    
} 

// LEDs snake in a figure of 8 pattern in a group of 10
void FigureOf8Animation(bool robot, bool robotColour, bool player, bool playerColour, int numberOfRepititions)
{        
    for (int loops = 0; loops < numberOfRepititions; loops++) {

        for (int i = 0; i < 9; i++) { // loop for LED snake to separate from centre of segment A
            
            mainArray[0][9+i] = 1;
            mainArray[1][8-i] = 1;
            if (i < 9) {
                mainArray[0][i] = 0;
                mainArray[1][17-i] = 0;
            } 
            wait_ms(10);

            if (player == true) {
                WritePxAnimation(0,false,playerColour);
                playerScoreLED.write(playerScorePx.getBuf());
            }
            if (robot == true) {
                WritePxAnimation(1,true,robotColour);
                robotScoreLED.write(robotScorePx.getBuf());
            } 
        }
        
        for (int i = 0; i < 17; i++) { // loop for LED snake to go down segments B and F
            
            mainArray[0][18+i] = 1;
            mainArray[1][103-i] = 1;
            mainArray[0][9+i] = 0;
            if (i < 9) {
                mainArray[1][8-i] = 0;
            } else {
                mainArray[1][112-i] = 0;
            }
            wait_ms(10);
            
            if (player == true) {
                WritePxAnimation(0,false,playerColour);
                playerScoreLED.write(playerScorePx.getBuf());
            }
            if (robot == true) {
                WritePxAnimation(1,true,robotColour);
                robotScoreLED.write(robotScorePx.getBuf());
            }
        }
        
        for (int i = 0; i < 18; i++) { // loop for LED snake to pass through segment G
            
            mainArray[0][121-i] = 1;
            mainArray[1][104+i] = 1;
            if (i < 9) {
                mainArray[0][26+i] = 0;
                mainArray[1][95-i] = 0;
            } else {
                mainArray[0][130-i] = 0;
                mainArray[1][95+i] = 0;
            }
           wait_ms(10);
            
            if (player == true) {
                WritePxAnimation(0,false,playerColour);
                playerScoreLED.write(playerScorePx.getBuf());
            }
            if (robot == true) {
                WritePxAnimation(1,true,robotColour);
                robotScoreLED.write(robotScorePx.getBuf());
            }
        }
        
        for (int i = 0; i < 17; i++) { // loop for LED snake to pass down through segments E and C
            
            mainArray[0][86-i] = 1;
            mainArray[1][35+i] = 1;
            if (i < 9) {
                mainArray[0][112-i] = 0;
                mainArray[1][113+i] = 0;
            } else {
                mainArray[0][95-i] = 0;
                mainArray[1][26+i] = 0;
            }
            wait_ms(10);
            
            if (player == true) {
                WritePxAnimation(0,false,playerColour);
                playerScoreLED.write(playerScorePx.getBuf());
            }
            if (robot == true) {
                WritePxAnimation(1,true,robotColour);
                robotScoreLED.write(robotScorePx.getBuf());
            }
        }
        
        for (int i = 0; i < 35; i++) { // loop for LED snake to pass through segments D and E AND D and C
            
            mainArray[0][69-i] = 1;
            mainArray[1][52+i] = 1;
            mainArray[0][80-i] = 0;
            mainArray[1][43+i] = 0;
      
            wait_ms(10);
            
            if (player == true) {
                WritePxAnimation(0,false,playerColour);
                playerScoreLED.write(playerScorePx.getBuf());
            }
            if (robot == true) {
                WritePxAnimation(1,true,robotColour);
                robotScoreLED.write(robotScorePx.getBuf());
            }
        }
        
        for (int i = 0; i < 18; i++) { // loop for LED snake to pass through segment G
            
            mainArray[0][121-i] = 1;
            mainArray[1][104+i] = 1;
            if (i < 9) { ////////////////
                mainArray[0][45-i] = 0;
                mainArray[1][78+i] = 0;
                if (i == 8) {
                    mainArray[0][36] = 0;
                    mainArray[0][35] = 0;
                }
            } else {
                mainArray[0][130-i] = 0;
                mainArray[1][95+i] = 0;
            }
            wait_ms(10);
            
            if (player == true) {
                WritePxAnimation(0,false,playerColour);
                playerScoreLED.write(playerScorePx.getBuf());
            }
            if (robot == true) {
                WritePxAnimation(1,true,robotColour);
                robotScoreLED.write(robotScorePx.getBuf());
            }
        }
        
        for (int i = 0; i < 17; i++) { // loop for LED snake to go up segments B and F
            
            mainArray[0][87+i] = 1;
            mainArray[1][34-i] = 1;
            if (i < 9) {
                mainArray[0][112-i] = 0;
                mainArray[1][113+i] = 0;
            } else {
                mainArray[0][78+i] = 0;
                mainArray[1][43-i] = 0;
            }
            wait_ms(10);
            
            if (player == true) {
                WritePxAnimation(0,false,playerColour);
                playerScoreLED.write(playerScorePx.getBuf());
            }
            if (robot == true) {
                WritePxAnimation(1,true,robotColour);
                robotScoreLED.write(robotScorePx.getBuf());
            }
        }
        
        for (int i = 0; i < 9; i++) { // loop for LED snake to pass through the centre of segment A
            
            mainArray[0][i] = 1;
            mainArray[1][17-i] = 1;
            if (i < 9) {
                mainArray[0][95+i] = 0;
                mainArray[1][26-i] = 0;
            } else {
                mainArray[0][i-9] = 0;
                mainArray[1][26-i] = 0;
            }
            wait_ms(10);

            if (player == true) {
                WritePxAnimation(0,false,playerColour);
                playerScoreLED.write(playerScorePx.getBuf());
            }
            if (robot == true) {
                WritePxAnimation(1,true,robotColour);
                robotScoreLED.write(robotScorePx.getBuf());
            } 
        }
        
        
    }
    
}  

// LEDs snake in a figure of 8 pattern in a group of 10
void CircleAnimation(bool robot, bool robotColour, bool player, bool playerColour, int numberOfRepititions)
{        
    for (int loops = 0; loops < numberOfRepititions; loops++) {

        for (int i = 9; i < 115; i++) { // loop to set LEDs around the circle from start of segment A to end of Segment F
            
            if (i < 104) {
                mainArray[0][i] = 1;
                for (int j = 0; j < i-10; j++) { // Loop to clear previously set LEDs so that the chain that animates round the display is 10 LEDs long
                    mainArray[0][j] = 0;
                }
            }
            if (i > 103) {
                mainArray[0][i-104] = 1;
                for (int j = 90; j < i-10; j++) { // Loop to clear previously set LEDs so that the chain that animates round the display is 10 LEDs long
                    mainArray[0][j] = 0;
                }
            }
            if (i < 19) {
                mainArray[1][18-i] = 1;
                for (int j = 17; j > 25-i; j--) { // Loop to clear previously set LEDs so that the chain that animates round the display is 10 LEDs long
                    mainArray[1][j] = 0;
                }
            }
            if (i > 18) {
                
                mainArray[1][122-i] = 1;
                
                if (i < 29) {
                    for (int j = 9; j > 27-i; j--) { // Loop to clear previously set LEDs so that the chain that animates round the display is 10 LEDs long
                        mainArray[1][j] = 0;
                    }
                }
                if (i > 28) {
                    for (int j = 103; j > 131-i; j--) { // Loop to clear previously set LEDs so that the chain that animates round the display is 10 LEDs long
                        mainArray[1][j] = 0;
                    }                        
                }
            }
            
            wait_ms(10);

            if (player == true) {
                WritePxAnimation(0,false,playerColour);
                playerScoreLED.write(playerScorePx.getBuf());
            }
            if (robot == true) {
                WritePxAnimation(1,true,robotColour);
                robotScoreLED.write(robotScorePx.getBuf());
            }
            
        }
        
    }
    
}  

// animation that drains the display from the top segment A to the bottom segment D
void DrainAnimation(bool robot, bool robotColour,bool player, bool playerColour)
{
    SetNumberPatterns();
    
    for (int i = 0; i < 10; i++) { // drain A from the centre of the segment
        mainArray[robotScore][9+i] = 0;
        mainArray[robotScore][9-i] = 0;
        mainArray[playerScore][9+i] = 0;
        mainArray[playerScore][9-i] = 0;
        wait_ms(30);
        if (player == true){
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        if (robot == true) {
        WritePxAnimation(robotScore,true,robotColour);
        robotScoreLED.write(robotScorePx.getBuf());
        }
    }
    for (int i = 0; i < 18; i++) { // drain B and F downwards
        mainArray[robotScore][18+i] = 0;
        mainArray[robotScore][103-i] = 0;
        mainArray[playerScore][18+i] = 0;
        mainArray[playerScore][103-i] = 0;
        wait_ms(30);
        if (player == true){
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        if (robot == true) {
        WritePxAnimation(robotScore,true,robotColour);
        robotScoreLED.write(robotScorePx.getBuf());
        }
    }
    if (robotScore != 0 | playerScore != 0) { // endgame - at least one score is three

        for (int i = 0; i < 18; i++) { // drain G across
            mainArray[robotScore][104+i] = 0;
            mainArray[playerScore][104+i] = 0;
            if (player == true) {
                WritePxAnimation(playerScore,false,playerColour);
                playerScoreLED.write(playerScorePx.getBuf());
            }
            if (robot == true) {
                WritePxAnimation(robotScore,true,robotColour);
                robotScoreLED.write(robotScorePx.getBuf());
            }
        }
    } else { // start of game - both scores are 0
        for (int i = 0; i < 18; i++) { // instantly delete G
            mainArray[robotScore][104+i] = 0;
            mainArray[playerScore][104+i] = 0;
        }
        if (player == true) {
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        if (robot == true) {
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
    }

    for (int i = 0; i < 18; i++) { // drain C and E downwards
        mainArray[robotScore][35+i] = 0;
        mainArray[robotScore][86-i] = 0;
        mainArray[playerScore][35+i] = 0;
        mainArray[playerScore][86-i] = 0;
        wait_ms(30);
        if (player == true){
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        if (robot == true) {
        WritePxAnimation(robotScore,true,robotColour);
        robotScoreLED.write(robotScorePx.getBuf());
        }
    }
    for (int i = 0; i < 10; i++) { // drain D from edges to centre
        mainArray[robotScore][52+i] = 0;
        mainArray[robotScore][70-i] = 0;
        mainArray[playerScore][52+i] = 0;
        mainArray[playerScore][70-i] = 0;
        wait_ms(30);
        if (player == true){
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        if (robot == true) {
        WritePxAnimation(robotScore,true,robotColour);
        robotScoreLED.write(robotScorePx.getBuf());
        }
    }
    

    memset(mainArray, 0, sizeof mainArray);

} 

void ZeroInAnimation(bool robot, bool robotColour, bool player, bool playerColour)
{
     memset(mainArray, 0, sizeof mainArray);
     
     if (player == true && robot == true) {

        for (int i = 0; i < 10; i++) { // write A from centre
            mainArray[0][9+i] = 1;
            mainArray[0][9-i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) { // write B and F downwards
            mainArray[0][18+i] = 1;
            mainArray[0][103-i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) { // write C and E downwards
            mainArray[0][35+i] = 1;
            mainArray[0][86-i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 10; i++) { // write D from edges to centre
            mainArray[0][52+i] = 1;
            mainArray[0][70-i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }

    }

    if (player == true && robot == false) {
        
        for (int i = 0; i < 10; i++) { // A
            mainArray[0][9+i] = 1;
            mainArray[0][9-i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) { // B and F
            mainArray[0][18+i] = 1;
            mainArray[0][103-i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) { // G
            mainArray[0][104+i] = 1;
        }

        WritePxAnimation(playerScore,false,playerColour);
        playerScoreLED.write(playerScorePx.getBuf());

        for (int i = 0; i < 18; i++) { // C and E
            mainArray[0][35+i] = 1;
            mainArray[0][86-i] = 1;
            wait_ms(30);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        for (int i = 0; i < 10; i++) { // D
            mainArray[0][52+i] = 1;
            mainArray[0][70-i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        
    }

    if (player == false && robot == true) {

        for (int i = 0; i < 10; i++) { // A
            mainArray[0][9+i] = 1;
            mainArray[0][9-i] = 1;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) { // B and F
            mainArray[0][18+i] = 1;
            mainArray[0][103-i] = 1;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) { // G
            mainArray[0][104+i] = 1;
        }

        WritePxAnimation(robotScore,true,robotColour);
        robotScoreLED.write(robotScorePx.getBuf());

        for (int i = 0; i < 18; i++) { // C and E
            mainArray[0][35+i] = 1;
            mainArray[0][86-i] = 1;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 10; i++) { // D
            mainArray[0][52+i] = 1;
            mainArray[0][70-i] = 1;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        
    }    
    
}  

void ZeroOutAnimation(bool robot, bool robotColour, bool player, bool playerColour)
{
     if (player == true && robot == true) {

        for (int i = 0; i < 10; i++) { // A
            mainArray[0][9+i] = 0;
            mainArray[0][9-i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) { // B and F
            mainArray[0][18+i] = 0;
            mainArray[0][103-i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) { // G
            mainArray[0][104+i] = 0;
        }

        WritePxAnimation(playerScore,false,playerColour);
        playerScoreLED.write(playerScorePx.getBuf());
        WritePxAnimation(robotScore,true,robotColour);
        robotScoreLED.write(robotScorePx.getBuf());

        for (int i = 0; i < 18; i++) { // C and E
            mainArray[0][35+i] = 0;
            mainArray[0][86-i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 10; i++) { // D
            mainArray[0][52+i] = 0;
            mainArray[0][70-i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }

    }

    if (player == true && robot == false) {
        
        for (int i = 0; i < 10; i++) { // A
            mainArray[0][9+i] = 0;
            mainArray[0][9-i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) { // B and F
            mainArray[0][18+i] = 0;
            mainArray[0][103-i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) { // G
            mainArray[0][104+i] = 0;
        }

        WritePxAnimation(playerScore,false,playerColour);
        playerScoreLED.write(playerScorePx.getBuf());

        for (int i = 0; i < 18; i++) { // C and E
            mainArray[0][35+i] = 0;
            mainArray[0][86-i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        for (int i = 0; i < 10; i++) { // D
            mainArray[0][52+i] = 0;
            mainArray[0][70-i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        
    }

    if (player == false && robot == true) {

        for (int i = 0; i < 10; i++) { // A
            mainArray[0][9+i] = 0;
            mainArray[0][9-i] = 0;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) { // B and F
            mainArray[0][18+i] = 0;
            mainArray[0][103-i] = 0;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) { // G
            mainArray[0][104+i] = 0;
        }

        WritePxAnimation(robotScore,true,robotColour);
        robotScoreLED.write(robotScorePx.getBuf());

        for (int i = 0; i < 18; i++) { // C and E
            mainArray[0][35+i] = 0;
            mainArray[0][86-i] = 0;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 10; i++) { // D
            mainArray[0][52+i] = 0;
            mainArray[0][70-i] = 0;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        
    }    
    
}  

void OneInAnimation(bool robot, bool robotColour,bool player, bool playerColour)
{
     memset(mainArray, 0, sizeof mainArray);
     
     if (player == true && robot == true) {

        for (int i = 51; i > 17; i--) {
            mainArray[1][i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }

    }

    if (player == true && robot == false) {
        
        for (int i = 51; i > 17; i--) { 
            mainArray[1][i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        
    }

    if (player == false && robot == true) {

        for (int i = 51; i > 17; i--) { // B and F
            mainArray[1][i] = 1;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        
    }
    
}   
 
void OneOutAnimation(bool robot, bool robotColour,bool player, bool playerColour)
{
    if (player == true && robot == true) {

        for (int i = 18; i < 52; i++) {
            mainArray[1][i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }

    }

    if (player == true && robot == false) {
        
        for (int i = 18; i < 52; i++) {
            mainArray[1][i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());;
        }
        
    }

    if (player == false && robot == true) {

        for (int i = 18; i < 52; i++) {
            mainArray[1][i] = 0;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        
    }
    
}

void TwoInAnimation(bool robot, bool robotColour,bool player, bool playerColour)
{
    memset(mainArray, 0, sizeof mainArray);
    
    if (player == true && robot == true) {

        for (int i = 0; i < 35; i++) {
            mainArray[2][i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 121; i > 103; i--) {
            mainArray[2][i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 86; i > 51; i--) {
            mainArray[2][i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }

    }

    if (player == true && robot == false) {
        
        for (int i = 0; i < 35; i++) {
            mainArray[2][i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        for (int i = 121; i > 103; i--) {
            mainArray[2][i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        for (int i = 86; i > 51; i--) {
            mainArray[2][i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        
    }

    if (player == false && robot == true) {

        for (int i = 0; i < 35; i++) {
            mainArray[2][i] = 1;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 121; i > 103; i--) {
            mainArray[2][i] = 1;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 86; i > 51; i--) {
            mainArray[2][i] = 1;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        
    }
    
}

void TwoOutAnimation(bool robot, bool robotColour,bool player, bool playerColour)
{
    if (player == true && robot == true) {

        for (int i = 0; i < 35; i++) {
            mainArray[2][i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 121; i > 103; i--) {
            mainArray[2][i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 86; i > 51; i--) {
            mainArray[2][i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }

    }

    if (player == true && robot == false) {
        
        for (int i = 0; i < 35; i++) {
            mainArray[2][i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        for (int i = 121; i > 103; i--) {
            mainArray[2][i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        for (int i = 86; i > 51; i--) {
            mainArray[2][i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        
    }

    if (player == false && robot == true) {

        for (int i = 0; i < 35; i++) {
            mainArray[2][i] = 0;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 121; i > 103; i--) {
            mainArray[2][i] = 0;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 86; i > 51; i--) {
            mainArray[2][i] = 0;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        
    }
    
}

void ThreeInAnimation(bool robot, bool robotColour,bool player, bool playerColour)
{
    memset(mainArray, 0, sizeof mainArray);
    
    idle_button_pressed = 1; // doesn't allow button isr to change idle status
    
    if (player == true && robot == true) {

        for (int i = 0; i < 35; i++) {
            mainArray[3][i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 17; i++) {
            mainArray[3][35+i] = 1;
            mainArray[3][121-i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 52; i < 70; i++) {
            mainArray[3][i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }

    }

    if (player == true && robot == false) {
        
        for (int i = 0; i < 35; i++) {
            mainArray[3][i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        for (int i = 0; i < 17; i++) {
            mainArray[3][35+i] = 1;
            mainArray[3][121-i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        for (int i = 52; i < 70; i++) {
            mainArray[3][i] = 1;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
          
    }

    if (player == false && robot == true) {

        for (int i = 0; i < 35; i++) {
            mainArray[3][i] = 1;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) {
            mainArray[3][35+i] = 1;
            mainArray[3][121-i] = 1;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 52; i < 70; i++) {
            mainArray[3][i] = 1;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        
    }
    
}

void ThreeOutAnimation(bool robot, bool robotColour,bool player, bool playerColour)
{
    if (player == true && robot == true) {

        for (int i = 0; i < 18; i++) {
            mainArray[3][i] = 0;
            mainArray[3][104+i] = 0;
            mainArray[3][69-i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) {
            mainArray[3][18+i] = 0;
            mainArray[3][51-i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }

    }

    if (player == true && robot == false) {
        
        for (int i = 0; i < 18; i++) {
            mainArray[3][i] = 0;
            mainArray[3][104+i] = 0;
            mainArray[3][69-i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) {
            mainArray[3][18+i] = 0;
            mainArray[3][51-i] = 0;
            wait_ms(10);
            WritePxAnimation(playerScore,false,playerColour);
            playerScoreLED.write(playerScorePx.getBuf());
        }
        
    }

    if (player == false && robot == true) {

        for (int i = 0; i < 18; i++) {
            mainArray[3][i] = 0;
            mainArray[3][104+i] = 0;
            mainArray[3][69-i] = 0;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        for (int i = 0; i < 18; i++) {
            mainArray[3][18+i] = 0;
            mainArray[3][51-i] = 0;
            wait_ms(10);
            WritePxAnimation(robotScore,true,robotColour);
            robotScoreLED.write(robotScorePx.getBuf());
        }
        
    }
    
}

void idleButtonISR()
{
    if (idle_button_pressed == 0) {
        if (idle_flag == 0) {
            previous_state = 0;
        }
        else {
            previous_state = 1;
        }
        idle_flag = 1;
        idle_button_pressed = 1;
    }
}
