/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//**
 * \file minitls.h
 * \copyright Copyright (c) AppNearMe Ltd 2013
 * \author Donatien Garnier
 */

#ifndef MINITLS_H_
#define MINITLS_H_

/*
http://tools.ietf.org/html/rfc5246
http://tools.ietf.org/html/rfc4492
http://tools.ietf.org/html/rfc4366#page-11 //Limit record length
http://security.stackexchange.com/questions/3204/computationally-simple-lightweight-replacement-for-ssl-tls
*/

#ifdef __cplusplus
extern "C" {
#endif

//Implementation of the TLS1.2 protocol with TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA cipher suite

#include "core/fwk.h"
#include "inc/minitls_config.h"
#include "inc/minitls_errors.h"

#include "crypto/crypto_ecc.h"
#include "crypto/crypto_rsa.h"
#include "crypto/crypto_prng.h"

typedef struct __tls_x509_certificate //If we know the server's certificate, we just have to do a memcmp to "verify" it
{
  const uint8_t* certificate;
  size_t certificate_size;

  //These fields can either be decoded from the certificate (using ASN module -- TODO) or prepopulated

  //Decoded -- or prepopulated
  //crypto_ecc_curve_type_t ecc_curve;
  union
  {
#if CRYPTO_ECC
    crypto_ecc_public_key_t ecc;
#endif
#if CRYPTO_RSA
    crypto_rsa_public_key_t rsa;
#endif
  } public_key;

  //public_key_type (ECDH-capable)
  //signature_algorithm (ECDSA-SHA1) -- certificate is encrypted using private key and then hashed with SHA1
} tls_x509_certificate_t;


typedef struct __minitls
{
  crypto_prng_t* prng;
  const tls_x509_certificate_t* certificate; //Certificate is global to all connections
} minitls_t;


minitls_err_t minitls_init(minitls_t* minitls, crypto_prng_t* prng);
minitls_err_t minitls_certificate_add(minitls_t* minitls, const tls_x509_certificate_t* cert); //Only one supported now

#ifdef __cplusplus
}
#endif

#endif /* MINITLS_H_ */
