/*
MiniTLS - A super trimmed down TLS/SSL Library for embedded devices
Author: Donatien Garnier
Copyright (C) 2013-2014 AppNearMe Ltd

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*//**
 * \file crypto_arc4.c
 * \copyright Copyright (c) AppNearMe Ltd 2014
 * \author Donatien Garnier
 */

#include "core/fwk.h"
#include "crypto_arc4.h"
#include "crypto_macros.h"

void crypto_arc4_init(crypto_arc4_t* arc4, const uint8_t* key, size_t key_size)
{
  int i;
  int j;
  uint8_t b;

  arc4->i = 0;
  arc4->j = 0;

  for(i = 0; i < 256; i++)
  {
    arc4->state[i] = i;
  }

  for( i = j = 0; i < 256; i++)
  {
    //Shuffle using key
    j = (j + arc4->state[i] + key[i % key_size]) & 0xFF;

    //Swap
    b = arc4->state[i];
    arc4->state[i] = arc4->state[j];
    arc4->state[j] = b;
  }
}

void crypto_arc4_encrypt(crypto_arc4_t* arc4, const uint8_t* plaintext, uint8_t* ciphertext, size_t length)
{
  uint8_t b;

  while(length--)
  {
    arc4->i = (arc4->i + 1) & 0xFF;
    arc4->j = (arc4->j + arc4->state[arc4->i]) & 0xFF;

    //Swap
    b = arc4->state[arc4->i];
    arc4->state[arc4->i] = arc4->state[arc4->j];
    arc4->state[arc4->j] = b;

    *ciphertext = *plaintext ^ arc4->state[ (arc4->state[arc4->i] + arc4->state[arc4->j]) & 0xFF ];
    ciphertext++;
    plaintext++;
  }
}

void crypto_arc4_process(crypto_arc4_t* arc4, buffer_t* buffer)
{
  uint8_t* text = buffer_current_read_position(buffer);
  size_t size = buffer_length(buffer);

  crypto_arc4_encrypt(arc4, text, text, size);
}
