#include "Mat.h"

 
template<typename T>
Mat<T> PIDCONTROL( Mat<T> phi_d, Mat<T> u_p, Mat<T> e_old, T dt, int behaviour = 3)
{
    Mat<T> r(phi_d-u_p);
    e_old = e_old + r;

    /*PID ANGULAR ROTATION : PERFECT*/
    Mat<T> Kp((T) -0.01, 2,1);
    Mat<T> Ki((T)50, 2,1);
    Mat<T> Kd((T)0, 2,1);
    /*-------------------------------*/
    
    if(behaviour==2)
    {
        phi_d.afficher();
        Kp = (T)10000*Kp;
        Ki = ((T)10000)*Ki;
        Kd = Mat<T>((T)1,2,1);
        Kd = ((T)0)*Kd;
    
    }
    
    r = Kp%r + dt*Ki%e_old + ((double)1.0/dt)*Kd%r;
        
    return r;
}


template<typename T>
class EKF
{
    private :
    
    Serial *pcs;
    T time;
    T dt;   /* par default : = 0.005 */
    int nbr_state;
    int nbr_ctrl;
    int nbr_obs;
    
    Mat<T> dX;  /*desired state*/
    Mat<T>* _X; /*previous state*/
    Mat<T>* X;      /*states*/
    Mat<T>* X_; /*derivated states or next states... (continuous/discrete)*/
    Mat<T>* u;      /*controls*/
    Mat<T>* z;      /*observations/measurements*/
    Mat<T> obs_old;
    Mat<T> e_old;
    
    Mat<T>* Ki;
    Mat<T>* Kp;
    Mat<T>* Kd;
    Mat<T>* Kdd;    
    
    Mat<T>* A;      /*linear relation matrix between states and derivated states.*/
    /*par default : X_ = A * X + B * u / x_i = x_i + dt * x_i_ + b_i * u_i... */
    Mat<T>* B;      /*linear relation matrix between derivated states and control.*/
    /*par default : B = 1 */
    Mat<T>* C;      /*linear relation matrix between states and observation.*/
    /*par default : C = [1 0], on observe les positions, non leurs dérivées. */

    /*Noise*/
    T std_noise;    /*par defaut : 0.0005*/
    Mat<T>* Sigma;  /*covariance matrix*/
    Mat<T>* Q;      /*process noise*/
    Mat<T>* R;      /*measurement noise*/
    
    /*Prediction*/
    Mat<T> K;       // Kalman Gain...
    Mat<T> Sigma_p;
    
    /*Others*/
    bool filterOn;
    Mat<T>* Identity;
    
    
    /*Extended*/
    bool extended;
    Mat<T> (*ptrMotion)(Mat<T> state, Mat<T> command, T dt);
    Mat<T> (*ptrSensor)(Mat<T> state, Mat<T> command, Mat<T> d_state, T dt);
    Mat<T> G;
    Mat<T> H;
    Mat<T> (*ptrJMotion)(Mat<T> state, Mat<T> command, T dt);
    Mat<T> (*ptrJSensor)(Mat<T> state, Mat<T> command, Mat<T> d_state, T dt);
    
    
    public :
    
    EKF(Serial *pc, int nbr_state_, int nbr_ctrl_, int nbr_obs_, T dt_, T std_noise_, Mat<T> currentState, bool ext = false, bool filterOn = true)
    {
        this->filterOn = filterOn;
        this->pcs = pc;
        /*extension*/
        time = (T)0;
        extended = ext;
        ptrMotion = NULL;
        ptrSensor = NULL;
        ptrJMotion = NULL;
        ptrJSensor = NULL;
        G = Mat<T>((T)0, nbr_state_, nbr_state_);
        H = Mat<T>((T)0, nbr_obs_, nbr_state_);
        
        /*----------------*/
        
        dt = dt_;
        nbr_state = nbr_state_;
        nbr_ctrl = nbr_ctrl_;
        nbr_obs = nbr_obs_;
        
        _X = new Mat<T>((T)0, nbr_state, (int)1);       /*previous state*/
        X = new Mat<T>(currentState);                   /*states*/
        X_ = new Mat<T>((T)0, nbr_state, (int)1);       /*derivated states*/
        u = new Mat<T>((T)0, nbr_ctrl, (int)1);         /*controls*/
        z = new Mat<T>((T)0, nbr_obs, (int)1);  
        obs_old = *z;                       /*observations*/
        e_old = *z;
        A = new Mat<T>((T)0, nbr_state, nbr_state);     /*linear relation or jacobian matrix between states and derivated states.*/
        B = new Mat<T>((T)0, nbr_state, nbr_ctrl);      /*linear relation matrix between derivated states and control.*/
        C = new Mat<T>((T)0, nbr_obs, nbr_state);       /*linear relation or jacobian matrix between states and observation.*/
    
        Ki = new Mat<T>((T)0.08, nbr_ctrl, nbr_state);
        Kp = new Mat<T>((T)0.08, nbr_ctrl, nbr_state);
        Kd = new Mat<T>((T)0.08, nbr_ctrl, nbr_state);
        Kdd = new Mat<T>((T)0.08, nbr_ctrl, nbr_state);
    
    
        std_noise = std_noise_;
        Sigma = new Mat<T>((T)0, nbr_state, nbr_state);
        Q = new Mat<T>((T)0, nbr_state, nbr_state);
        R = new Mat<T>((T)0, nbr_obs, nbr_obs/*1*/);
    
        /*Initialize Covariance matrix as the identity matrix.*/
        for(int i=1;i<=nbr_state;i++)
        {
            Sigma->set((T)1, i,i);
            R->set( (T)1, i,i);
            /*
            for(int j=1;j<=nbr_state;j++)
            {
                Sigma->set((T)1, i, j);
            
                //if(i<=nbr_obs && j==1)
                //  R->set(std_noise*std_noise, i, j);
                
            }
            */
        }
    
        Identity = new Mat<T>(*Sigma);
        *Q = (std_noise*std_noise)*(*Identity);
        *R = (std_noise*std_noise)*(*R);
    
    }

    ~EKF()
    {
        delete _X;
        delete X;
        delete X_;
        delete u;
        delete z;
        delete A;
        delete B;
        delete C;
    
        delete Ki;
        delete Kp;
        delete Kd;
        delete Kdd;
    
        delete Sigma;
        delete Q;
        delete R;
    
        delete Identity;
    }

/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/


    int initA( Mat<T> A_)
    {
        if(A_ == *Identity)
        {
            for(int i=1;i<=(int)(nbr_state/2);i++)
            {
                A->set( dt, i, i+(int)(nbr_state/2));
            }
        
            return 1;
        }
        else
        {
            if(A_.getColumn() == nbr_state && A_.getLine() == nbr_state)
            {
                *A = A_;
                return 1;
            }
            else
            {
                cout << "ERREUR : mauvais format de matrice d'initialisation de A." << endl;
                return 0;
            }
        }
    }


    int initB( Mat<T> B_)
    {   
        if(B_.getColumn() == nbr_ctrl && B_.getLine() == nbr_state)
        {
            *B = B_;
            return 1;
        }
        else
        {
            cout << "ERREUR : mauvais format de matrice d'initialisation de B." << endl;
            return 0;
        }
    
    }
    
    
    
    int initC( Mat<T> C_)
    {   
        if(C_.getColumn() == nbr_state && C_.getLine() == nbr_obs)
        {
            *C = C_;
            return 1;
        }
        else
        {
            cout << "ERREUR : mauvais format de matrice d'initialisation de C." << endl;
            return 0;
        }
    
    }
    
    /*extension*/
    void initMotion( Mat<T> motion(Mat<T>, Mat<T>, T) )
    {
        ptrMotion = motion;
    }
    
    
    
    void initSensor( Mat<T> sensor(Mat<T>, Mat<T>, Mat<T>, T) )
    {   
        ptrSensor = sensor; 
    }
    
    void initJMotion( Mat<T> jmotion(Mat<T>, Mat<T>, T) )
    {
        ptrJMotion = jmotion;
    }
    
    
    
    void initJSensor( Mat<T> jsensor(Mat<T>, Mat<T>, Mat<T>, T) )
    {   
        ptrJSensor = jsensor;   
    }
    
    
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/


    int setKi( Mat<T> Ki_)
    {
        if(Ki_.getColumn() == nbr_state && Ki_.getLine() == nbr_ctrl)
        {
            *Ki = Ki_;
            return 1;
        }
        else
        {
            cout << "ERREUR : mauvais format de vecteur d'initialisation de Ki." << endl;
            return 0;
        }
    }
    
    int setKp( Mat<T> Kp_)
    {
        if(Kp_.getColumn() == nbr_state && Kp_.getLine() == nbr_ctrl)
        {
            *Kp = Kp_;
            return 1;
        }
        else
        {
            cout << "ERREUR : mauvais format de vecteur d'initialisation de Kp." << endl;
            return 0;
        }
    }
    
    
    
    int setKd( Mat<T> Kd_)
    {
        if(Kd_.getColumn() == nbr_state && Kd_.getLine() == nbr_ctrl)
        {
            *Kd = Kd_;
            return 1;
        }
        else
        {
            cout << "ERREUR : mauvais format de vecteur d'initialisation de Kd." << endl;
            return 0;
        }

    }
    
    int setKdd( Mat<T> Kdd_)
    {
        if(Kdd_.getColumn() == nbr_state && Kdd_.getLine() == nbr_ctrl)
        {
            *Kdd = Kdd_;
            return 1;
        }
        else
        {
            cout << "ERREUR : mauvais format de vecteur d'initialisation de Kdd." << endl;
            return 0;
        }

    }
    
    
    void setdt( float dt_)
    {
        dt = dt_;
    }
    
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/   
    
    
    Mat<T> getCommand()
    {
        return *u;
    }   
    
    Mat<T> getX()
    {
        return *X;
    }

    Mat<T> getSigma()
    {
        return *Sigma;
    }   


/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/       
    
    
    Mat<T> predictState()       /*return the computed predicted state*/
    {
        return (!extended ? (*A)*(*X)+(*B)*(*u) : ptrMotion(*X, *u, dt) );
    }
    
    
    Mat<T> predictCovariance()  /*return the predicted covariance matrix.*/
    {        
        
        if(extended)
            G = ptrJMotion(*X, *u, dt);
                    
            
        return (filterOn ? ( (!extended ? ((*A)*(*Sigma))* transpose(*A) : G*(*Sigma)*transpose(G) ) + *Q) : *Identity);
    }
    
        
    Mat<T> calculateKalmanGain()    /*return the Kalman Gain K = C*Sigma_p * (C*Sigma_p*C.T +R).inv */
    {       
        
        Sigma_p = predictCovariance();  
                
        if(extended)
            H = ptrJSensor(*X, *u, dX,dt);
                

        if(!extended)                    
            return (filterOn ? Sigma_p * transpose(*C) * invGJ( (*C) * Sigma_p * transpose(*C) + *R) : *Identity);
        else
        {
            pcm.printf("Sigma\n");
            Sigma_p.afficherM();
            Mat<double> test( H * Sigma_p * transpose(H) + *R);            
            test = invGJ(test);            
            pcm.printf("test inverse\n");            
                     
            return (filterOn ? Sigma_p * transpose(H) * test : *Identity);        
        }
    }
    
        
    Mat<T> correctState()       /*update X */
    {
        Mat<T> X_p( predictState());        
    
        *_X = *X;
        
        if(filterOn)
            *X = X_p + K*( (*z) - (!extended ? (*C)*X_p  : ptrSensor(X_p, *u, dX, dt) ) );
        else
            *X = X_p;
    
        return *X;
    }
    
        
    Mat<T> correctCovariance()  /*update Sigma*/
    {
        *Sigma = (filterOn ? (*Identity - K* (!extended ? (*C) : H) ) *Sigma_p : *Identity);
    
        return *Sigma;
    }
    
    
    void state_Callback()       /* Update all... */
    {            
        time += dt;
        
        if( extended && (ptrMotion == NULL || ptrSensor == NULL || ptrJMotion == NULL || ptrJSensor == NULL) )
        {
            //~EKF();
            throw("ERREUR : les fonctions ne sont pas initialisées...");
        }       
        
        K = calculateKalmanGain();              
        correctState();                
        correctCovariance();        
    }
    
    void measurement_Callback(Mat<T> measurements, Mat<T> dX_, bool measure = false)
    {
        if( extended && (ptrMotion == NULL || ptrSensor == NULL || ptrJMotion == NULL || ptrJSensor == NULL) )
        {
            //~EKF();
            throw("ERREUR : les fonctions ne sont pas initialisées...");
        }
        
        dX = dX_;
        
        *z = (!extended || measure ? measurements : ptrSensor(*X,*u, dX, dt) ); 
    }
    
    void measurement_Callback(Mat<T> measurements)
    {
        if( extended && (ptrMotion == NULL || ptrSensor == NULL || ptrJMotion == NULL || ptrJSensor == NULL) )
        {
            //~EKF();
            throw("ERREUR : les fonctions ne sont pas initialisées...");
        }
        
        
        *z = (!extended ? measurements : ptrSensor(*X,*u, dX, dt) );    
    }
    
    
    void computeCommand( Mat<T> desiredX, T dt_, int mode)
    {
        obs_old = obs_old + dt_*(*z);
        
        /*work on observation instead of state vector ???? */
        
        
        if(dt_ != (T)0 && mode != -1)
        {
            *u = (*Kp)*(desiredX - (*X));
            if(mode >= 1)
                *u = *u + (T)(dt_)*(*Ki)*(desiredX - (*X));
            if(mode >= 2)
                *u = *u + (T)((double)1.0/dt_)*(*Kd)*(desiredX - (*X));
            if(mode >= 3)
                *u = *u + (T)((double)1.0/(dt_*dt_))*(*Kdd)*(desiredX - (*X));                      
        }       
        else if(mode !=-1)
        {
            *u = (*Kp)*(desiredX - (*X));       
        }
            
        if(mode == -1)
        {
        
            Mat<T> bicycle((T)0, 3,1);
                        bicycle.set((T)100, 1,1);  /*radius*/
                bicycle.set((T)100, 2,1);
                bicycle.set((T)66, 3,1);   /*entre-roue*/
                
            T rho = (T)sqrt(z->get(1,1)*z->get(1,1) + z->get(2,1)*z->get(2,1));
            T alpha = atan21( dX.get(2,1)-X->get(2,1), dX.get(1,1) - X->get(1,1) );
            T beta = alpha + X->get(3,1) + dX.get(3,1);
            
            cout << "alpha = " << alpha << endl;
            cout << "rho = " << rho << endl;
            cout << "beta = " << beta << endl;          
            
            T krho = (double)(rho > 1 ? rho : 0.1);
            T kalpha = (double)rho*10;
            T kbeta = (double)2*beta/(rho+1);
            //T krho = 1;
            //T kbeta = 1;
            //T kalpha = 1-5.0/3.0*kbeta+2.0/PI*krho*rho;
            
            if(alpha >= 0.2 || alpha <= -0.2)
            {
                krho = krho/100;
                kbeta = kbeta/100;
                //kalpha = kalpha*alpha;
            }       
            else if(rho >=1 )
            {
                kalpha = kalpha/100;
                kbeta = kbeta/100;
                krho = krho*(rho-1);
            }           
            else 
            {
                kbeta = kbeta*beta;
                kalpha = 0;
                krho = krho/100;
            }       
                        
            T vd = krho*rho;
            T wd = kalpha*alpha +  kbeta*beta;
            
            
            u->set( vd, 1,1);
            u->set( wd, 2,1);       
            
        }
        
        if(mode == -2)
        {
            T r = (T)100;  /*radius*/
                T l = (T)66;   /*entre-roue*/
                T phi_max = 200;
                
            T rho = (T)sqrt((dX.get(1,1)-X->get(1,1))*(dX.get(1,1)-X->get(1,1)) + (dX.get(2,1)-X->get(2,1))*(dX.get(2,1)-X->get(2,1)));
            T alpha = atan21( dX.get(2,1)-X->get(2,1), dX.get(1,1) - X->get(1,1) ) - atan21(sin(X->get(3,1)), cos(X->get(3,1)));
            alpha = atan21( sin(alpha), cos(alpha));
            T beta = -X->get(3,1) + dX.get(3,1);
            beta = atan21(sin(beta), cos(beta));
            
            cout << "alpha = " << alpha << endl;
            cout << "rho = " << rho << endl;
            cout << "beta = " << beta << endl;          
            
            //T krho = (double)(rho > 1 ? rho : 0.1);
            //T kalpha = (double)rho*10;
            //T kbeta = (double)2*beta/(rho+1);
            T krho = 1;
            T kbeta = 0.2;
            T kalpha = 1-5.0/3.0*kbeta+2.0/PI*krho*rho;
            int behaviour = 0;
            
            cout << "test : " << ( fabs_(alpha) >= 0.1 && (( (dX.get(1,1)+1 <= X->get(1,1)) || (dX.get(1,1)-1 >= X->get(1,1)) ) || ( (dX.get(2,1)+1 <= X->get(2,1)) || (dX.get(2,1)-1 >= X->get(2,1)) ) )) << endl;
            
            if( fabs_(alpha) >= 0.1 && ( ( (dX.get(1,1)+1 <= X->get(1,1)) || (dX.get(1,1)-1 >= X->get(1,1)) ) || ( (dX.get(2,1)+1 <= X->get(2,1)) || (dX.get(2,1)-1 >= X->get(2,1)) ) ) )
            {
                krho = krho/1000;
                kalpha = kbeta/10;
                kbeta = 0;
                behaviour = 1;
            }       
            else if(rho >=0.1 )
            {
                kalpha = kalpha/10000;
                kbeta = 0;
                krho = krho;
                behaviour = 2;
            }           
            else if(fabs_(beta) >= 0.1) 
            {
                kbeta = kbeta*10;
                kalpha = 0;
                krho = krho/1000;
                behaviour = 3;
            }
            else
            {   
                kbeta = kbeta/100000;
                kalpha = kalpha/100000;
                krho = krho/10000;          
                behaviour = 4;
            }
            
            cout << "behaviour = " << behaviour << endl;
                        
            T vd = krho*rho ;
            T wd = kalpha*alpha +  kbeta*beta;
            vd = ( fabs_(l/r*vd) <= phi_max ? vd : (vd >= 0 ? r/l*phi_max : -r/l*phi_max) );
            wd = ( fabs_(l/r*wd) <= phi_max ? wd : (wd >= 0 ? r/l*phi_max : -r/l*phi_max) );    
            
            cout << "Vd = " << vd << endl;
            cout << "Wd = " << wd << endl;
            cout << "max = " << r/l*phi_max << endl;        
            
            T phi_r_d = l/r*(vd/l+wd);
            T phi_l_d = l/r*(vd/l-wd);
            
            
            Mat<T> phi(2,1);
            phi.set( phi_r_d, 1,1);
            phi.set( phi_l_d, 2,1);
            phi.afficher();
            
            Mat<T> phi_p(phi);
            phi_p.set( u->get(1,1)+u->get(2,1), 1,1);
            phi_p.set( u->get(1,1)-u->get(2,1), 2,1);
            phi_p = r/(2*l)*phi_p;
            
            if(e_old.getLine() != 2)
                e_old = (T)0*phi;               

            
            phi = PIDCONTROL( phi, phi_p, e_old, dt, behaviour);
            
            u->set( r/(2*l)*(phi.get(1,1)+phi.get(2,1)), 1,1);
            u->set( r/(2*l)*(phi.get(1,1)-phi.get(2,1)), 2,1);
        }
    }   
    

};