#include "Matv2.h"
#include "rand.h"
#include "PIDcontroller.h"
#define default_ACC 0.05 //Asserv 100 Hz -> T = 0.01 -> 50 mm/s2 = 0.05 mm/T2
#define default_DEC 0.05
//#define debug
//#define verbose_debug
//#define verbose_debug_phi
//#define verbose_Jacobian

template<typename T>
class EKF
{
    private :
    
    T time;
    T dt;   /* par default : = 0.005 */
    int nbr_state;
    int nbr_ctrl;
    int nbr_obs;
    
    Mat<T> dX;  /*desired state*/
    Mat<T> lastdX;
    Mat<T>* _X; /*previous state*/
    Mat<T>* X;      /*states*/
    Mat<T> X_p; /*predicted state*/
    Mat<T>* X_; /*derivated states or next states... (continuous/discrete)*/
    Mat<T>* u;      /*controls*/
    Mat<T>* z;      /*observations/measurements*/
    Mat<T> obs_old;
    Mat<T> eSum;
    Mat<T> e_old;
    int behaviour;
    int lastBehaviour;    
    Mat<T> lastPhi_consigne;
    Mat<T> lastPhi_desired;
    Mat<T> phi_consigne;
    Mat<T> phi_desired;
    bool initPidVD;
    bool initPidVG;
    PIDcontroller<T>* pidVD;
    PIDcontroller<T>* pidVG;
    PIDcontroller<T>* pidPosV;
    PIDcontroller<T>* pidAngleW;    
    bool pidPos;
    bool backward;
    
    const static T r = (T)36;  /*radius*/
    const static T l = (T)220;   /*entre-roue*/
    T elapsed_T;
    double Vd_max;    
    
    Mat<T>* Ki;
    Mat<T>* Kp;
    Mat<T>* Kd;
    Mat<T>* Kdd;    
    
    Mat<T>* A;      /*linear relation matrix between states and derivated states.*/
    /*par default : X_ = A * X + B * u / x_i = x_i + dt * x_i_ + b_i * u_i... */
    Mat<T>* B;      /*linear relation matrix between derivated states and control.*/
    /*par default : B = 1 */
    Mat<T>* C;      /*linear relation matrix between states and observation.*/
    /*par default : C = [1 0], on observe les positions, non leurs dérivées. */

    /*Noise*/
    T std_noise;    /*par defaut : 0.0005*/
    Mat<T>* Sigma;  /*covariance matrix*/
    Mat<T>* Q;      /*process noise*/
    Mat<T>* R;      /*measurement noise*/
    NormalRand* rgenQ;
    NormalRand* rgenR;
    bool noise;
    
    /*Prediction*/
    Mat<T> K;       // Kalman Gain...
    Mat<T> Sigma_p;
    
    /*Others*/
    bool filterOn;
    Mat<T>* Identity;
    
    
    /*Extended*/
    bool extended;
    Mat<T> (*ptrMotion)(Mat<T> state, Mat<T> command, T dt);
    Mat<T> (*ptrSensor)(Mat<T> state, Mat<T> command, Mat<T> d_state, T dt);
    Mat<T> G;
    Mat<T> H;
    Mat<T> (*ptrJMotion)(Mat<T> state, Mat<T> command, T dt);
    Mat<T> (*ptrJMotionCommand)(Mat<T> state, Mat<T> command, T dt);
    Mat<T> (*ptrJSensor)(Mat<T> state, Mat<T> command, Mat<T> d_state, T dt);
    
    
    public :
    
    EKF()
    {
        backward = false;
        this->filterOn = true;
        /*extension*/
        time = (T)0;
        extended = true;
        ptrMotion = NULL;
        ptrSensor = NULL;
        ptrJMotion = NULL;
        ptrJMotionCommand = NULL;
        ptrJSensor = NULL;
        G = Mat<T>((T)0, 1, 1);
        H = Mat<T>((T)0, 1, 1);
        
        /*----------------*/
        
        dt = (T)0.5;
        nbr_state = 1;
        nbr_ctrl = 1;
        nbr_obs = 1;
        elapsed_T = (T)0;
        
        _X = new Mat<T>((T)0, nbr_state, (int)1);       /*previous state*/
        X = new Mat<T>(*_X);   
        dX = *X;                /*states*/
        lastdX = dX;
        X_ = new Mat<T>((T)0, nbr_state, (int)1);       /*derivated states*/
        u = new Mat<T>((T)0, nbr_ctrl, (int)1);         /*controls*/
        z = new Mat<T>((T)0, nbr_obs, (int)1);  
        obs_old = *z;                       /*observations*/
        e_old = *z;
        eSum = *z;
        lastBehaviour = 4;
        behaviour = 0;
        pidVD = new PIDcontroller<T>();
        pidVG = new PIDcontroller<T>();
        pidPosV = new PIDcontroller<T>();
        pidAngleW = new PIDcontroller<T>;
        initPidVG = false;
        initPidVD = false;
        pidPos = true;
        
        lastPhi_consigne = Mat<T>((T)0,2,1);
        lastPhi_desired = Mat<T>((T)0,2,1);
        phi_consigne = Mat<T>((T)0,2,1);
        phi_desired = Mat<T>((T)0,2,1);
        
        A = new Mat<T>((T)0, nbr_state, nbr_state);     /*linear relation or jacobian matrix between states and derivated states.*/
        B = new Mat<T>((T)0, nbr_state, nbr_ctrl);      /*linear relation matrix between derivated states and control.*/
        C = new Mat<T>((T)0, nbr_obs, nbr_state);       /*linear relation or jacobian matrix between states and observation.*/
    
        Ki = new Mat<T>((T)0.08, nbr_ctrl, nbr_state);
        Kp = new Mat<T>((T)0.08, nbr_ctrl, nbr_state);
        Kd = new Mat<T>((T)0.08, nbr_ctrl, nbr_state);
        Kdd = new Mat<T>((T)0.08, nbr_ctrl, nbr_state);
    
    
        std_noise = (T)0.2;
        Sigma = new Mat<T>((T)0, nbr_state, nbr_state);
        Q = new Mat<T>((T)0, nbr_state, nbr_state);
        R = new Mat<T>((T)0, nbr_obs, nbr_obs/*1*/);
        this->noise = true;
        if(noise)
        {
            rgenQ = new NormalRand(0.0,std_noise,(long)10);
            rgenR = new NormalRand(0.0,std_noise,(long)100);
        }
    
        /*Initialize Covariance matrix as the identity matrix.*/
        for(int i=1;i<=nbr_state;i++)
        {
            Sigma->set((T)1, i,i);
            if(noise)
                R->set( (T)rgenR->dev(), i,i);
            else
                R->set( (T)1, i,i);
            /*
            for(int j=1;j<=nbr_state;j++)
            {
                Sigma->set((T)1, i, j);
            
                //if(i<=nbr_obs && j==1)
                //  R->set(std_noise*std_noise, i, j);
                
            }
            */
        }
    
        Identity = new Mat<T>(*Sigma);
        *Q = (std_noise*std_noise)*(*Identity);
        *R = (std_noise*std_noise)*(*R);
        
    }
    
    EKF(int nbr_state_, int nbr_ctrl_, int nbr_obs_, T dt_, T std_noise_, Mat<T> currentState, bool ext = false, bool filterOn = true, bool noise = false)
    {
        backward = false;
        this->filterOn = filterOn;
        /*extension*/
        time = (T)0;
        extended = ext;
        ptrMotion = NULL;
        ptrSensor = NULL;
        ptrJMotion = NULL;
        ptrJMotionCommand = NULL;
        ptrJSensor = NULL;
        G = Mat<T>((T)0, nbr_state_, nbr_state_);
        H = Mat<T>((T)0, nbr_obs_, nbr_state_);
        
        /*----------------*/
        
        dt = dt_;
        nbr_state = nbr_state_;
        nbr_ctrl = nbr_ctrl_;
        nbr_obs = nbr_obs_;
        elapsed_T = (T)0;
        
        _X = new Mat<T>((T)0, nbr_state, (int)1);       /*previous state*/
        X = new Mat<T>(currentState);   
        dX = *X;                /*states*/
        lastdX = dX;
        X_ = new Mat<T>((T)0, nbr_state, (int)1);       /*derivated states*/
        u = new Mat<T>((T)0, nbr_ctrl, (int)1);         /*controls*/
        z = new Mat<T>((T)0, nbr_obs, (int)1);  
        obs_old = *z;                       /*observations*/
        e_old = *z;
        eSum = *z;
        lastBehaviour = 0;
        behaviour = 0;
        pidVD = new PIDcontroller<T>();
        pidVG = new PIDcontroller<T>();
        initPidVG = false;
        initPidVD = false;
        pidPosV = new PIDcontroller<T>();
        pidAngleW = new PIDcontroller<T>();
        pidPos = true;
        
        lastPhi_consigne = Mat<T>((T)0,2,1);
        lastPhi_desired = Mat<T>((T)0,2,1);
        phi_consigne = Mat<T>((T)0,2,1);
        phi_desired = Mat<T>((T)0,2,1);
        
        A = new Mat<T>((T)0, nbr_state, nbr_state);     /*linear relation or jacobian matrix between states and derivated states.*/
        B = new Mat<T>((T)0, nbr_state, nbr_ctrl);      /*linear relation matrix between derivated states and control.*/
        C = new Mat<T>((T)0, nbr_obs, nbr_state);       /*linear relation or jacobian matrix between states and observation.*/
    
        Ki = new Mat<T>((T)0.08, nbr_ctrl, nbr_state);
        Kp = new Mat<T>((T)0.08, nbr_ctrl, nbr_state);
        Kd = new Mat<T>((T)0.08, nbr_ctrl, nbr_state);
        Kdd = new Mat<T>((T)0.08, nbr_ctrl, nbr_state);
    
    
        std_noise = std_noise_;
        Sigma = new Mat<T>((T)0, nbr_state, nbr_state);
        Q = new Mat<T>((T)0, nbr_state, nbr_state);
        R = new Mat<T>((T)0, nbr_obs, nbr_obs/*1*/);
        this->noise = noise;
        if(noise)
        {
            rgenQ = new NormalRand(0.0,std_noise,(long)10);
            rgenR = new NormalRand(0.0,std_noise,(long)100);
        }
    
        /*Initialize Covariance matrix as the identity matrix.*/
        for(int i=1;i<=nbr_state;i++)
        {
            Sigma->set((T)1, i,i);
            if(noise)
                R->set( (T)rgenR->dev(), i,i);
            else
                R->set( (T)1, i,i);
            /*
            for(int j=1;j<=nbr_state;j++)
            {
                Sigma->set((T)1, i, j);
            
                //if(i<=nbr_obs && j==1)
                //  R->set(std_noise*std_noise, i, j);
                
            }
            */
        }
    
        Identity = new Mat<T>(*Sigma);
        *Q = (std_noise*std_noise)*(*Identity);
        *R = (std_noise*std_noise)*(*R);
    
    }

    ~EKF()
    {
        delete _X;
        delete X;
        delete X_;
        delete u;
        delete z;
        delete A;
        delete B;
        delete C;
    
        delete Ki;
        delete Kp;
        delete Kd;
        delete Kdd;
        delete pidVG;
        delete pidVD;
        delete pidPosV;
        delete pidAngleW;
    
        delete Sigma;
        delete Q;
        delete R;
    
        delete Identity;
    }

/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/


    int initA( Mat<T> A_)
    {
        if(A_ == *Identity)
        {
            for(int i=1;i<=(int)(nbr_state/2);i++)
            {
                A->set( dt, i, i+(int)(nbr_state/2));
            }
        
            return 1;
        }
        else
        {
            if(A_.getColumn() == nbr_state && A_.getLine() == nbr_state)
            {
                *A = A_;
                return 1;
            }
            else
            {
                //cout << "ERREUR : mauvais format de matrice d'initialisation de A." << endl;
                return 0;
            }
        }
    }


    int initB( Mat<T> B_)
    {   
        if(B_.getColumn() == nbr_ctrl && B_.getLine() == nbr_state)
        {
            *B = B_;
            return 1;
        }
        else
        {
            //cout << "ERREUR : mauvais format de matrice d'initialisation de B." << endl;
            return 0;
        }
    
    }
    
    
    
    int initC( Mat<T> C_)
    {   
        if(C_.getColumn() == nbr_state && C_.getLine() == nbr_obs)
        {
            *C = C_;
            return 1;
        }
        else
        {
            //cout << "ERREUR : mauvais format de matrice d'initialisation de C." << endl;
            return 0;
        }
    
    }
    
    /*extension*/
    void initMotion( Mat<T> motion(Mat<T>, Mat<T>, T) )
    {
        ptrMotion = motion;
    }
    
    
    
    void initSensor( Mat<T> sensor(Mat<T>, Mat<T>, Mat<T>, T) )
    {   
        ptrSensor = sensor; 
    }
    
    void initJMotion( Mat<T> jmotion(Mat<T>, Mat<T>, T) )
    {
        ptrJMotion = jmotion;
    }
    
    void initJMotionCommand(Mat<T> jmotioncommand(Mat<T>,Mat<T>,T) )
    {
        ptrJMotionCommand = jmotioncommand;
    }
    
    
    void initJSensor( Mat<T> jsensor(Mat<T>, Mat<T>, Mat<T>, T) )
    {   
        ptrJSensor = jsensor;   
    }
    
    
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/


    int setKi( Mat<T> Ki_)
    {
        if(Ki_.getColumn() == nbr_state && Ki_.getLine() == nbr_ctrl)
        {
            *Ki = Ki_;
            return 1;
        }
        else
        {
            //pcm.printf("ERREUR : mauvais format de vecteur d'initialisation de Ki.");
            return 0;
        }
    }
    
    int setKp( Mat<T> Kp_)
    {
        if(Kp_.getColumn() == nbr_state && Kp_.getLine() == nbr_ctrl)
        {
            *Kp = Kp_;
            return 1;
        }
        else
        {
            ////pcm.printf("ERREUR : mauvais format de vecteur d'initialisation de Kp.");
            return 0;
        }
    }
    
    
    
    int setKd( Mat<T> Kd_)
    {
        if(Kd_.getColumn() == nbr_state && Kd_.getLine() == nbr_ctrl)
        {
            *Kd = Kd_;
            return 1;
        }
        else
        {
            //pcm.printf("ERREUR : mauvais format de vecteur d'initialisation de Kd.");
            return 0;
        }

    }
    
    int setKdd( Mat<T> Kdd_)
    {
        if(Kdd_.getColumn() == nbr_state && Kdd_.getLine() == nbr_ctrl)
        {
            *Kdd = Kdd_;
            return 1;
        }
        else
        {
            //cout << "ERREUR : mauvais format de vecteur d'initialisation de Kdd." << endl;
            return 0;
        }

    }
    
    
    void setdt( float dt_)
    {
        dt = dt_;
    }
    
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/   
    
    
    Mat<T> getCommand()
    {
        return *u;
    }   
    
    Mat<T> getX()
    {
        return *X;
    }
    
    void setX(Mat<T> x_)
    {
        *X = x_;
    }

    Mat<T> getSigma()
    {
        return *Sigma;
    }   
    
    Mat<T> getKi()
    {
        return *Ki;
    }
    
    Mat<T> getKGain()
    {
        return K;
    }


/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/
/*-------------------------------------------------------------*/       
    
    
    Mat<T> predictState()       /*return the computed predicted state*/
    {        
        if(!extended )
        {
            return (*A)*(*X)+(*B)*(*u) ;
        }
        else
        {
            if(ptrJMotionCommand == NULL )
            {                
                return  ptrMotion(*X, *u, dt) ;
            }
            else
            {                
                return ptrJMotion(dX,(T)(0)*(*u),dt)*(*X-dX) + ptrJMotionCommand(dX,(T)(0)*(*u),dt)*(*u) ;
            }
        }
    }
    
    
    Mat<T> predictCovariance()  /*return the predicted covariance matrix.*/
    {
        if(extended)
            G = ptrJMotion(*X, *u, dt);
            
        //return (filterOn ? ( (!extended ? ((*A)*(*Sigma))* transpose(*A) : G*(*Sigma)*transpose(G) ) + *Q) : *Identity);
        if(filterOn)
        {            
            if(!extended)
            {                
                Mat<T> ret( ((*A)*(*Sigma)) * transpose(*A));
                return ret + *Q;
            }
            else
            {                
                Mat<T> ret(G*(*Sigma)*transpose(G));                
                return ret + *Q;
            }
        }
        else
        {
            return *Identity;
        }
    }
    
        
    Mat<T> calculateKalmanGain()    /*return the Kalman Gain K = C*Sigma_p * (C*Sigma_p*C.T +R).inv */
    {
        if(filterOn)
        {            
            if(extended)
            {
                H = ptrJSensor(X_p, *u, dX,dt);
                if(H.get(1,1) == (T)0)
                {
                    H = Mat<T>((T)0,H.getLine(),H.getColumn());
                    for(int i=H.getLine();i--;) H.set((T)1,i+1,i+1);
                }
            }
            
            Mat<T> temp(invGJ( H * Sigma_p * transpose(H) + *R) );
            
#ifdef verbose_Jacobian             
            //pcm.printf("H : \n\r");
            H.afficher();
            //pcm.printf("sigma_p : \n\r");
            Sigma_p.afficher();
            //pcm.printf("inv GJ : \n\r");
            temp.afficher();
#endif          
            
            

            return ( !extended ? Sigma_p * transpose(*C) * invGJ( (*C) * Sigma_p * transpose(*C) + *R)  : Sigma_p * transpose(H) * temp );
        }
        else
            return *Identity;
    }
    
        
    Mat<T> correctState()       /*update X */
    {              
    
        *_X = *X;
        
        if(filterOn)
            return X_p + K*( (*z) - (!extended ? (*C)*X_p  : ptrSensor(X_p, *u, dX, dt) ) );
        else
            return X_p;                        
    }
    
        
    Mat<T> correctCovariance()  /*update Sigma*/
    {
        if(filterOn)
        {           
            return (*Identity - K* (!extended ? (*C) : H) ) *Sigma_p;
        }
        else
            return *Identity;    
    }
    
    
    void state_Callback()       /* Update all... */
    {        
        if(noise)
        {
            for(int i=R->getLine();i--;)    R->set((T)rgenR->dev(),i+1,i+1);
        }
        //Normal noise for measurement only...        
        
        time += dt;
        
        if( extended && (ptrMotion == NULL || ptrSensor == NULL || ptrJMotion == NULL || ptrJSensor == NULL) )
        {
            //~EKF();
            //pcm.printf("ERREUR : les fonctions ne sont pas initialisees...");
            exit(1);
        }       
        
        X_p = predictState();        
        Sigma_p = predictCovariance();    
               
        K = calculateKalmanGain();              
        
#ifdef debug
        logger.printf("X_p : ");
        transpose(X_p).afficherMblue();
        logger.printf("Sigma_p : \r\n");
        Sigma_p.afficherMblue();
        logger.printf("Kalman gain : \r\n");
        K.afficherMblue();
#endif        
        
        *X = correctState();        
        *Sigma = correctCovariance();
    }
    
    void measurement_Callback(Mat<T> measurements, Mat<T> dX_, bool measure = false)
    {
        if( extended && (ptrMotion == NULL || ptrSensor == NULL || ptrJMotion == NULL || ptrJSensor == NULL) )
        {
            //~EKF();
            //pcm.printf("ERREUR : les fonctions ne sont pas initialisees...");
            exit(1);
        }
        
        dX = dX_;
        
        *z = (!extended || measure ? measurements : ptrSensor(measurements,*u, dX, dt) ); 
        
        if(!measure)
        {
            Mat<T> obsvar(z->getLine(),z->getColumn(), (char)1);
            obsvar.afficher();
            *z = *z + obsvar;
            //pcm.printf("------------Observation :");
            z->afficher();
        }
    }
    
    void measurement_Callback(Mat<T> measurements)
    {
        if( extended && (ptrMotion == NULL || ptrSensor == NULL || ptrJMotion == NULL || ptrJSensor == NULL) )
        {
            //~EKF();
            throw("ERREUR : les fonctions ne sont pas initialisees...");
        }
        
        
        //*z = (!extended ? measurements : ptrSensor(*measurements,*u, dX, dt) );        
        //*z = (!extended ? measurements : ptrSensor(*X,*u, dX, dt) );
        *z = (*C)*(*X);             
    }
    
    
    void computeCommand( Mat<T> desiredX, T dt_, int mode)
    {
        /*
        obs_old = obs_old + dt_*(*z);
        if(!(desiredX == lastdX))
        {
            pidVG->reset();
            pidVD->reset();
            lastdX = desiredX;
        }
        */
        /*work on observation instead of state vector ???? */
        
        /*
        if(dt_ != (T)0 && mode != -1)
        {
            *u = (*Kp)*(desiredX - (*X));
            if(mode >= 1)
                *u = *u + (T)(dt_)*(*Ki)*(desiredX - (*X));
            if(mode >= 2)
                *u = *u + (T)((double)1.0/dt_)*(*Kd)*(desiredX - (*X));
            if(mode >= 3)
                *u = *u + (T)((double)1.0/(dt_*dt_))*(*Kdd)*(desiredX - (*X));                      
        }       
        else if(mode !=-1)
        {
            *u = (*Kp)*(desiredX - (*X));       
        }
        */
        logger.printf("debut commande");
        Timer time;
        time.start();
        
        if( dt_ != (T)0 && mode == 1)
        {
            T r = (T)36;  /*radius*/
            T l = (T)220;   /*entre-roue*/
            T phi_max = (T)100.0;                                        
            
            T rho = X->get(1,1)-dX.get(1,1);
            T alpha = atan2((dX.get(2,1)-X->get(2,1)),( dX.get(1,1) - X->get(1,1)));                        
            alpha = alpha - X->get(3,1);//atan2(sin(X->get(3,1)),cos(X->get(3,1)));                    
            T beta = -X->get(3,1) + dX.get(3,1);    
        
            while(alpha > (T)PI)
            {
                 alpha -= 2.0f*(T)PI;                 
            }
            while(alpha <= -(T)PI)
            {
                alpha += 2.0f*(T)PI;                
            }
            
            while(beta > (T)PI)
            {
                beta -= 2.0f*(T)PI;                
            }
            while(beta <= -(T)PI)
            {
                beta += 2.0f*(T)PI;                
            }
            
            if( fabs_(alpha) >= (T)0.05 && fabs_(rho) >= 2)//&& rho >= 5.0)
            {
                behaviour = 1;
                //real value behaviour
                if(behaviour != lastBehaviour)
                {                    
                    lastBehaviour = behaviour;
                }
                                                            
            }           
            else if(fabs_(rho) >= 2)
            {                
                //real value behaviour
                behaviour = 2;
                if(behaviour != lastBehaviour)
                {
                    lastBehaviour = behaviour;
                }
                
            }                       
            else if(fabs_(beta) >= (T)0.02) 
            {
                behaviour = 3;
                if(behaviour != lastBehaviour)
                {
                    lastBehaviour = behaviour;   
                    pidAngleW->reset();
                    pidPosV->reset();                 
                }
            }
            else
            {                   
                behaviour = 4;
                rho = 0;
                beta = 0;
                alpha = 0;
            }                                    
                                                                                
                                                
            
            /*--------------------------------------------*/
            /*asservissement de vD */                               
            T vd = 0;
            T wd = 0;
                                    
            if(pidPos)
            {
                pidPosV->set( (T)0.01/dt_, (T)0.0, (T)0.0);
                pidPosV->setConsigne(0);
                vd = pidPosV->update(rho);            
                
                pidAngleW->setConsigne(0);
                if(behaviour != 3 && behaviour != 4)
                {
                    pidAngleW->set((T)0.01/dt_, (T)0.0, (T)0.0);                
                    wd = pidAngleW->update(alpha);
                }
                else
                {                    
                    pidAngleW->set((T)0.01/dt_, (T)0.0, (T)0.0);
                    wd = pidAngleW->update(beta);
                }                                        
                
            }                        
                        
            
            Mat<T> Xd(2,1);
            Xd.set(vd, 1,1);
            Xd.set(wd,2,1);
            T phi_r_d = (l/r)*(vd/l+wd);
            T phi_l_d = (l/r)*(vd/l-wd);
            
            
            if(fabs_(phi_r_d) > phi_max)
                phi_r_d = (phi_r_d > 0 ? phi_max : -phi_max);
            if(fabs_(phi_l_d) > phi_max)
                phi_l_d = (phi_l_d > 0 ? phi_max : -phi_max);            
                        
            //Phi consigne             
            phi_consigne.set( phi_l_d, 1,1);            
            phi_consigne.set( phi_r_d, 2,1);            

            //Current value of Phi
            Mat<T> phi_old(2,1);  
            /*state : v w */
            /*                           
            phi_old.set( (l/r)*( X->get(4,1)/l+X->get(5,1) ), 1,1);
            phi_old.set( (l/r)*( X->get(4,1)/l-X->get(5,1) ), 2,1);            
            */
            /*state : phi_l phi_r*/
            phi_old.set( X->get(4,1), 1,1);
            phi_old.set( X->get(5,1), 2,1);            

#ifdef verbose_debug_phi            
            logger.printf("Behaviour = %d \r\n", behaviour);
            logger.printf("Current value of Phi : Consigne : state : \n");
            operatorL( operatorL(transpose(phi_old),transpose(phi_consigne)),transpose(*X)).afficherMblue();            
#endif                                                                                           
                        
            /*erreur phi_consigne - phi_current : asservissement de phi_current
             * asservissement par un Pid de cette dernière relation avec phi_d qui ne change que lorsque phi_consigne chang:
             */
            if( !(lastPhi_consigne == phi_consigne))
            {                                                       
                pidVD->set((T)0.16, (T)0.0, (T)0.0);
                pidVG->set((T)0.306, (T)0.0, (T)0.0);                
                
                pidVD->setConsigne(phi_consigne.get(2,1));
                pidVG->setConsigne(phi_consigne.get(1,1));
                                                
                initPidVG = true;
                initPidVD = true;  
                lastPhi_consigne = phi_consigne;              
            }
            
            
            u->set( pidVG->update( phi_old.get(1,1), dt), 1,1);
            u->set( pidVD->update( phi_old.get(2,1), dt), 2,1);             
            
            //*u = phi_consigne;
                                     
            elapsed_T += dt;
            
        }
        
        logger.printf("fin commande : %f secondes", time.read());
            
        if(mode == -1)
        {
        
            T r = (T)36;  /*radius*/
            T l = (T)220;   /*entre-roue*/
            T phi_max = (T)100.0;                                        
            
            T rho = (T)sqrt((dX.get(1,1)-X->get(1,1))*(dX.get(1,1)-X->get(1,1)) + (dX.get(2,1)-X->get(2,1))*(dX.get(2,1)-X->get(2,1)));
            T alpha = atan2((dX.get(2,1)-X->get(2,1)),( dX.get(1,1) - X->get(1,1)));                        
            alpha = alpha - X->get(3,1);//atan2(sin(X->get(3,1)),cos(X->get(3,1)));            
            //alpha = atan2( sin(alpha), cos(alpha));            
            T beta = -X->get(3,1) + dX.get(3,1);
            beta = atan2(sin(beta),cos(beta));                        
                                   
            while(alpha > (T)PI)
            {
                 alpha -= 2.0f*(T)PI;                 
            }
            while(alpha <= -(T)PI)
            {
                alpha += 2.0f*(T)PI;                
            }
            
            while(beta > (T)PI)
            {
                beta -= 2.0f*(T)PI;                
            }
            while(beta <= -(T)PI)
            {
                beta += 2.0f*(T)PI;                
            }
            //T krho = (double)(rho > 1 ? rho : 0.1);
            //T kalpha = (double)rho*10;
            //T kbeta = (double)2*beta/(rho+1);
            /*original value with behaviour*/
            /*
            T krho = 1;
            T kbeta = 0.2;
            T kalpha = 1-5.0/3.0*kbeta+2.0/PI*krho*rho;            
            */
            
            /*Gestion marche arrière */                        
            Mat<T> normal(2,1);
            normal.set( cos(X->get(3,1)), 1,1);
            normal.set( sin(X->get(3,1)), 2,2);
            Mat<T> tempps(extract((dX-(*X)),1,1,2,1));            
            
            
            if(sum( sum( normal % tempps) ).get(1,1) < (T)0)// Mat<T>((T)0,2,1))
            {
                if(!backward)
                {
                    //if transition --> reset pid :
                    //pidVG->reset();
                    //pidVD->reset();
                    //pidPosV->reset();
                    //pidAngleW->reset();
                }
                
                backward = true;
            }
            else
            {
                if(backward)
                {
                    //if transition --> reset pid :
                    //pidVG->reset();
                    //pidVD->reset();
                    //pidPosV->reset();
                    //pidAngleW->reset();
                }
                
                backward = false;
            }
            
            
            if(backward)
            {
                rho =-rho;
                //it could have been done here but it doesn't work...?
                //we only want to go backward, that is to say that we want vd to be negative.
                alpha = alpha;
                //it is the same alpha that needs to be rediuced...
                beta =beta;
                // idem                
            }            
            
            
            T coeff = (T)1.0f;
            T krho = (T)8.51f*coeff;
            T kbeta = (T)1.6f*coeff;
            T kalpha = (T)3.7f*coeff;
                                             
            if( fabs_(alpha) >= (T)0.01 && rho >= 2)//&& rho >= 5.0)
            {
                behaviour = 1;
                //real value behaviour
                if(behaviour != lastBehaviour)
                {                    
                    lastBehaviour = behaviour;
                }
                                                            
            }           
            else if(rho >= 2)
            {                
                //real value behaviour
                behaviour = 2;
                if(behaviour != lastBehaviour)
                {
                    lastBehaviour = behaviour;
                }
                
            }                       
            else if(fabs_(beta) >= (T)0.02) 
            {
                behaviour = 3;
                if(behaviour != lastBehaviour)
                {
                    lastBehaviour = behaviour;   
                    pidAngleW->reset();
                    pidPosV->reset();                 
                }
            }
            else
            {                   
                behaviour = 4;
                rho = 0;
                beta = 0;
                alpha = 0;
            }                                    
                                                                    
            T vd = krho*rho ;            
            T wd = kalpha*alpha +  kbeta*beta;
                                                
            
            /*--------------------------------------------*/
            /*asservissement de vD */                                                       
            if(pidPos)
            {
                pidPosV->set(-krho, (T)0.0, (T)0.0);
                pidPosV->setConsigne(0);
                vd = pidPosV->update(rho);            
                
                pidAngleW->setConsigne(0);
                if(behaviour != 3 && behaviour != 4)
                {
                    pidAngleW->set(-kalpha, (T)0.0, (T)0.0);                
                    wd = pidAngleW->update(alpha);
                }
                else
                {                    
                    pidAngleW->set(-kalpha, (T)0.0, (T)0.0);
                    wd = pidAngleW->update(beta);
                }                                        
                
            }
            
            if(backward)
            {
                vd = vd;
                //wd = -wd;
            }
                        
            
            Mat<T> Xd(2,1);
            Xd.set(vd, 1,1);
            Xd.set(wd,2,1);
            T phi_r_d = (l/r)*(vd/l+wd);
            T phi_l_d = (l/r)*(vd/l-wd);
            
            
            if(fabs_(phi_r_d) > phi_max)
                phi_r_d = (phi_r_d > 0 ? phi_max : -phi_max);
            if(fabs_(phi_l_d) > phi_max)
                phi_l_d = (phi_l_d > 0 ? phi_max : -phi_max);            
                        
            //Phi consigne             
            phi_consigne.set( phi_r_d, 1,1);            
            phi_consigne.set( phi_l_d, 2,1);            

            //Current value of Phi
            Mat<T> phi_old(2,1);  
            /*state : v w */
            /*                           
            phi_old.set( (l/r)*( X->get(4,1)/l+X->get(5,1) ), 1,1);
            phi_old.set( (l/r)*( X->get(4,1)/l-X->get(5,1) ), 2,1);            
            */
            /*state : phi_r phi_l*/
            phi_old.set( X->get(4,1), 1,1);
            phi_old.set( X->get(5,1), 2,1);            

#ifdef verbose_debug_phi            
            logger.printf("Behaviour = %d \r\n", behaviour);
            logger.printf("Current value of Phi : Consigne : state : \n");
            operatorL( operatorL(transpose(phi_old),transpose(phi_consigne)),transpose(*X)).afficherMblue();            
#endif                                                                                           
                        
            /*erreur phi_consigne - phi_current : asservissement de phi_current
             * asservissement par un Pid de cette dernière relation avec phi_d qui ne change que lorsque phi_consigne chang:
             */
            if( !(lastPhi_consigne == phi_consigne))
            {                                                       
                pidVG->set((T)1.0, (T)0.0, (T)0.005);
                pidVD->set((T)1.010, (T)0.0, (T)0.0055);                
                
                pidVD->setConsigne(phi_consigne.get(2,1));
                pidVG->setConsigne(phi_consigne.get(1,1));
                                                
                initPidVG = true;
                initPidVD = true;  
                lastPhi_consigne = phi_consigne;              
            }
            
            
            u->set( pidVG->update( phi_old.get(1,1), dt), 1,1);
            u->set( pidVD->update( phi_old.get(2,1), dt), 2,1);             
            
            //*u = phi_consigne;
                                     
            elapsed_T += dt;
            
        }
        
        if(mode == -2)
        {
            
            T r = (T)36;  /*radius*/
            T l = (T)220;   /*entre-roue*/
            T phi_max = (T)100.0;
              
            
            T rho = (T)sqrt((dX.get(1,1)-X->get(1,1))*(dX.get(1,1)-X->get(1,1)) + (dX.get(2,1)-X->get(2,1))*(dX.get(2,1)-X->get(2,1)));
            T alpha = atan2( dX.get(2,1)-X->get(2,1), dX.get(1,1) - X->get(1,1) ) - atan2(sin(X->get(3,1)), cos(X->get(3,1)));
            alpha = atan2( sin(alpha), cos(alpha));
            T beta = -X->get(3,1) + dX.get(3,1);
            
            
            /*
            T rho = (T)sqrt((dX.get(1,1)-z->get(1,1))*(dX.get(1,1)-z->get(1,1)) + (dX.get(2,1)-z->get(2,1))*(dX.get(2,1)-z->get(2,1)));
            T alpha = atan2( dX.get(2,1)-z->get(2,1), dX.get(1,1) - z->get(1,1) ) - atan2(sin(z->get(3,1)), cos(z->get(3,1)));
            alpha = atan2( sin(alpha), cos(alpha));
            T beta = -z->get(3,1) + dX.get(3,1);
            */
            //beta = atan2(sin(beta), cos(beta)); 
            
            //T krho = (double)(rho > 1 ? rho : 0.1);
            //T kalpha = (double)rho*10;
            //T kbeta = (double)2*beta/(rho+1);
            /*original value with behaviour*/
            /*
            T krho = 1;
            T kbeta = 0.2;
            T kalpha = 1-5.0/3.0*kbeta+2.0/PI*krho*rho;            
            */
            T krho = 0.5;
            T kbeta = 0.2;
            T kalpha = 0.3;
                                    
            if( fabs_(alpha) >= (T)0.1 && rho >= 5)//&& rho >= 5.0)//( ( (dX.get(1,1)+2 <= X->get(1,1)) || (dX.get(1,1)-2 >= X->get(1,1)) ) || ( (dX.get(2,1)+2 <= X->get(2,1)) || (dX.get(2,1)-2 >= X->get(2,1)) ) ) ) 
            {
                behaviour = 1;
                /*real value behaviour*/
                /*
                krho = kalpha/10000;
                kbeta = 0;
                kalpha = 150*krho;
                behaviour = 1;
                */
                
                /*
                kalpha = kalpha/10000;
                kbeta = 0;
                krho = 20*krho;
                behaviour = 2;          
                */
            }           
            else if(rho >= 5)
            {                
                /*real value behaviour*/
                /*
                kalpha = kalpha/100;
                kbeta = 0;
                //krho = 20*krho;
                */
                behaviour = 2;
                
            }                       
            else if(fabs_(beta) >= 1) 
            {
                //kbeta = kbeta*10;
                //kbeta = kalpha;
                //kalpha = 0;
                //krho = krho/1000;
                behaviour = 3;
            }
            else
            {   
                kbeta = 0;//kbeta/10e10;
                kalpha = 0;
                krho = 0;            
                behaviour = 4;
            }
            
            //pcm.printf("behaviour = %f", behaviour);
                        
            T vd = krho*rho ;
            T wd = kalpha*alpha +  kbeta*beta;
            
            if(fabs_((l/r)*vd) > phi_max )
            {
                //cout << "Vd DEPASSEE : saturation : " << endl;
                if(vd >= 0 )
                    vd = (r/l)*phi_max;
                else
                    vd = -(r/l)*phi_max;
            }
            
            if(fabs_((l/r)*wd) > phi_max )
            {
                //cout << "Wd DEPASSEE : saturation : " << endl;          
                if(wd >= 0 )
                    wd = (r/l)*phi_max;
                else
                    wd = -(r/l)*phi_max;
            }   

            T phi_r_d = (l/r)*(vd/l+wd);
            T phi_l_d = (l/r)*(vd/l-wd);
            
            
            //Phi consigne             
            phi_consigne.set( phi_r_d, 1,1);            
            phi_consigne.set( phi_l_d, 2,1);            

            //Current value of Phi
            Mat<T> phi_old(2,1);                             
            phi_old.set( (l/r)*( X->get(4,1)/l+X->get(5,1) ), 1,1);
            phi_old.set( (l/r)*( X->get(4,1)/l-X->get(5,1) ), 2,1);            

#ifdef verbose_debug_phi            
            logger.printf("Current value of Phi : \n");
            phi_old.afficherMblue();
#endif            
            double acc = default_ACC;
            double dec = default_DEC;   
            //value of the linear acceleration and linear deceleration for the trapezoidal
            Mat<T> phi_dT(2,1);
            Mat<T> phi_d(2,1);
            
             
            if( false )//&& computeTrapezoidalCommandLinear(&phi_dT, phi_old, rho_initial, Vd_max, acc, dec, &phase) )
            {
                phi_consigne=phi_dT;
                
                //pcm.printf("NEW Echelon Phi desired :\n");                
                phi_consigne.afficher();
                //pcm.printf("Vd_max = %f \n",Vd_max);                
            }
            else if( computeCommandLinearG(&phi_dT,lastPhi_desired, phi_consigne,Vd_max,acc,dec) && computeCommandLinearD(&phi_dT,lastPhi_desired,phi_consigne,Vd_max,acc,dec) )
            {
                lastPhi_desired = phi_desired;                
                phi_desired=phi_dT;
#ifdef verbose_debug_phi                            
                //logger.printf("Consigne phi :\n");
                //transpose(phi_consigne).afficherMblue();                
                //logger.printf("NEW Echelon Phi desired :\n");
                operatorL(transpose(phi_desired),transpose(phi_consigne) ).afficherMblue();                
                //logger.printf("Vd_max = %f \n",Vd_max);
#endif            
            }            
            
            /*erreur phi_consigne - phi_desire : asservisseemnt de phi_desire pour faire une rampe*/
            /*erreur phi_desire - phi_current : asservissement de phi_current
             * asservissement par un Pid de cette dernière relation avec phi_d qui ne change que lorsque phi_consigne chang:
             */
            if( !(lastPhi_desired == phi_desired))
            {                
                pidVG->set(1,0,0);
                pidVD->set(1,0,0);
                pidVD->setConsigne(phi_desired.get(2,1));
                pidVG->setConsigne(phi_desired.get(1,1));
                                                
                initPidVG = true;
                initPidVD = true;                
            }
            phi_d.set( pidVG->update( phi_old.get(1,1), dt), 1,1);
            phi_d.set( pidVD->update( phi_old.get(2,1), dt), 2,1); 
            
            
            
            if(eSum.getLine() != 2)
                eSum = (T)0*phi_d;
            if(e_old.getLine() != 2)
                e_old = (T)0*phi_d;                             

            
            //*u = PIDCONTROL( phi_d, phi_old, dt, behaviour);
            u->set(phi_d.get(1,1), 1,1);
            u->set(phi_d.get(2,1), 2,1);
            
            elapsed_T += dt;
            
        }
        
        if(mode == -3)
        {
            *u = (T)(-1)*(*Ki)*((*X)-dX);
        }
    }   
    
    bool computeCommandLinearG(Mat<T>* phi_dT, Mat<T> phi_current, Mat<T> phi_consigne, double Vd_max, double acc, double dec)
    {
        if( phi_consigne.get(1,1) - phi_current.get(1,1) >= (T)0)
        {
            //acceleration
            if(/*0.9**/acc*dt > fabs_(phi_consigne.get(1,1)-phi_current.get(1,1)) )
            {
                //increase only the difference...
                phi_dT->set(phi_consigne.get(1,1), 1,1);                
                return true;
            }
            else if( phi_consigne.get(1,1) == phi_current.get(1,1))
            {
                return true;
            }
            else
            {
                phi_dT->set( /*0.9**/acc*dt+phi_current.get(1,1), 1,1);
                return true;
            }
        }
        else
        {
            //deceleration
            if( /*1.1**/dec*dt > fabs_(phi_consigne.get(1,1)-phi_current.get(1,1)) )
            {
                //increase only the difference...
                phi_dT->set( phi_consigne.get(1,1),1,1);                
                return true;
            }
            else if( phi_consigne.get(1,1) == phi_current.get(1,1))
            {
                return true;
            }
            else
            {
                phi_dT->set( -/*1.1**/dec*dt+phi_current.get(1,1), 1,1);
                return true;
            }
        }                
    
    
    }
    
    bool computeCommandLinearD(Mat<T>* phi_dT, Mat<T> phi_current, Mat<T> phi_consigne, double Vd_max, double acc, double dec)
    {
        if( phi_consigne.get(2,1) - phi_current.get(2,1) >= (T)0)
        {
            //acceleration
            if(acc*dt > fabs_(phi_consigne.get(2,1)-phi_current.get(2,1)))
            {
                //increase only the difference...
                phi_dT->set(phi_consigne.get(2,1), 2,1);                
                return true;
            }
            else if( phi_consigne.get(2,1) == phi_current.get(2,1))
            {
                return true;
            }
            else
            {
                phi_dT->set( 1*acc*dt+phi_current.get(2,1), 2,1);
                return true;
            }
        }
        else
        {
            //deceleration
            if( dec*dt > fabs_(phi_consigne.get(2,1)-phi_current.get(2,1)) )
            {
                //increase only the difference...
                phi_dT->set( phi_consigne.get(2,1),2,1);                
                return true;
            }
            else if( phi_consigne.get(2,1) == phi_current.get(2,1))
            {
                return true;
            }
            else
            {
                phi_dT->set( -1*dec*dt+phi_current.get(2,1), 2,1);
                return true;
            }
        }                    
    
    }
    
    Mat<T> PIDCONTROL( Mat<T> phi_d, Mat<T> u_p, T dt, int behaviour = 3)
    {
        Mat<T> r(phi_d-u_p); 
        /*behaviour test...
        if(lastBehaviour != behaviour)       
            eSum = (T)0*eSum;
        */
        if(behaviour == 3 && lastBehaviour != behaviour)        
            eSum = ((T)0)*eSum;                    
        lastBehaviour = behaviour;
        eSum = eSum + r;
        Mat<T> eDiff(r-e_old);
        e_old = r;
        
        Mat<T> P((T)0.01, 2,1);
        Mat<T> I((T)0.00001, 2,1);
        Mat<T> D((T)0, 2,1);

        /*PID ANGULAR ROTATION : PERFECT*/
        //Mat<T> P((T)-0.01, 2,1);
        //Mat<T> I((T)50, 2,1);
        //Mat<T> D((T)0, 2,1);    
        
        /*-------------------------------*/        
        if(behaviour==2)
        {
            
            P = Mat<T>((T)0, 2,1);
            I = Mat<T>((T)0, 2,1);            
            D = Mat<T>((T)0,2,1); 
            
            P.set(1000,1,1);//gauche
            P.set(1000,2,1);
    
        }
        
        else if(behaviour ==1 || behaviour == 3)
        {
            P = Mat<T>((T)0, 2,1);
            I = Mat<T>((T)0, 2,1);
            D = Mat<T>((T)0,2,1);                          
            
            P.set(1000,1,1);
            P.set(1000,2,1);
        }        
    
#ifdef verbose_pid      
        cout << "Error = " << endl;
        r.afficher();
        cout << " eSum = " << endl;
        eSum.afficher();
        cout << "eDiff = " << endl;
        eDiff.afficher();
#endif      
        
        r = P%r + I%eSum + /*((double)1.0/dt)*/D%eDiff; 
        
#ifdef verbose_pid             
        //pcm.printf(" New value COMMAND = \n");
        r.afficher();
#endif

    
        return r;
    }
    
    
    void resetPID()
    {   
        pidPosV->reset();
        pidAngleW->reset();
        pidVD->reset();
        pidVG->reset();
    }

};