/*******************************************************************************************
* FILE: Coordonator
* File name: main.cpp
* Authors: Pierluc Asselin & Marc-Antoine Dupuis
* Object: S5 Info - APP5
* Original Date: 22/02/2014
* Last Modified: 25/02/2014
********************************************************************************************/


/*******************
    INCLUDES
*******************/
#include "mbed.h"
#include <string>
#include <sstream>
#include "ConfigFile.h"
#include "EthernetInterface.h"
#include "Websocket.h"


/*******************
    PINS
*******************/
DigitalOut myled(LED1);
DigitalOut myled2(LED2);
DigitalOut reset(p8);


/*******************
    SERIAL
*******************/
Serial pc(USBTX, USBRX);
Serial xbee(p13, p14);


/*******************
    VARIABLES
*******************/
char url_s[BUFSIZ];
char dataBuf[20];
int dataBuf_i;
bool trame_ready;
string panID;


/*******************
    ETHERNET
*******************/
EthernetInterface eth;


/*******************
    CONFIG FILE
*******************/
LocalFileSystem local("local");
ConfigFile cfg;

/*******************
 STATE MACHINE ENUM
*******************/
enum {STEP_START, STEP_LENGTH, STEP_TYPE, STEP_MAC, STEP_NET, STEP_OPT, STEP_DATA, STEP_CHECK};


/*******************
configInit
This function get the information from the configuration init file.
*******************/
void configInit()
{
    char *key1 = "PANID";
    char *key2 = "URL";
    char value[BUFSIZ];
    
    // Read a configuration file from a mbed.   
    if (!cfg.read("/local/initconf.cfg")) 
    {
        error("Failure to read a configuration file.\n\r");
    }
 
    /*
     * Get a configuration value.
     */
    if (cfg.getValue(key1, &value[0], sizeof(value))) 
    {
        // Get PanID
        std::stringstream panID_s(value); 
        panID_s >> panID;
    }
    
    if (cfg.getValue(key2, &value[0], sizeof(value))) 
    {
        // Get URL of web socket
        memcpy(url_s,value,BUFSIZ);
    }
}
/*******************
eth_ws_init
This function initialize the ethernet port and the web socket connection
*******************/
void eth_ws_init(Websocket* ws)
{
    // Init and connect to ethernet port
    eth.init();
    eth.connect();
    
    // Get IP adress
    pc.printf("IP Address is %s\n\r", eth.getIPAddress());
    
    // Wait 2 seconds since we just connected to the network
    wait(2);

    // Connect to the web socket
    if(!ws->connect())
    {
         error("Failure establish connection to the WebSocket.\n\r");
    }
}
/*******************
xbee_init
This function initialize the xbee module and set his panID
*******************/
void xbee_init()
{
    // Reset the xbee
    reset = 0;
    wait_ms(400);
    reset = 1;
    
    // Wait for the xbee to be ready to receive data    
    wait(4);
    
    // Build trame and send the PanID to the xbee
    int panID_size;
    int panID_array[8];
    int i;
    long int panID_d;
    
    panID_size = panID.length(); //vérifie si le panID est pair ou impair
    if(panID_size%2 != 0)
    {
        panID_size++;
        panID = "0" + panID;
    }
    
    panID_d = strtol(panID.c_str(),NULL,16); //converti le panID en chiffre
    for(i=0;i<panID_size/2;i++)
    {
        panID_array[i] = panID_d%256; //decoupe en packet de 8 bytes (2 characteres)
        panID_d = panID_d >> 8;
        
        
    }
    i--; //i est incrémentré une fois de trop dans le for avant de sortir de la boucle
    
    
    //construction des 3 trames pour changer le panID
    char size = panID_size/2 + 0x04; //length + 0x04 
    char checkSum = 0;
    xbee.putc(0x7E); // start delimeter
    xbee.putc(0x00); // length
    xbee.putc(size); // length
    xbee.putc(0x09); // frame type : AT command queue
    xbee.putc(0x47); // frame ID
    xbee.putc(0x49); //AT command : (ID)
    xbee.putc(0x44); //AT command : (ID)
    checkSum = checkSum + 0x09 + 0x47 + 0x49 + 0x44;
    while(i>=0)
    {
        xbee.putc(panID_array[i]);
        checkSum += panID_array[i];
        i--;
    }
    checkSum = 0xff - checkSum;
    xbee.putc(checkSum); // checksum

    xbee.putc(0x7E); // start delimeter
    xbee.putc(0x00); // length
    xbee.putc(0x04); // length
    xbee.putc(0x09); // frame type : AT command queue
    xbee.putc(0x49); // frame ID
    xbee.putc(0x57); //AT command : (WR)
    xbee.putc(0x52); //AT command : (WR)
    xbee.putc(0x04); //checksum
    
    
    xbee.putc(0x7E); // start delimeter
    xbee.putc(0x00); // length
    xbee.putc(0x04); // length
    xbee.putc(0x09); // frame type : AT command
    xbee.putc(0x4A); // frame ID
    xbee.putc(0x41); //AT command : (AC)
    xbee.putc(0x43); //AT command : (AC)
    xbee.putc(0x28); //checksum

}
/*******************
xbee_receive
This function receive and decode the information from the xbee communication
*******************/
void xbee_receive()
{
    // Initialize step-machine
    static int state = STEP_START;
    
    // Get the next caracter from xbee
    char data = xbee.getc();
    
    static int length_i;
    static int length;  
    static int mac_i;   
    static int net_i;
    
    // Step-Machine
    switch(state)
    {
        case STEP_START:    if(data == 0x7E) // Verifie if Start byte is OK.
                            {
                                state = STEP_LENGTH;
                                length_i = 0;
                                length = 0;
                                dataBuf_i = 0;
                            }       
                            break;
                            
        case STEP_LENGTH:   length += data; //Read the length in the next 2 bytes.
                            length_i++;
                            if(length_i == 2)
                            {
                                state = STEP_TYPE;   
                                length -= 12;
                            }
                            break;
                            
        case STEP_TYPE:     if(data == 0x90) // Verifie if Type byte is OK.
                            {
                                state = STEP_MAC;  
                                mac_i = 0;  
                            }
                            else
                            {
                                state = STEP_START;
                            }
                            break;
                            
        case STEP_MAC:      mac_i++;    // Skip the next 8 bytes. We do not need the 64bits MAC adresse since we already receive the data. xbee already accepted it
                            if(mac_i == 8)
                            {
                                state = STEP_NET;
                                net_i = 0;
                            }
                            break;
                            
        case STEP_NET:      net_i++; // Skip the next 2 bytes. We do not need the 16bits adresse since we already receive the data. xbee already accepted it
                            if(net_i == 2)
                            {
                                state = STEP_OPT;
                            }
                            break;
                            
        case STEP_OPT:      if(data == 0x01)  // Verifie that the option bytes is OK
                            {
                                state = STEP_DATA;    
                            }
                            else
                            {
                                state = STEP_START;
                            }
                            break;                     
                            
        case STEP_DATA:     length--;   // Get the data. The number of bytes to read is given by the length previously detected.
                            dataBuf[dataBuf_i] = data;
                            dataBuf_i++;
                            if(length == 0)
                            {
                                dataBuf[dataBuf_i] = '\0';
                                state = STEP_CHECK; 
                                trame_ready = true;
                            }
                            break;
                            
        case STEP_CHECK:    //check ChecksumS
                            state = STEP_START;
                            break;                            
    }
  
}
/*******************
main
This is the main function of the coordinator program
*******************/
int main() 
{
    // Get the data from config init file
    configInit();

    // Create the web socket with the specified URl from the cofig init file
    Websocket ws(url_s);
    
    // Initialize the ethernet port and the web socket
    eth_ws_init(&ws);
    
    // initialize the xbee communication and the PAN
    xbee_init();

    // Light on SENDING led and RECEIVING led (led1, led2)
    myled = 1;
    myled2 = 1;
    
    // initialise the confimation received flag
    trame_ready = false;
    
    //Main loop
    while(1) 
    {
        // If data is received from the xbee, toggle RECEIVING led and analyse the data
        if(xbee.readable())
        {
            xbee_receive();
            myled2 = ! myled2;
        }
        
        // if received trame is complete, send it to the web socket and reset the confirmation flag.
        if(trame_ready == true)
        {          
            ws.send(dataBuf);           
            trame_ready = false;
        }
    }
}
