/*******************************************************************************************
* FILE: Router
* File name: main.cpp
* Authors: Pierluc Asselin & Marc-Antoine Dupuis
* Object: S5 Info - APP5
* Original Date: 22/02/2014
* Last Modified: 25/02/2014
********************************************************************************************/

/*******************
    INCLUDES
*******************/
#include "mbed.h"
#include <sstream>
#include <stdlib.h>
#include <string>
#include "ConfigFile.h"
 

/**********************************
    DEFINES REGISTRES POUR I2C
***********************************/
#define MMA8452_WRITE_ADDRESS 0x3A
#define MMA8452_READ_ADDRESS  0x3B
#define WHO_AM_I        0x0D
#define OUT_X_MSB       0x01
#define OUT_X_LSB       0x02
#define OUT_Y_MSB       0x03
#define OUT_Y_LSB       0x04
#define OUT_Z_MSB       0x05
#define OUT_Z_LSB       0x06
#define CTRL_REG1       0x2A 
#define CTRL_REG2       0x2B


/*******************
    PINS
*******************/
DigitalOut myled(LED1);
DigitalIn pb1(p5);
DigitalOut reset(p8);
Serial pc(USBTX, USBRX);
Serial xbee(p13, p14);
I2C i2c(p9, p10);            // I2C port (sda, scl)

/*******************
    CONFIG FILE
*******************/
LocalFileSystem local("local");
ConfigFile cfg;

/***********************
    VARIABLES GLOBALES
************************/
int period_ms; //frequence que l'on obtien dans config file
string panID; //panID que l'on obtient dans config file
 
 
 /*******************
    convertInt
    Converti un int en string
*******************/
 string convertInt(int number)  
{
   stringstream ss;//create a stringstream
   ss.str("");
   ss << number;//add int number to the stream
   return ss.str();//return a string with the contents of the stream
}

 /*******************
    configInit
    fonction de config avec le config file
*******************/
void configInit() 
{
    char *key1 = "PANID";
    char *key2 = "FREQ";
    char value[BUFSIZ];
    
    // Read a configuration file from a mbed.   
    if (!cfg.read("/local/initconf.cfg")) 
    {
        error("Failure to read a configuration file.\n");
    }
 
    
    // Get a configuration value.
     
    if (cfg.getValue(key1, &value[0], sizeof(value))) 
    {
        std::stringstream panID_s(value); //transforme la valeur du panID en string
        panID_s >> panID; //place le string dans la variable globale
    }
    
    if (cfg.getValue(key2, &value[0], sizeof(value))) 
    {
        std::stringstream freq_s(value);  //transforme la frequence en string
        freq_s >> period_ms; //place le string dans la variable globale
    }
}

 /*******************
    xbee_init
    fonction de config du module xBee
*******************/
void xbee_init() 
{
    reset = 0; //initialisation
    wait_ms(400);
    reset = 1;
    wait(4);
    //variables
    int panID_size;
    int panID_array[8];
    int i;
    long int panID_d;
    
    panID_size = panID.length(); //vérifie si le panID est pair ou impair
    if(panID_size%2 != 0)
    {
        panID_size++; //si impair size++ (i.e. on veut que 5/2 = 3 au lieu de 2)
        panID = "0" + panID;
    }
    
    panID_d = strtol(panID.c_str(),NULL,16); //converti le panID en chiffre
    for(i=0;i<panID_size/2;i++)
    {
        panID_array[i] = panID_d%256; //decoupe en packet de 8 bytes (2 characteres) et place dans tableau
        panID_d = panID_d >> 8;       //à partir de la fin (i.e. CAFE --> array[0] = FE, array[1] = CA )
        
        
    }
    i--; //i est incrémentré une fois de trop dans le for avant de sortir de la boucle
    
    
    //construction des 3 trames pour changer le panID
    char size = panID_size/2 + 0x04; //length + 0x04 
    char checkSum = 0;
    xbee.putc(0x7E); // start delimeter
    xbee.putc(0x00); // length
    xbee.putc(size); // length
    xbee.putc(0x09); // frame type : AT command queue
    xbee.putc(0x47); // frame ID
    xbee.putc(0x49); //AT command : (ID)
    xbee.putc(0x44); //AT command : (ID)
    checkSum = checkSum + 0x09 + 0x47 + 0x49 + 0x44;
    while(i>=0)
    {                                                  
        xbee.putc(panID_array[i]);                   //putc avec les valeur du tableau 
        checkSum += panID_array[i];      //les valeurs sont mises à l'envers dans le tableau               
        i--;//on doit donc décrémenter le i
    }
    checkSum = 0xff - checkSum;
    xbee.putc(checkSum); // checksum
    

    xbee.putc(0x7E); // start delimeter
    xbee.putc(0x00); // length
    xbee.putc(0x04); // length
    xbee.putc(0x09); // frame type : AT command queue
    xbee.putc(0x49); // frame ID
    xbee.putc(0x57); //AT command : (WR)
    xbee.putc(0x52); //AT command : (WR)
    xbee.putc(0x04); //checksum
    
    
    xbee.putc(0x7E); // start delimeter
    xbee.putc(0x00); // length
    xbee.putc(0x04); // length
    xbee.putc(0x09); // frame type : AT command
    xbee.putc(0x4A); // frame ID
    xbee.putc(0x41); //AT command : (AC)
    xbee.putc(0x43); //AT command : (AC)
    xbee.putc(0x28); //checksum

}

char i2c_read_reg(char address) //fonction i2c pour l'accelerometre
{   
    // Read from selected register adress
    i2c.start();                                         
    i2c.write(MMA8452_WRITE_ADDRESS);   
    i2c.write(address);                 
    i2c.start();                        
    i2c.write(MMA8452_READ_ADDRESS);    
    char data = i2c.read(0);            
    i2c.stop();                         
 
    // return the data readed
    return data;   
}

bool initAccel() //fonction qui initialise l'accelerometre avec i2c
{
    // Start I2C communication   
    char data = i2c_read_reg(WHO_AM_I);
    if(data != 0x2A)
    {
        return false;
    }
    
    // Set accelerometer in Low Noise Low Power mode
    i2c.start();                           
    i2c.write(MMA8452_WRITE_ADDRESS); 
    i2c.write(CTRL_REG2); 
    i2c.write(0x01); 
    i2c.stop(); 
    
    // Set accelerometer in active mode
    i2c.start();                           
    i2c.write(MMA8452_WRITE_ADDRESS); 
    i2c.write(CTRL_REG1); 
    i2c.write(0x01); 
    i2c.stop();  
    
    return true;     
}


 /*******************
    getAccelValue
    fonction qui lit les registres de données de l'accelerometre
*******************/
unsigned short getAccelValue(char MSB_addr) //fonction qui lit les registres de données de l'accelerometre
{
    return i2c_read_reg(MSB_addr);           // Read MSB data from MSB register
}


 /*******************
    getAccelData
    fonction qui construit la string a envoyer par xbee de l'accélérometre
*******************/
string getAccelData()
{
    unsigned short x, y, z; 
        
    // Get accelerometer values
    x= getAccelValue(OUT_X_MSB);
    y= getAccelValue(OUT_Y_MSB);
    z= getAccelValue(OUT_Z_MSB);  
    
    return "AX" + convertInt(x) + " AY" + convertInt(y) + " AZ" + convertInt(z);  
}


 /*******************
    getPushButtonData
    fonction qui construit la string a envoyer par xbee du bouton poussoir
*******************/
string getPushButtonData()
{
     return "PB" + convertInt(pb1);
}


 /*******************
    xbee_transmit
    fonction d'envoie de trame xbee
*******************/
void xbee_transmit(string data_s)
{
       
    char checkSum = 0;
    char size = data_s.length() + 0x0E; //0x0E + data type
    xbee.putc(0x7E); // start delimeter
    xbee.putc(0x00); // length
    xbee.putc(size); // length
    xbee.putc(0x10); // frame type : transmit request
    xbee.putc(0x01); // frame ID
    xbee.putc(0x00); // adress MAC octet 1 de 8
    xbee.putc(0x13); // adress MAC octet 2 de 8
    xbee.putc(0xA2); // adress MAC octet 3 de 8
    xbee.putc(0x00); // adress MAC octet 4 de 8
    xbee.putc(0x40); // adress MAC octet 5 de 8
    xbee.putc(0x8B); // adress MAC octet 6 de 8
    xbee.putc(0x41); // adress MAC octet 7 de 8
    xbee.putc(0x98); // adress MAC octet 8 de 8
    xbee.putc(0x00); // dest. adress network octet 1 de 2
    xbee.putc(0x00); // dest. adress network octet 2 de 2
    xbee.putc(0x00); // broadcast radius
    xbee.putc(0x00); // option
    checkSum = checkSum + 0x10 + 0x01 + 0x00 + 0x13+ 0xA2 + 0x00 + 0x40 + 0x8B +0x41 + 0x98 + 0x00 + 0x00 + 0x00 + 0x00;
    
    for (int i=0;i<data_s.length();i++)  //data
    {
        xbee.putc(data_s[i]); //envoie le data de la string passée en parametre
        checkSum += data_s[i]; //additionne le checksum de chaque data
    }
    checkSum = 0xff - checkSum;
    xbee.putc(checkSum); // checksum
    
}


 /*******************
 
   MAIN
   
*******************/
int main() 
{
    //appelle des fonctions d'initialisations
    configInit();   //fichier de config
    xbee_init();    //xbee
    initAccel();    //accelerometre
    
    myled = 0;
    
  
    while(1) 
    {
        wait_ms(period_ms);
        myled = !myled;
        
        //Transmit PUSHBUTTON to coordonitator
        xbee_transmit(getPushButtonData()); 
        
        //Transmit ACCEL xyz to coordonitator
        xbee_transmit(getAccelData());

       

        
        
         
    }
}

