/*******************************************************************\
*                                                                   *
*   Library         : lib_crc                                       *
*   File            : lib_crc.c                                     *
*   Author          : Lammert Bies  1999-2008                       *
*   E-mail          : info@lammertbies.nl                           *
*   Language        : ANSI C                                        *
*                                                                   *
*                                                                   *
*   Description                                                     *
*   ===========                                                     *
*                                                                   *
*   The file lib_crc.c contains the private  and  public  func-     *
*   tions  used  for  the  calculation of CRC-16, CRC-CCITT and     *
*   CRC-32 cyclic redundancy values.                                *
*                                                                   *
*                                                                   *
*   Dependencies                                                    *
*   ============                                                    *
*                                                                   *
*   lib_crc.h       CRC definitions and prototypes                  *
*                                                                   *
*                                                                   *
*   Modification history                                            *
*   ====================                                            *
*                                                                   *
*   Date        Version Comment                                     *
*                                                                   *
*   2010-10-24  2.0     Rewritten to add more algos (JP)            *
*   2010-10-20  1.17    Added several routines (JP)                 *
*   2008-04-20  1.16    Added CRC-CCITT calculation for Kermit      *
*                                                                   *
*   2007-04-01  1.15    Added CRC16 calculation for Modbus          *
*                                                                   *
*   2007-03-28  1.14    Added CRC16 routine for Sick devices        *
*                                                                   *
*   2005-12-17  1.13    Added CRC-CCITT with initial 0x1D0F         *
*                                                                   *
*   2005-05-14  1.12    Added CRC-CCITT with start value 0          *
*                                                                   *
*   2005-02-05  1.11    Fixed bug in CRC-DNP routine                *
*                                                                   *
*   2005-02-04  1.10    Added CRC-DNP routines                      *
*                                                                   *
*   1999-02-21  1.01    Added FALSE and TRUE mnemonics              *
*                                                                   *
*   1999-01-22  1.00    Initial source                              *
*                                                                   *
\*******************************************************************/
#include <stdint.h>

const uint8_t crc8_table_atm[] =
{
0x00, 0x07, 0x0E, 0x09, 0x1C, 0x1B, 0x12, 0x15, 0x38, 0x3F, 0x36, 0x31, 0x24, 0x23, 0x2A, 0x2D,
0x70, 0x77, 0x7E, 0x79, 0x6C, 0x6B, 0x62, 0x65, 0x48, 0x4F, 0x46, 0x41, 0x54, 0x53, 0x5A, 0x5D,
0xE0, 0xE7, 0xEE, 0xE9, 0xFC, 0xFB, 0xF2, 0xF5, 0xD8, 0xDF, 0xD6, 0xD1, 0xC4, 0xC3, 0xCA, 0xCD,
0x90, 0x97, 0x9E, 0x99, 0x8C, 0x8B, 0x82, 0x85, 0xA8, 0xAF, 0xA6, 0xA1, 0xB4, 0xB3, 0xBA, 0xBD,
0xC7, 0xC0, 0xC9, 0xCE, 0xDB, 0xDC, 0xD5, 0xD2, 0xFF, 0xF8, 0xF1, 0xF6, 0xE3, 0xE4, 0xED, 0xEA,
0xB7, 0xB0, 0xB9, 0xBE, 0xAB, 0xAC, 0xA5, 0xA2, 0x8F, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9D, 0x9A,
0x27, 0x20, 0x29, 0x2E, 0x3B, 0x3C, 0x35, 0x32, 0x1F, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0D, 0x0A,
0x57, 0x50, 0x59, 0x5E, 0x4B, 0x4C, 0x45, 0x42, 0x6F, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7D, 0x7A,
0x89, 0x8E, 0x87, 0x80, 0x95, 0x92, 0x9B, 0x9C, 0xB1, 0xB6, 0xBF, 0xB8, 0xAD, 0xAA, 0xA3, 0xA4,
0xF9, 0xFE, 0xF7, 0xF0, 0xE5, 0xE2, 0xEB, 0xEC, 0xC1, 0xC6, 0xCF, 0xC8, 0xDD, 0xDA, 0xD3, 0xD4,
0x69, 0x6E, 0x67, 0x60, 0x75, 0x72, 0x7B, 0x7C, 0x51, 0x56, 0x5F, 0x58, 0x4D, 0x4A, 0x43, 0x44,
0x19, 0x1E, 0x17, 0x10, 0x05, 0x02, 0x0B, 0x0C, 0x21, 0x26, 0x2F, 0x28, 0x3D, 0x3A, 0x33, 0x34,
0x4E, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5C, 0x5B, 0x76, 0x71, 0x78, 0x7F, 0x6A, 0x6D, 0x64, 0x63,
0x3E, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2C, 0x2B, 0x06, 0x01, 0x08, 0x0F, 0x1A, 0x1D, 0x14, 0x13,
0xAE, 0xA9, 0xA0, 0xA7, 0xB2, 0xB5, 0xBC, 0xBB, 0x96, 0x91, 0x98, 0x9F, 0x8A, 0x8D, 0x84, 0x83,
0xDE, 0xD9, 0xD0, 0xD7, 0xC2, 0xC5, 0xCC, 0xCB, 0xE6, 0xE1, 0xE8, 0xEF, 0xFA, 0xFD, 0xF4, 0xF3
};

const uint8_t crc8_table_ccitt[] =
{
0x00, 0x8D, 0x97, 0x1A, 0xA3, 0x2E, 0x34, 0xB9, 0xCB, 0x46, 0x5C, 0xD1, 0x68, 0xE5, 0xFF, 0x72,
0x1B, 0x96, 0x8C, 0x01, 0xB8, 0x35, 0x2F, 0xA2, 0xD0, 0x5D, 0x47, 0xCA, 0x73, 0xFE, 0xE4, 0x69,
0x36, 0xBB, 0xA1, 0x2C, 0x95, 0x18, 0x02, 0x8F, 0xFD, 0x70, 0x6A, 0xE7, 0x5E, 0xD3, 0xC9, 0x44,
0x2D, 0xA0, 0xBA, 0x37, 0x8E, 0x03, 0x19, 0x94, 0xE6, 0x6B, 0x71, 0xFC, 0x45, 0xC8, 0xD2, 0x5F,
0x6C, 0xE1, 0xFB, 0x76, 0xCF, 0x42, 0x58, 0xD5, 0xA7, 0x2A, 0x30, 0xBD, 0x04, 0x89, 0x93, 0x1E,
0x77, 0xFA, 0xE0, 0x6D, 0xD4, 0x59, 0x43, 0xCE, 0xBC, 0x31, 0x2B, 0xA6, 0x1F, 0x92, 0x88, 0x05,
0x5A, 0xD7, 0xCD, 0x40, 0xF9, 0x74, 0x6E, 0xE3, 0x91, 0x1C, 0x06, 0x8B, 0x32, 0xBF, 0xA5, 0x28,
0x41, 0xCC, 0xD6, 0x5B, 0xE2, 0x6F, 0x75, 0xF8, 0x8A, 0x07, 0x1D, 0x90, 0x29, 0xA4, 0xBE, 0x33,
0xD8, 0x55, 0x4F, 0xC2, 0x7B, 0xF6, 0xEC, 0x61, 0x13, 0x9E, 0x84, 0x09, 0xB0, 0x3D, 0x27, 0xAA,
0xC3, 0x4E, 0x54, 0xD9, 0x60, 0xED, 0xF7, 0x7A, 0x08, 0x85, 0x9F, 0x12, 0xAB, 0x26, 0x3C, 0xB1,
0xEE, 0x63, 0x79, 0xF4, 0x4D, 0xC0, 0xDA, 0x57, 0x25, 0xA8, 0xB2, 0x3F, 0x86, 0x0B, 0x11, 0x9C,
0xF5, 0x78, 0x62, 0xEF, 0x56, 0xDB, 0xC1, 0x4C, 0x3E, 0xB3, 0xA9, 0x24, 0x9D, 0x10, 0x0A, 0x87,
0xB4, 0x39, 0x23, 0xAE, 0x17, 0x9A, 0x80, 0x0D, 0x7F, 0xF2, 0xE8, 0x65, 0xDC, 0x51, 0x4B, 0xC6,
0xAF, 0x22, 0x38, 0xB5, 0x0C, 0x81, 0x9B, 0x16, 0x64, 0xE9, 0xF3, 0x7E, 0xC7, 0x4A, 0x50, 0xDD,
0x82, 0x0F, 0x15, 0x98, 0x21, 0xAC, 0xB6, 0x3B, 0x49, 0xC4, 0xDE, 0x53, 0xEA, 0x67, 0x7D, 0xF0,
0x99, 0x14, 0x0E, 0x83, 0x3A, 0xB7, 0xAD, 0x20, 0x52, 0xDF, 0xC5, 0x48, 0xF1, 0x7C, 0x66, 0xEB
};

const uint8_t crc8_table_maxim[] =
{
0x00, 0x5E, 0xBC, 0xE2, 0x61, 0x3F, 0xDD, 0x83, 0xC2, 0x9C, 0x7E, 0x20, 0xA3, 0xFD, 0x1F, 0x41,
0x9D, 0xC3, 0x21, 0x7F, 0xFC, 0xA2, 0x40, 0x1E, 0x5F, 0x01, 0xE3, 0xBD, 0x3E, 0x60, 0x82, 0xDC,
0x23, 0x7D, 0x9F, 0xC1, 0x42, 0x1C, 0xFE, 0xA0, 0xE1, 0xBF, 0x5D, 0x03, 0x80, 0xDE, 0x3C, 0x62,
0xBE, 0xE0, 0x02, 0x5C, 0xDF, 0x81, 0x63, 0x3D, 0x7C, 0x22, 0xC0, 0x9E, 0x1D, 0x43, 0xA1, 0xFF,
0x46, 0x18, 0xFA, 0xA4, 0x27, 0x79, 0x9B, 0xC5, 0x84, 0xDA, 0x38, 0x66, 0xE5, 0xBB, 0x59, 0x07,
0xDB, 0x85, 0x67, 0x39, 0xBA, 0xE4, 0x06, 0x58, 0x19, 0x47, 0xA5, 0xFB, 0x78, 0x26, 0xC4, 0x9A,
0x65, 0x3B, 0xD9, 0x87, 0x04, 0x5A, 0xB8, 0xE6, 0xA7, 0xF9, 0x1B, 0x45, 0xC6, 0x98, 0x7A, 0x24,
0xF8, 0xA6, 0x44, 0x1A, 0x99, 0xC7, 0x25, 0x7B, 0x3A, 0x64, 0x86, 0xD8, 0x5B, 0x05, 0xE7, 0xB9,
0x8C, 0xD2, 0x30, 0x6E, 0xED, 0xB3, 0x51, 0x0F, 0x4E, 0x10, 0xF2, 0xAC, 0x2F, 0x71, 0x93, 0xCD,
0x11, 0x4F, 0xAD, 0xF3, 0x70, 0x2E, 0xCC, 0x92, 0xD3, 0x8D, 0x6F, 0x31, 0xB2, 0xEC, 0x0E, 0x50,
0xAF, 0xF1, 0x13, 0x4D, 0xCE, 0x90, 0x72, 0x2C, 0x6D, 0x33, 0xD1, 0x8F, 0x0C, 0x52, 0xB0, 0xEE,
0x32, 0x6C, 0x8E, 0xD0, 0x53, 0x0D, 0xEF, 0xB1, 0xF0, 0xAE, 0x4C, 0x12, 0x91, 0xCF, 0x2D, 0x73,
0xCA, 0x94, 0x76, 0x28, 0xAB, 0xF5, 0x17, 0x49, 0x08, 0x56, 0xB4, 0xEA, 0x69, 0x37, 0xD5, 0x8B,
0x57, 0x09, 0xEB, 0xB5, 0x36, 0x68, 0x8A, 0xD4, 0x95, 0xCB, 0x29, 0x77, 0xF4, 0xAA, 0x48, 0x16,
0xE9, 0xB7, 0x55, 0x0B, 0x88, 0xD6, 0x34, 0x6A, 0x2B, 0x75, 0x97, 0xC9, 0x4A, 0x14, 0xF6, 0xA8,
0x74, 0x2A, 0xC8, 0x96, 0x15, 0x4B, 0xA9, 0xF7, 0xB6, 0xE8, 0x0A, 0x54, 0xD7, 0x89, 0x6B, 0x35
};

const uint8_t crc8_table[] =
{
0x00, 0xD5, 0x7F, 0xAA, 0xFE, 0x2B, 0x81, 0x54, 0x29, 0xFC, 0x56, 0x83, 0xD7, 0x02, 0xA8, 0x7D,
0x52, 0x87, 0x2D, 0xF8, 0xAC, 0x79, 0xD3, 0x06, 0x7B, 0xAE, 0x04, 0xD1, 0x85, 0x50, 0xFA, 0x2F,
0xA4, 0x71, 0xDB, 0x0E, 0x5A, 0x8F, 0x25, 0xF0, 0x8D, 0x58, 0xF2, 0x27, 0x73, 0xA6, 0x0C, 0xD9,
0xF6, 0x23, 0x89, 0x5C, 0x08, 0xDD, 0x77, 0xA2, 0xDF, 0x0A, 0xA0, 0x75, 0x21, 0xF4, 0x5E, 0x8B,
0x9D, 0x48, 0xE2, 0x37, 0x63, 0xB6, 0x1C, 0xC9, 0xB4, 0x61, 0xCB, 0x1E, 0x4A, 0x9F, 0x35, 0xE0,
0xCF, 0x1A, 0xB0, 0x65, 0x31, 0xE4, 0x4E, 0x9B, 0xE6, 0x33, 0x99, 0x4C, 0x18, 0xCD, 0x67, 0xB2,
0x39, 0xEC, 0x46, 0x93, 0xC7, 0x12, 0xB8, 0x6D, 0x10, 0xC5, 0x6F, 0xBA, 0xEE, 0x3B, 0x91, 0x44,
0x6B, 0xBE, 0x14, 0xC1, 0x95, 0x40, 0xEA, 0x3F, 0x42, 0x97, 0x3D, 0xE8, 0xBC, 0x69, 0xC3, 0x16,
0xEF, 0x3A, 0x90, 0x45, 0x11, 0xC4, 0x6E, 0xBB, 0xC6, 0x13, 0xB9, 0x6C, 0x38, 0xED, 0x47, 0x92,
0xBD, 0x68, 0xC2, 0x17, 0x43, 0x96, 0x3C, 0xE9, 0x94, 0x41, 0xEB, 0x3E, 0x6A, 0xBF, 0x15, 0xC0,
0x4B, 0x9E, 0x34, 0xE1, 0xB5, 0x60, 0xCA, 0x1F, 0x62, 0xB7, 0x1D, 0xC8, 0x9C, 0x49, 0xE3, 0x36,
0x19, 0xCC, 0x66, 0xB3, 0xE7, 0x32, 0x98, 0x4D, 0x30, 0xE5, 0x4F, 0x9A, 0xCE, 0x1B, 0xB1, 0x64,
0x72, 0xA7, 0x0D, 0xD8, 0x8C, 0x59, 0xF3, 0x26, 0x5B, 0x8E, 0x24, 0xF1, 0xA5, 0x70, 0xDA, 0x0F,
0x20, 0xF5, 0x5F, 0x8A, 0xDE, 0x0B, 0xA1, 0x74, 0x09, 0xDC, 0x76, 0xA3, 0xF7, 0x22, 0x88, 0x5D,
0xD6, 0x03, 0xA9, 0x7C, 0x28, 0xFD, 0x57, 0x82, 0xFF, 0x2A, 0x80, 0x55, 0x01, 0xD4, 0x7E, 0xAB,
0x84, 0x51, 0xFB, 0x2E, 0x7A, 0xAF, 0x05, 0xD0, 0xAD, 0x78, 0xD2, 0x07, 0x53, 0x86, 0x2C, 0xF9
};

const uint8_t crc8_table_j1850[] =
{
0x00, 0x1D, 0x3A, 0x27, 0x74, 0x69, 0x4E, 0x53, 0xE8, 0xF5, 0xD2, 0xCF, 0x9C, 0x81, 0xA6, 0xBB,
0xCD, 0xD0, 0xF7, 0xEA, 0xB9, 0xA4, 0x83, 0x9E, 0x25, 0x38, 0x1F, 0x02, 0x51, 0x4C, 0x6B, 0x76,
0x87, 0x9A, 0xBD, 0xA0, 0xF3, 0xEE, 0xC9, 0xD4, 0x6F, 0x72, 0x55, 0x48, 0x1B, 0x06, 0x21, 0x3C,
0x4A, 0x57, 0x70, 0x6D, 0x3E, 0x23, 0x04, 0x19, 0xA2, 0xBF, 0x98, 0x85, 0xD6, 0xCB, 0xEC, 0xF1,
0x13, 0x0E, 0x29, 0x34, 0x67, 0x7A, 0x5D, 0x40, 0xFB, 0xE6, 0xC1, 0xDC, 0x8F, 0x92, 0xB5, 0xA8,
0xDE, 0xC3, 0xE4, 0xF9, 0xAA, 0xB7, 0x90, 0x8D, 0x36, 0x2B, 0x0C, 0x11, 0x42, 0x5F, 0x78, 0x65,
0x94, 0x89, 0xAE, 0xB3, 0xE0, 0xFD, 0xDA, 0xC7, 0x7C, 0x61, 0x46, 0x5B, 0x08, 0x15, 0x32, 0x2F,
0x59, 0x44, 0x63, 0x7E, 0x2D, 0x30, 0x17, 0x0A, 0xB1, 0xAC, 0x8B, 0x96, 0xC5, 0xD8, 0xFF, 0xE2,
0x26, 0x3B, 0x1C, 0x01, 0x52, 0x4F, 0x68, 0x75, 0xCE, 0xD3, 0xF4, 0xE9, 0xBA, 0xA7, 0x80, 0x9D,
0xEB, 0xF6, 0xD1, 0xCC, 0x9F, 0x82, 0xA5, 0xB8, 0x03, 0x1E, 0x39, 0x24, 0x77, 0x6A, 0x4D, 0x50,
0xA1, 0xBC, 0x9B, 0x86, 0xD5, 0xC8, 0xEF, 0xF2, 0x49, 0x54, 0x73, 0x6E, 0x3D, 0x20, 0x07, 0x1A,
0x6C, 0x71, 0x56, 0x4B, 0x18, 0x05, 0x22, 0x3F, 0x84, 0x99, 0xBE, 0xA3, 0xF0, 0xED, 0xCA, 0xD7,
0x35, 0x28, 0x0F, 0x12, 0x41, 0x5C, 0x7B, 0x66, 0xDD, 0xC0, 0xE7, 0xFA, 0xA9, 0xB4, 0x93, 0x8E,
0xF8, 0xE5, 0xC2, 0xDF, 0x8C, 0x91, 0xB6, 0xAB, 0x10, 0x0D, 0x2A, 0x37, 0x64, 0x79, 0x5E, 0x43,
0xB2, 0xAF, 0x88, 0x95, 0xC6, 0xDB, 0xFC, 0xE1, 0x5A, 0x47, 0x60, 0x7D, 0x2E, 0x33, 0x14, 0x09,
0x7F, 0x62, 0x45, 0x58, 0x0B, 0x16, 0x31, 0x2C, 0x97, 0x8A, 0xAD, 0xB0, 0xE3, 0xFE, 0xD9, 0xC4
};

const uint8_t crc8_table_wcdma[] =
{
0x00, 0xD0, 0x13, 0xC3, 0x26, 0xF6, 0x35, 0xE5, 0x4C, 0x9C, 0x5F, 0x8F, 0x6A, 0xBA, 0x79, 0xA9,
0x98, 0x48, 0x8B, 0x5B, 0xBE, 0x6E, 0xAD, 0x7D, 0xD4, 0x04, 0xC7, 0x17, 0xF2, 0x22, 0xE1, 0x31,
0x83, 0x53, 0x90, 0x40, 0xA5, 0x75, 0xB6, 0x66, 0xCF, 0x1F, 0xDC, 0x0C, 0xE9, 0x39, 0xFA, 0x2A,
0x1B, 0xCB, 0x08, 0xD8, 0x3D, 0xED, 0x2E, 0xFE, 0x57, 0x87, 0x44, 0x94, 0x71, 0xA1, 0x62, 0xB2,
0xB5, 0x65, 0xA6, 0x76, 0x93, 0x43, 0x80, 0x50, 0xF9, 0x29, 0xEA, 0x3A, 0xDF, 0x0F, 0xCC, 0x1C,
0x2D, 0xFD, 0x3E, 0xEE, 0x0B, 0xDB, 0x18, 0xC8, 0x61, 0xB1, 0x72, 0xA2, 0x47, 0x97, 0x54, 0x84,
0x36, 0xE6, 0x25, 0xF5, 0x10, 0xC0, 0x03, 0xD3, 0x7A, 0xAA, 0x69, 0xB9, 0x5C, 0x8C, 0x4F, 0x9F,
0xAE, 0x7E, 0xBD, 0x6D, 0x88, 0x58, 0x9B, 0x4B, 0xE2, 0x32, 0xF1, 0x21, 0xC4, 0x14, 0xD7, 0x07,
0xD9, 0x09, 0xCA, 0x1A, 0xFF, 0x2F, 0xEC, 0x3C, 0x95, 0x45, 0x86, 0x56, 0xB3, 0x63, 0xA0, 0x70,
0x41, 0x91, 0x52, 0x82, 0x67, 0xB7, 0x74, 0xA4, 0x0D, 0xDD, 0x1E, 0xCE, 0x2B, 0xFB, 0x38, 0xE8,
0x5A, 0x8A, 0x49, 0x99, 0x7C, 0xAC, 0x6F, 0xBF, 0x16, 0xC6, 0x05, 0xD5, 0x30, 0xE0, 0x23, 0xF3,
0xC2, 0x12, 0xD1, 0x01, 0xE4, 0x34, 0xF7, 0x27, 0x8E, 0x5E, 0x9D, 0x4D, 0xA8, 0x78, 0xBB, 0x6B,
0x6C, 0xBC, 0x7F, 0xAF, 0x4A, 0x9A, 0x59, 0x89, 0x20, 0xF0, 0x33, 0xE3, 0x06, 0xD6, 0x15, 0xC5,
0xF4, 0x24, 0xE7, 0x37, 0xD2, 0x02, 0xC1, 0x11, 0xB8, 0x68, 0xAB, 0x7B, 0x9E, 0x4E, 0x8D, 0x5D,
0xEF, 0x3F, 0xFC, 0x2C, 0xC9, 0x19, 0xDA, 0x0A, 0xA3, 0x73, 0xB0, 0x60, 0x85, 0x55, 0x96, 0x46,
0x77, 0xA7, 0x64, 0xB4, 0x51, 0x81, 0x42, 0x92, 0x3B, 0xEB, 0x28, 0xF8, 0x1D, 0xCD, 0x0E, 0xDE
};

const uint8_t crc8_table_rohc[] =
{
0x00, 0x91, 0xE3, 0x72, 0x07, 0x96, 0xE4, 0x75, 0x0E, 0x9F, 0xED, 0x7C, 0x09, 0x98, 0xEA, 0x7B,
0x1C, 0x8D, 0xFF, 0x6E, 0x1B, 0x8A, 0xF8, 0x69, 0x12, 0x83, 0xF1, 0x60, 0x15, 0x84, 0xF6, 0x67,
0x38, 0xA9, 0xDB, 0x4A, 0x3F, 0xAE, 0xDC, 0x4D, 0x36, 0xA7, 0xD5, 0x44, 0x31, 0xA0, 0xD2, 0x43,
0x24, 0xB5, 0xC7, 0x56, 0x23, 0xB2, 0xC0, 0x51, 0x2A, 0xBB, 0xC9, 0x58, 0x2D, 0xBC, 0xCE, 0x5F,
0x70, 0xE1, 0x93, 0x02, 0x77, 0xE6, 0x94, 0x05, 0x7E, 0xEF, 0x9D, 0x0C, 0x79, 0xE8, 0x9A, 0x0B,
0x6C, 0xFD, 0x8F, 0x1E, 0x6B, 0xFA, 0x88, 0x19, 0x62, 0xF3, 0x81, 0x10, 0x65, 0xF4, 0x86, 0x17,
0x48, 0xD9, 0xAB, 0x3A, 0x4F, 0xDE, 0xAC, 0x3D, 0x46, 0xD7, 0xA5, 0x34, 0x41, 0xD0, 0xA2, 0x33,
0x54, 0xC5, 0xB7, 0x26, 0x53, 0xC2, 0xB0, 0x21, 0x5A, 0xCB, 0xB9, 0x28, 0x5D, 0xCC, 0xBE, 0x2F,
0xE0, 0x71, 0x03, 0x92, 0xE7, 0x76, 0x04, 0x95, 0xEE, 0x7F, 0x0D, 0x9C, 0xE9, 0x78, 0x0A, 0x9B,
0xFC, 0x6D, 0x1F, 0x8E, 0xFB, 0x6A, 0x18, 0x89, 0xF2, 0x63, 0x11, 0x80, 0xF5, 0x64, 0x16, 0x87,
0xD8, 0x49, 0x3B, 0xAA, 0xDF, 0x4E, 0x3C, 0xAD, 0xD6, 0x47, 0x35, 0xA4, 0xD1, 0x40, 0x32, 0xA3,
0xC4, 0x55, 0x27, 0xB6, 0xC3, 0x52, 0x20, 0xB1, 0xCA, 0x5B, 0x29, 0xB8, 0xCD, 0x5C, 0x2E, 0xBF,
0x90, 0x01, 0x73, 0xE2, 0x97, 0x06, 0x74, 0xE5, 0x9E, 0x0F, 0x7D, 0xEC, 0x99, 0x08, 0x7A, 0xEB,
0x8C, 0x1D, 0x6F, 0xFE, 0x8B, 0x1A, 0x68, 0xF9, 0x82, 0x13, 0x61, 0xF0, 0x85, 0x14, 0x66, 0xF7,
0xA8, 0x39, 0x4B, 0xDA, 0xAF, 0x3E, 0x4C, 0xDD, 0xA6, 0x37, 0x45, 0xD4, 0xA1, 0x30, 0x42, 0xD3,
0xB4, 0x25, 0x57, 0xC6, 0xB3, 0x22, 0x50, 0xC1, 0xBA, 0x2B, 0x59, 0xC8, 0xBD, 0x2C, 0x5E, 0xCF
};

const uint8_t crc8_table_darc[] =
{
0x00, 0x72, 0xE4, 0x96, 0xF1, 0x83, 0x15, 0x67, 0xDB, 0xA9, 0x3F, 0x4D, 0x2A, 0x58, 0xCE, 0xBC,
0x8F, 0xFD, 0x6B, 0x19, 0x7E, 0x0C, 0x9A, 0xE8, 0x54, 0x26, 0xB0, 0xC2, 0xA5, 0xD7, 0x41, 0x33,
0x27, 0x55, 0xC3, 0xB1, 0xD6, 0xA4, 0x32, 0x40, 0xFC, 0x8E, 0x18, 0x6A, 0x0D, 0x7F, 0xE9, 0x9B,
0xA8, 0xDA, 0x4C, 0x3E, 0x59, 0x2B, 0xBD, 0xCF, 0x73, 0x01, 0x97, 0xE5, 0x82, 0xF0, 0x66, 0x14,
0x4E, 0x3C, 0xAA, 0xD8, 0xBF, 0xCD, 0x5B, 0x29, 0x95, 0xE7, 0x71, 0x03, 0x64, 0x16, 0x80, 0xF2,
0xC1, 0xB3, 0x25, 0x57, 0x30, 0x42, 0xD4, 0xA6, 0x1A, 0x68, 0xFE, 0x8C, 0xEB, 0x99, 0x0F, 0x7D,
0x69, 0x1B, 0x8D, 0xFF, 0x98, 0xEA, 0x7C, 0x0E, 0xB2, 0xC0, 0x56, 0x24, 0x43, 0x31, 0xA7, 0xD5,
0xE6, 0x94, 0x02, 0x70, 0x17, 0x65, 0xF3, 0x81, 0x3D, 0x4F, 0xD9, 0xAB, 0xCC, 0xBE, 0x28, 0x5A,
0x9C, 0xEE, 0x78, 0x0A, 0x6D, 0x1F, 0x89, 0xFB, 0x47, 0x35, 0xA3, 0xD1, 0xB6, 0xC4, 0x52, 0x20,
0x13, 0x61, 0xF7, 0x85, 0xE2, 0x90, 0x06, 0x74, 0xC8, 0xBA, 0x2C, 0x5E, 0x39, 0x4B, 0xDD, 0xAF,
0xBB, 0xC9, 0x5F, 0x2D, 0x4A, 0x38, 0xAE, 0xDC, 0x60, 0x12, 0x84, 0xF6, 0x91, 0xE3, 0x75, 0x07,
0x34, 0x46, 0xD0, 0xA2, 0xC5, 0xB7, 0x21, 0x53, 0xEF, 0x9D, 0x0B, 0x79, 0x1E, 0x6C, 0xFA, 0x88,
0xD2, 0xA0, 0x36, 0x44, 0x23, 0x51, 0xC7, 0xB5, 0x09, 0x7B, 0xED, 0x9F, 0xF8, 0x8A, 0x1C, 0x6E,
0x5D, 0x2F, 0xB9, 0xCB, 0xAC, 0xDE, 0x48, 0x3A, 0x86, 0xF4, 0x62, 0x10, 0x77, 0x05, 0x93, 0xE1,
0xF5, 0x87, 0x11, 0x63, 0x04, 0x76, 0xE0, 0x92, 0x2E, 0x5C, 0xCA, 0xB8, 0xDF, 0xAD, 0x3B, 0x49,
0x7A, 0x08, 0x9E, 0xEC, 0x8B, 0xF9, 0x6F, 0x1D, 0xA1, 0xD3, 0x45, 0x37, 0x50, 0x22, 0xB4, 0xC6
};

const uint16_t crc_tab_1021_normal[] =
{
0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50A5, 0x60C6, 0x70E7, 
0x8108, 0x9129, 0xA14A, 0xB16B, 0xC18C, 0xD1AD, 0xE1CE, 0xF1EF, 
0x1231, 0x0210, 0x3273, 0x2252, 0x52B5, 0x4294, 0x72F7, 0x62D6, 
0x9339, 0x8318, 0xB37B, 0xA35A, 0xD3BD, 0xC39C, 0xF3FF, 0xE3DE, 
0x2462, 0x3443, 0x0420, 0x1401, 0x64E6, 0x74C7, 0x44A4, 0x5485, 
0xA56A, 0xB54B, 0x8528, 0x9509, 0xE5EE, 0xF5CF, 0xC5AC, 0xD58D, 
0x3653, 0x2672, 0x1611, 0x0630, 0x76D7, 0x66F6, 0x5695, 0x46B4, 
0xB75B, 0xA77A, 0x9719, 0x8738, 0xF7DF, 0xE7FE, 0xD79D, 0xC7BC, 
0x48C4, 0x58E5, 0x6886, 0x78A7, 0x0840, 0x1861, 0x2802, 0x3823, 
0xC9CC, 0xD9ED, 0xE98E, 0xF9AF, 0x8948, 0x9969, 0xA90A, 0xB92B, 
0x5AF5, 0x4AD4, 0x7AB7, 0x6A96, 0x1A71, 0x0A50, 0x3A33, 0x2A12, 
0xDBFD, 0xCBDC, 0xFBBF, 0xEB9E, 0x9B79, 0x8B58, 0xBB3B, 0xAB1A, 
0x6CA6, 0x7C87, 0x4CE4, 0x5CC5, 0x2C22, 0x3C03, 0x0C60, 0x1C41, 
0xEDAE, 0xFD8F, 0xCDEC, 0xDDCD, 0xAD2A, 0xBD0B, 0x8D68, 0x9D49, 
0x7E97, 0x6EB6, 0x5ED5, 0x4EF4, 0x3E13, 0x2E32, 0x1E51, 0x0E70, 
0xFF9F, 0xEFBE, 0xDFDD, 0xCFFC, 0xBF1B, 0xAF3A, 0x9F59, 0x8F78, 
                                                                
0x9188, 0x81A9, 0xB1CA, 0xA1EB, 0xD10C, 0xC12D, 0xF14E, 0xE16F, 
0x1080, 0x00A1, 0x30C2, 0x20E3, 0x5004, 0x4025, 0x7046, 0x6067, 
0x83B9, 0x9398, 0xA3FB, 0xB3DA, 0xC33D, 0xD31C, 0xE37F, 0xF35E, 
0x02B1, 0x1290, 0x22F3, 0x32D2, 0x4235, 0x5214, 0x6277, 0x7256, 
0xB5EA, 0xA5CB, 0x95A8, 0x8589, 0xF56E, 0xE54F, 0xD52C, 0xC50D, 
0x34E2, 0x24C3, 0x14A0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405, 
0xA7DB, 0xB7FA, 0x8799, 0x97B8, 0xE75F, 0xF77E, 0xC71D, 0xD73C, 
0x26D3, 0x36F2, 0x0691, 0x16B0, 0x6657, 0x7676, 0x4615, 0x5634, 
0xD94C, 0xC96D, 0xF90E, 0xE92F, 0x99C8, 0x89E9, 0xB98A, 0xA9AB, 
0x5844, 0x4865, 0x7806, 0x6827, 0x18C0, 0x08E1, 0x3882, 0x28A3, 
0xCB7D, 0xDB5C, 0xEB3F, 0xFB1E, 0x8BF9, 0x9BD8, 0xABBB, 0xBB9A, 
0x4A75, 0x5A54, 0x6A37, 0x7A16, 0x0AF1, 0x1AD0, 0x2AB3, 0x3A92, 
0xFD2E, 0xED0F, 0xDD6C, 0xCD4D, 0xBDAA, 0xAD8B, 0x9DE8, 0x8DC9, 
0x7C26, 0x6C07, 0x5C64, 0x4C45, 0x3CA2, 0x2C83, 0x1CE0, 0x0CC1, 
0xEF1F, 0xFF3E, 0xCF5D, 0xDF7C, 0xAF9B, 0xBFBA, 0x8FD9, 0x9FF8, 
0x6E17, 0x7E36, 0x4E55, 0x5E74, 0x2E93, 0x3EB2, 0x0ED1, 0x1EF0  
};

const uint16_t crc_tab_1021_reflected[] =
{
0x0000, 0x1189, 0x2312, 0x329B, 0x4624, 0x57AD, 0x6536, 0x74BF,
0x8C48, 0x9DC1, 0xAF5A, 0xBED3, 0xCA6C, 0xDBE5, 0xE97E, 0xF8F7,
0x1081, 0x0108, 0x3393, 0x221A, 0x56A5, 0x472C, 0x75B7, 0x643E,
0x9CC9, 0x8D40, 0xBFDB, 0xAE52, 0xDAED, 0xCB64, 0xF9FF, 0xE876,
0x2102, 0x308B, 0x0210, 0x1399, 0x6726, 0x76AF, 0x4434, 0x55BD,
0xAD4A, 0xBCC3, 0x8E58, 0x9FD1, 0xEB6E, 0xFAE7, 0xC87C, 0xD9F5,
0x3183, 0x200A, 0x1291, 0x0318, 0x77A7, 0x662E, 0x54B5, 0x453C,
0xBDCB, 0xAC42, 0x9ED9, 0x8F50, 0xFBEF, 0xEA66, 0xD8FD, 0xC974,
0x4204, 0x538D, 0x6116, 0x709F, 0x0420, 0x15A9, 0x2732, 0x36BB,
0xCE4C, 0xDFC5, 0xED5E, 0xFCD7, 0x8868, 0x99E1, 0xAB7A, 0xBAF3,
0x5285, 0x430C, 0x7197, 0x601E, 0x14A1, 0x0528, 0x37B3, 0x263A,
0xDECD, 0xCF44, 0xFDDF, 0xEC56, 0x98E9, 0x8960, 0xBBFB, 0xAA72,
0x6306, 0x728F, 0x4014, 0x519D, 0x2522, 0x34AB, 0x0630, 0x17B9,
0xEF4E, 0xFEC7, 0xCC5C, 0xDDD5, 0xA96A, 0xB8E3, 0x8A78, 0x9BF1,
0x7387, 0x620E, 0x5095, 0x411C, 0x35A3, 0x242A, 0x16B1, 0x0738,
0xFFCF, 0xEE46, 0xDCDD, 0xCD54, 0xB9EB, 0xA862, 0x9AF9, 0x8B70,

0x8408, 0x9581, 0xA71A, 0xB693, 0xC22C, 0xD3A5, 0xE13E, 0xF0B7,
0x0840, 0x19C9, 0x2B52, 0x3ADB, 0x4E64, 0x5FED, 0x6D76, 0x7CFF,
0x9489, 0x8500, 0xB79B, 0xA612, 0xD2AD, 0xC324, 0xF1BF, 0xE036,
0x18C1, 0x0948, 0x3BD3, 0x2A5A, 0x5EE5, 0x4F6C, 0x7DF7, 0x6C7E,
0xA50A, 0xB483, 0x8618, 0x9791, 0xE32E, 0xF2A7, 0xC03C, 0xD1B5,
0x2942, 0x38CB, 0x0A50, 0x1BD9, 0x6F66, 0x7EEF, 0x4C74, 0x5DFD,
0xB58B, 0xA402, 0x9699, 0x8710, 0xF3AF, 0xE226, 0xD0BD, 0xC134,
0x39C3, 0x284A, 0x1AD1, 0x0B58, 0x7FE7, 0x6E6E, 0x5CF5, 0x4D7C,
0xC60C, 0xD785, 0xE51E, 0xF497, 0x8028, 0x91A1, 0xA33A, 0xB2B3,
0x4A44, 0x5BCD, 0x6956, 0x78DF, 0x0C60, 0x1DE9, 0x2F72, 0x3EFB,
0xD68D, 0xC704, 0xF59F, 0xE416, 0x90A9, 0x8120, 0xB3BB, 0xA232,
0x5AC5, 0x4B4C, 0x79D7, 0x685E, 0x1CE1, 0x0D68, 0x3FF3, 0x2E7A,
0xE70E, 0xF687, 0xC41C, 0xD595, 0xA12A, 0xB0A3, 0x8238, 0x93B1,
0x6B46, 0x7ACF, 0x4854, 0x59DD, 0x2D62, 0x3CEB, 0x0E70, 0x1FF9,
0xF78F, 0xE606, 0xD49D, 0xC514, 0xB1AB, 0xA022, 0x92B9, 0x8330,
0x7BC7, 0x6A4E, 0x58D5, 0x495C, 0x3DE3, 0x2C6A, 0x1EF1, 0x0F78
};

const uint16_t crc_tab_8005_normal[] =
{
0x0000, 0x8005, 0x800F, 0x000A, 0x801B, 0x001E, 0x0014, 0x8011,
0x8033, 0x0036, 0x003C, 0x8039, 0x0028, 0x802D, 0x8027, 0x0022,
0x8063, 0x0066, 0x006C, 0x8069, 0x0078, 0x807D, 0x8077, 0x0072,
0x0050, 0x8055, 0x805F, 0x005A, 0x804B, 0x004E, 0x0044, 0x8041,
0x80C3, 0x00C6, 0x00CC, 0x80C9, 0x00D8, 0x80DD, 0x80D7, 0x00D2,
0x00F0, 0x80F5, 0x80FF, 0x00FA, 0x80EB, 0x00EE, 0x00E4, 0x80E1,
0x00A0, 0x80A5, 0x80AF, 0x00AA, 0x80BB, 0x00BE, 0x00B4, 0x80B1,
0x8093, 0x0096, 0x009C, 0x8099, 0x0088, 0x808D, 0x8087, 0x0082,
0x8183, 0x0186, 0x018C, 0x8189, 0x0198, 0x819D, 0x8197, 0x0192,
0x01B0, 0x81B5, 0x81BF, 0x01BA, 0x81AB, 0x01AE, 0x01A4, 0x81A1,
0x01E0, 0x81E5, 0x81EF, 0x01EA, 0x81FB, 0x01FE, 0x01F4, 0x81F1,
0x81D3, 0x01D6, 0x01DC, 0x81D9, 0x01C8, 0x81CD, 0x81C7, 0x01C2,
0x0140, 0x8145, 0x814F, 0x014A, 0x815B, 0x015E, 0x0154, 0x8151,
0x8173, 0x0176, 0x017C, 0x8179, 0x0168, 0x816D, 0x8167, 0x0162,
0x8123, 0x0126, 0x012C, 0x8129, 0x0138, 0x813D, 0x8137, 0x0132,
0x0110, 0x8115, 0x811F, 0x011A, 0x810B, 0x010E, 0x0104, 0x8101,

0x8303, 0x0306, 0x030C, 0x8309, 0x0318, 0x831D, 0x8317, 0x0312,
0x0330, 0x8335, 0x833F, 0x033A, 0x832B, 0x032E, 0x0324, 0x8321,
0x0360, 0x8365, 0x836F, 0x036A, 0x837B, 0x037E, 0x0374, 0x8371,
0x8353, 0x0356, 0x035C, 0x8359, 0x0348, 0x834D, 0x8347, 0x0342,
0x03C0, 0x83C5, 0x83CF, 0x03CA, 0x83DB, 0x03DE, 0x03D4, 0x83D1,
0x83F3, 0x03F6, 0x03FC, 0x83F9, 0x03E8, 0x83ED, 0x83E7, 0x03E2,
0x83A3, 0x03A6, 0x03AC, 0x83A9, 0x03B8, 0x83BD, 0x83B7, 0x03B2,
0x0390, 0x8395, 0x839F, 0x039A, 0x838B, 0x038E, 0x0384, 0x8381,
0x0280, 0x8285, 0x828F, 0x028A, 0x829B, 0x029E, 0x0294, 0x8291,
0x82B3, 0x02B6, 0x02BC, 0x82B9, 0x02A8, 0x82AD, 0x82A7, 0x02A2,
0x82E3, 0x02E6, 0x02EC, 0x82E9, 0x02F8, 0x82FD, 0x82F7, 0x02F2,
0x02D0, 0x82D5, 0x82DF, 0x02DA, 0x82CB, 0x02CE, 0x02C4, 0x82C1,
0x8243, 0x0246, 0x024C, 0x8249, 0x0258, 0x825D, 0x8257, 0x0252,
0x0270, 0x8275, 0x827F, 0x027A, 0x826B, 0x026E, 0x0264, 0x8261,
0x0220, 0x8225, 0x822F, 0x022A, 0x823B, 0x023E, 0x0234, 0x8231,
0x8213, 0x0216, 0x021C, 0x8219, 0x0208, 0x820D, 0x8207, 0x0202
};

const uint16_t crc_tab_8005_reflected[] =
{
0x0000, 0xC0C1, 0xC181, 0x0140, 0xC301, 0x03C0, 0x0280, 0xC241,
0xC601, 0x06C0, 0x0780, 0xC741, 0x0500, 0xC5C1, 0xC481, 0x0440,
0xCC01, 0x0CC0, 0x0D80, 0xCD41, 0x0F00, 0xCFC1, 0xCE81, 0x0E40,
0x0A00, 0xCAC1, 0xCB81, 0x0B40, 0xC901, 0x09C0, 0x0880, 0xC841,
0xD801, 0x18C0, 0x1980, 0xD941, 0x1B00, 0xDBC1, 0xDA81, 0x1A40,
0x1E00, 0xDEC1, 0xDF81, 0x1F40, 0xDD01, 0x1DC0, 0x1C80, 0xDC41,
0x1400, 0xD4C1, 0xD581, 0x1540, 0xD701, 0x17C0, 0x1680, 0xD641,
0xD201, 0x12C0, 0x1380, 0xD341, 0x1100, 0xD1C1, 0xD081, 0x1040,
0xF001, 0x30C0, 0x3180, 0xF141, 0x3300, 0xF3C1, 0xF281, 0x3240,
0x3600, 0xF6C1, 0xF781, 0x3740, 0xF501, 0x35C0, 0x3480, 0xF441,
0x3C00, 0xFCC1, 0xFD81, 0x3D40, 0xFF01, 0x3FC0, 0x3E80, 0xFE41,
0xFA01, 0x3AC0, 0x3B80, 0xFB41, 0x3900, 0xF9C1, 0xF881, 0x3840,
0x2800, 0xE8C1, 0xE981, 0x2940, 0xEB01, 0x2BC0, 0x2A80, 0xEA41,
0xEE01, 0x2EC0, 0x2F80, 0xEF41, 0x2D00, 0xEDC1, 0xEC81, 0x2C40,
0xE401, 0x24C0, 0x2580, 0xE541, 0x2700, 0xE7C1, 0xE681, 0x2640,
0x2200, 0xE2C1, 0xE381, 0x2340, 0xE101, 0x21C0, 0x2080, 0xE041,

0xA001, 0x60C0, 0x6180, 0xA141, 0x6300, 0xA3C1, 0xA281, 0x6240,
0x6600, 0xA6C1, 0xA781, 0x6740, 0xA501, 0x65C0, 0x6480, 0xA441,
0x6C00, 0xACC1, 0xAD81, 0x6D40, 0xAF01, 0x6FC0, 0x6E80, 0xAE41,
0xAA01, 0x6AC0, 0x6B80, 0xAB41, 0x6900, 0xA9C1, 0xA881, 0x6840,
0x7800, 0xB8C1, 0xB981, 0x7940, 0xBB01, 0x7BC0, 0x7A80, 0xBA41,
0xBE01, 0x7EC0, 0x7F80, 0xBF41, 0x7D00, 0xBDC1, 0xBC81, 0x7C40,
0xB401, 0x74C0, 0x7580, 0xB541, 0x7700, 0xB7C1, 0xB681, 0x7640,
0x7200, 0xB2C1, 0xB381, 0x7340, 0xB101, 0x71C0, 0x7080, 0xB041,
0x5000, 0x90C1, 0x9181, 0x5140, 0x9301, 0x53C0, 0x5280, 0x9241,
0x9601, 0x56C0, 0x5780, 0x9741, 0x5500, 0x95C1, 0x9481, 0x5440,
0x9C01, 0x5CC0, 0x5D80, 0x9D41, 0x5F00, 0x9FC1, 0x9E81, 0x5E40,
0x5A00, 0x9AC1, 0x9B81, 0x5B40, 0x9901, 0x59C0, 0x5880, 0x9841,
0x8801, 0x48C0, 0x4980, 0x8941, 0x4B00, 0x8BC1, 0x8A81, 0x4A40,
0x4E00, 0x8EC1, 0x8F81, 0x4F40, 0x8D01, 0x4DC0, 0x4C80, 0x8C41,
0x4400, 0x84C1, 0x8581, 0x4540, 0x8701, 0x47C0, 0x4680, 0x8641,
0x8201, 0x42C0, 0x4380, 0x8341, 0x4100, 0x81C1, 0x8081, 0x4040
};

const uint16_t crc_tab_3d65_normal[] =
{
0x0000, 0x3D65, 0x7ACA, 0x47AF, 0xF594, 0xC8F1, 0x8F5E, 0xB23B,
0xD64D, 0xEB28, 0xAC87, 0x91E2, 0x23D9, 0x1EBC, 0x5913, 0x6476,
0x91FF, 0xAC9A, 0xEB35, 0xD650, 0x646B, 0x590E, 0x1EA1, 0x23C4,
0x47B2, 0x7AD7, 0x3D78, 0x001D, 0xB226, 0x8F43, 0xC8EC, 0xF589,
0x1E9B, 0x23FE, 0x6451, 0x5934, 0xEB0F, 0xD66A, 0x91C5, 0xACA0,
0xC8D6, 0xF5B3, 0xB21C, 0x8F79, 0x3D42, 0x0027, 0x4788, 0x7AED,
0x8F64, 0xB201, 0xF5AE, 0xC8CB, 0x7AF0, 0x4795, 0x003A, 0x3D5F,
0x5929, 0x644C, 0x23E3, 0x1E86, 0xACBD, 0x91D8, 0xD677, 0xEB12,
0x3D36, 0x0053, 0x47FC, 0x7A99, 0xC8A2, 0xF5C7, 0xB268, 0x8F0D,
0xEB7B, 0xD61E, 0x91B1, 0xACD4, 0x1EEF, 0x238A, 0x6425, 0x5940,
0xACC9, 0x91AC, 0xD603, 0xEB66, 0x595D, 0x6438, 0x2397, 0x1EF2,
0x7A84, 0x47E1, 0x004E, 0x3D2B, 0x8F10, 0xB275, 0xF5DA, 0xC8BF,
0x23AD, 0x1EC8, 0x5967, 0x6402, 0xD639, 0xEB5C, 0xACF3, 0x9196,
0xF5E0, 0xC885, 0x8F2A, 0xB24F, 0x0074, 0x3D11, 0x7ABE, 0x47DB,
0xB252, 0x8F37, 0xC898, 0xF5FD, 0x47C6, 0x7AA3, 0x3D0C, 0x0069,
0x641F, 0x597A, 0x1ED5, 0x23B0, 0x918B, 0xACEE, 0xEB41, 0xD624,

0x7A6C, 0x4709, 0x00A6, 0x3DC3, 0x8FF8, 0xB29D, 0xF532, 0xC857,
0xAC21, 0x9144, 0xD6EB, 0xEB8E, 0x59B5, 0x64D0, 0x237F, 0x1E1A,
0xEB93, 0xD6F6, 0x9159, 0xAC3C, 0x1E07, 0x2362, 0x64CD, 0x59A8,
0x3DDE, 0x00BB, 0x4714, 0x7A71, 0xC84A, 0xF52F, 0xB280, 0x8FE5,
0x64F7, 0x5992, 0x1E3D, 0x2358, 0x9163, 0xAC06, 0xEBA9, 0xD6CC,
0xB2BA, 0x8FDF, 0xC870, 0xF515, 0x472E, 0x7A4B, 0x3DE4, 0x0081,
0xF508, 0xC86D, 0x8FC2, 0xB2A7, 0x009C, 0x3DF9, 0x7A56, 0x4733,
0x2345, 0x1E20, 0x598F, 0x64EA, 0xD6D1, 0xEBB4, 0xAC1B, 0x917E,
0x475A, 0x7A3F, 0x3D90, 0x00F5, 0xB2CE, 0x8FAB, 0xC804, 0xF561,
0x9117, 0xAC72, 0xEBDD, 0xD6B8, 0x6483, 0x59E6, 0x1E49, 0x232C,
0xD6A5, 0xEBC0, 0xAC6F, 0x910A, 0x2331, 0x1E54, 0x59FB, 0x649E,
0x00E8, 0x3D8D, 0x7A22, 0x4747, 0xF57C, 0xC819, 0x8FB6, 0xB2D3,
0x59C1, 0x64A4, 0x230B, 0x1E6E, 0xAC55, 0x9130, 0xD69F, 0xEBFA,
0x8F8C, 0xB2E9, 0xF546, 0xC823, 0x7A18, 0x477D, 0x00D2, 0x3DB7,
0xC83E, 0xF55B, 0xB2F4, 0x8F91, 0x3DAA, 0x00CF, 0x4760, 0x7A05,
0x1E73, 0x2316, 0x64B9, 0x59DC, 0xEBE7, 0xD682, 0x912D, 0xAC48
};

const uint16_t crc_tab_3d65_reflected[] =
{
0x0000, 0x365E, 0x6CBC, 0x5AE2, 0xD978, 0xEF26, 0xB5C4, 0x839A,
0xFF89, 0xC9D7, 0x9335, 0xA56B, 0x26F1, 0x10AF, 0x4A4D, 0x7C13,
0xB26B, 0x8435, 0xDED7, 0xE889, 0x6B13, 0x5D4D, 0x07AF, 0x31F1,
0x4DE2, 0x7BBC, 0x215E, 0x1700, 0x949A, 0xA2C4, 0xF826, 0xCE78,
0x29AF, 0x1FF1, 0x4513, 0x734D, 0xF0D7, 0xC689, 0x9C6B, 0xAA35,
0xD626, 0xE078, 0xBA9A, 0x8CC4, 0x0F5E, 0x3900, 0x63E2, 0x55BC,
0x9BC4, 0xAD9A, 0xF778, 0xC126, 0x42BC, 0x74E2, 0x2E00, 0x185E,
0x644D, 0x5213, 0x08F1, 0x3EAF, 0xBD35, 0x8B6B, 0xD189, 0xE7D7,
0x535E, 0x6500, 0x3FE2, 0x09BC, 0x8A26, 0xBC78, 0xE69A, 0xD0C4,
0xACD7, 0x9A89, 0xC06B, 0xF635, 0x75AF, 0x43F1, 0x1913, 0x2F4D,
0xE135, 0xD76B, 0x8D89, 0xBBD7, 0x384D, 0x0E13, 0x54F1, 0x62AF,
0x1EBC, 0x28E2, 0x7200, 0x445E, 0xC7C4, 0xF19A, 0xAB78, 0x9D26,
0x7AF1, 0x4CAF, 0x164D, 0x2013, 0xA389, 0x95D7, 0xCF35, 0xF96B,
0x8578, 0xB326, 0xE9C4, 0xDF9A, 0x5C00, 0x6A5E, 0x30BC, 0x06E2,
0xC89A, 0xFEC4, 0xA426, 0x9278, 0x11E2, 0x27BC, 0x7D5E, 0x4B00,
0x3713, 0x014D, 0x5BAF, 0x6DF1, 0xEE6B, 0xD835, 0x82D7, 0xB489,

0xA6BC, 0x90E2, 0xCA00, 0xFC5E, 0x7FC4, 0x499A, 0x1378, 0x2526,
0x5935, 0x6F6B, 0x3589, 0x03D7, 0x804D, 0xB613, 0xECF1, 0xDAAF,
0x14D7, 0x2289, 0x786B, 0x4E35, 0xCDAF, 0xFBF1, 0xA113, 0x974D,
0xEB5E, 0xDD00, 0x87E2, 0xB1BC, 0x3226, 0x0478, 0x5E9A, 0x68C4,
0x8F13, 0xB94D, 0xE3AF, 0xD5F1, 0x566B, 0x6035, 0x3AD7, 0x0C89,
0x709A, 0x46C4, 0x1C26, 0x2A78, 0xA9E2, 0x9FBC, 0xC55E, 0xF300,
0x3D78, 0x0B26, 0x51C4, 0x679A, 0xE400, 0xD25E, 0x88BC, 0xBEE2,
0xC2F1, 0xF4AF, 0xAE4D, 0x9813, 0x1B89, 0x2DD7, 0x7735, 0x416B,
0xF5E2, 0xC3BC, 0x995E, 0xAF00, 0x2C9A, 0x1AC4, 0x4026, 0x7678,
0x0A6B, 0x3C35, 0x66D7, 0x5089, 0xD313, 0xE54D, 0xBFAF, 0x89F1,
0x4789, 0x71D7, 0x2B35, 0x1D6B, 0x9EF1, 0xA8AF, 0xF24D, 0xC413,
0xB800, 0x8E5E, 0xD4BC, 0xE2E2, 0x6178, 0x5726, 0x0DC4, 0x3B9A,
0xDC4D, 0xEA13, 0xB0F1, 0x86AF, 0x0535, 0x336B, 0x6989, 0x5FD7,
0x23C4, 0x159A, 0x4F78, 0x7926, 0xFABC, 0xCCE2, 0x9600, 0xA05E,
0x6E26, 0x5878, 0x029A, 0x34C4, 0xB75E, 0x8100, 0xDBE2, 0xEDBC,
0x91AF, 0xA7F1, 0xFD13, 0xCB4D, 0x48D7, 0x7E89, 0x246B, 0x1235
};

const uint16_t crc_tabt10dif[] =
{
0x0000, 0x8BB7, 0x9CD9, 0x176E, 0xB205, 0x39B2, 0x2EDC, 0xA56B,
0xEFBD, 0x640A, 0x7364, 0xF8D3, 0x5DB8, 0xD60F, 0xC161, 0x4AD6,
0x54CD, 0xDF7A, 0xC814, 0x43A3, 0xE6C8, 0x6D7F, 0x7A11, 0xF1A6,
0xBB70, 0x30C7, 0x27A9, 0xAC1E, 0x0975, 0x82C2, 0x95AC, 0x1E1B,
0xA99A, 0x222D, 0x3543, 0xBEF4, 0x1B9F, 0x9028, 0x8746, 0x0CF1,
0x4627, 0xCD90, 0xDAFE, 0x5149, 0xF422, 0x7F95, 0x68FB, 0xE34C,
0xFD57, 0x76E0, 0x618E, 0xEA39, 0x4F52, 0xC4E5, 0xD38B, 0x583C,
0x12EA, 0x995D, 0x8E33, 0x0584, 0xA0EF, 0x2B58, 0x3C36, 0xB781,
0xD883, 0x5334, 0x445A, 0xCFED, 0x6A86, 0xE131, 0xF65F, 0x7DE8,
0x373E, 0xBC89, 0xABE7, 0x2050, 0x853B, 0x0E8C, 0x19E2, 0x9255,
0x8C4E, 0x07F9, 0x1097, 0x9B20, 0x3E4B, 0xB5FC, 0xA292, 0x2925,
0x63F3, 0xE844, 0xFF2A, 0x749D, 0xD1F6, 0x5A41, 0x4D2F, 0xC698,
0x7119, 0xFAAE, 0xEDC0, 0x6677, 0xC31C, 0x48AB, 0x5FC5, 0xD472,
0x9EA4, 0x1513, 0x027D, 0x89CA, 0x2CA1, 0xA716, 0xB078, 0x3BCF,
0x25D4, 0xAE63, 0xB90D, 0x32BA, 0x97D1, 0x1C66, 0x0B08, 0x80BF,
0xCA69, 0x41DE, 0x56B0, 0xDD07, 0x786C, 0xF3DB, 0xE4B5, 0x6F02,

0x3AB1, 0xB106, 0xA668, 0x2DDF, 0x88B4, 0x0303, 0x146D, 0x9FDA,
0xD50C, 0x5EBB, 0x49D5, 0xC262, 0x6709, 0xECBE, 0xFBD0, 0x7067,
0x6E7C, 0xE5CB, 0xF2A5, 0x7912, 0xDC79, 0x57CE, 0x40A0, 0xCB17,
0x81C1, 0x0A76, 0x1D18, 0x96AF, 0x33C4, 0xB873, 0xAF1D, 0x24AA,
0x932B, 0x189C, 0x0FF2, 0x8445, 0x212E, 0xAA99, 0xBDF7, 0x3640,
0x7C96, 0xF721, 0xE04F, 0x6BF8, 0xCE93, 0x4524, 0x524A, 0xD9FD,
0xC7E6, 0x4C51, 0x5B3F, 0xD088, 0x75E3, 0xFE54, 0xE93A, 0x628D,
0x285B, 0xA3EC, 0xB482, 0x3F35, 0x9A5E, 0x11E9, 0x0687, 0x8D30,
0xE232, 0x6985, 0x7EEB, 0xF55C, 0x5037, 0xDB80, 0xCCEE, 0x4759,
0x0D8F, 0x8638, 0x9156, 0x1AE1, 0xBF8A, 0x343D, 0x2353, 0xA8E4,
0xB6FF, 0x3D48, 0x2A26, 0xA191, 0x04FA, 0x8F4D, 0x9823, 0x1394,
0x5942, 0xD2F5, 0xC59B, 0x4E2C, 0xEB47, 0x60F0, 0x779E, 0xFC29,
0x4BA8, 0xC01F, 0xD771, 0x5CC6, 0xF9AD, 0x721A, 0x6574, 0xEEC3,
0xA415, 0x2FA2, 0x38CC, 0xB37B, 0x1610, 0x9DA7, 0x8AC9, 0x017E,
0x1F65, 0x94D2, 0x83BC, 0x080B, 0xAD60, 0x26D7, 0x31B9, 0xBA0E,
0xF0D8, 0x7B6F, 0x6C01, 0xE7B6, 0x42DD, 0xC96A, 0xDE04, 0x55B3
};

const uint16_t crc_tabdect[] =
{
0x0000, 0x0589, 0x0B12, 0x0E9B, 0x1624, 0x13AD, 0x1D36, 0x18BF,
0x2C48, 0x29C1, 0x275A, 0x22D3, 0x3A6C, 0x3FE5, 0x317E, 0x34F7,
0x5890, 0x5D19, 0x5382, 0x560B, 0x4EB4, 0x4B3D, 0x45A6, 0x402F,
0x74D8, 0x7151, 0x7FCA, 0x7A43, 0x62FC, 0x6775, 0x69EE, 0x6C67,
0xB120, 0xB4A9, 0xBA32, 0xBFBB, 0xA704, 0xA28D, 0xAC16, 0xA99F,
0x9D68, 0x98E1, 0x967A, 0x93F3, 0x8B4C, 0x8EC5, 0x805E, 0x85D7,
0xE9B0, 0xEC39, 0xE2A2, 0xE72B, 0xFF94, 0xFA1D, 0xF486, 0xF10F,
0xC5F8, 0xC071, 0xCEEA, 0xCB63, 0xD3DC, 0xD655, 0xD8CE, 0xDD47,
0x67C9, 0x6240, 0x6CDB, 0x6952, 0x71ED, 0x7464, 0x7AFF, 0x7F76,
0x4B81, 0x4E08, 0x4093, 0x451A, 0x5DA5, 0x582C, 0x56B7, 0x533E,
0x3F59, 0x3AD0, 0x344B, 0x31C2, 0x297D, 0x2CF4, 0x226F, 0x27E6,
0x1311, 0x1698, 0x1803, 0x1D8A, 0x0535, 0x00BC, 0x0E27, 0x0BAE,
0xD6E9, 0xD360, 0xDDFB, 0xD872, 0xC0CD, 0xC544, 0xCBDF, 0xCE56,
0xFAA1, 0xFF28, 0xF1B3, 0xF43A, 0xEC85, 0xE90C, 0xE797, 0xE21E,
0x8E79, 0x8BF0, 0x856B, 0x80E2, 0x985D, 0x9DD4, 0x934F, 0x96C6,
0xA231, 0xA7B8, 0xA923, 0xACAA, 0xB415, 0xB19C, 0xBF07, 0xBA8E,

0xCF92, 0xCA1B, 0xC480, 0xC109, 0xD9B6, 0xDC3F, 0xD2A4, 0xD72D,
0xE3DA, 0xE653, 0xE8C8, 0xED41, 0xF5FE, 0xF077, 0xFEEC, 0xFB65,
0x9702, 0x928B, 0x9C10, 0x9999, 0x8126, 0x84AF, 0x8A34, 0x8FBD,
0xBB4A, 0xBEC3, 0xB058, 0xB5D1, 0xAD6E, 0xA8E7, 0xA67C, 0xA3F5,
0x7EB2, 0x7B3B, 0x75A0, 0x7029, 0x6896, 0x6D1F, 0x6384, 0x660D,
0x52FA, 0x5773, 0x59E8, 0x5C61, 0x44DE, 0x4157, 0x4FCC, 0x4A45,
0x2622, 0x23AB, 0x2D30, 0x28B9, 0x3006, 0x358F, 0x3B14, 0x3E9D,
0x0A6A, 0x0FE3, 0x0178, 0x04F1, 0x1C4E, 0x19C7, 0x175C, 0x12D5,
0xA85B, 0xADD2, 0xA349, 0xA6C0, 0xBE7F, 0xBBF6, 0xB56D, 0xB0E4,
0x8413, 0x819A, 0x8F01, 0x8A88, 0x9237, 0x97BE, 0x9925, 0x9CAC,
0xF0CB, 0xF542, 0xFBD9, 0xFE50, 0xE6EF, 0xE366, 0xEDFD, 0xE874,
0xDC83, 0xD90A, 0xD791, 0xD218, 0xCAA7, 0xCF2E, 0xC1B5, 0xC43C,
0x197B, 0x1CF2, 0x1269, 0x17E0, 0x0F5F, 0x0AD6, 0x044D, 0x01C4,
0x3533, 0x30BA, 0x3E21, 0x3BA8, 0x2317, 0x269E, 0x2805, 0x2D8C,
0x41EB, 0x4462, 0x4AF9, 0x4F70, 0x57CF, 0x5246, 0x5CDD, 0x5954,
0x6DA3, 0x682A, 0x66B1, 0x6338, 0x7B87, 0x7E0E, 0x7095, 0x751C
};

const uint16_t crc_tabteledisk[] =
{
0x0000, 0xA097, 0xE1B9, 0x412E, 0x63E5, 0xC372, 0x825C, 0x22CB,
0xC7CA, 0x675D, 0x2673, 0x86E4, 0xA42F, 0x04B8, 0x4596, 0xE501,
0x2F03, 0x8F94, 0xCEBA, 0x6E2D, 0x4CE6, 0xEC71, 0xAD5F, 0x0DC8,
0xE8C9, 0x485E, 0x0970, 0xA9E7, 0x8B2C, 0x2BBB, 0x6A95, 0xCA02,
0x5E06, 0xFE91, 0xBFBF, 0x1F28, 0x3DE3, 0x9D74, 0xDC5A, 0x7CCD,
0x99CC, 0x395B, 0x7875, 0xD8E2, 0xFA29, 0x5ABE, 0x1B90, 0xBB07,
0x7105, 0xD192, 0x90BC, 0x302B, 0x12E0, 0xB277, 0xF359, 0x53CE,
0xB6CF, 0x1658, 0x5776, 0xF7E1, 0xD52A, 0x75BD, 0x3493, 0x9404,
0xBC0C, 0x1C9B, 0x5DB5, 0xFD22, 0xDFE9, 0x7F7E, 0x3E50, 0x9EC7,
0x7BC6, 0xDB51, 0x9A7F, 0x3AE8, 0x1823, 0xB8B4, 0xF99A, 0x590D,
0x930F, 0x3398, 0x72B6, 0xD221, 0xF0EA, 0x507D, 0x1153, 0xB1C4,
0x54C5, 0xF452, 0xB57C, 0x15EB, 0x3720, 0x97B7, 0xD699, 0x760E,
0xE20A, 0x429D, 0x03B3, 0xA324, 0x81EF, 0x2178, 0x6056, 0xC0C1,
0x25C0, 0x8557, 0xC479, 0x64EE, 0x4625, 0xE6B2, 0xA79C, 0x070B,
0xCD09, 0x6D9E, 0x2CB0, 0x8C27, 0xAEEC, 0x0E7B, 0x4F55, 0xEFC2,
0x0AC3, 0xAA54, 0xEB7A, 0x4BED, 0x6926, 0xC9B1, 0x889F, 0x2808,

0xD88F, 0x7818, 0x3936, 0x99A1, 0xBB6A, 0x1BFD, 0x5AD3, 0xFA44,
0x1F45, 0xBFD2, 0xFEFC, 0x5E6B, 0x7CA0, 0xDC37, 0x9D19, 0x3D8E,
0xF78C, 0x571B, 0x1635, 0xB6A2, 0x9469, 0x34FE, 0x75D0, 0xD547,
0x3046, 0x90D1, 0xD1FF, 0x7168, 0x53A3, 0xF334, 0xB21A, 0x128D,
0x8689, 0x261E, 0x6730, 0xC7A7, 0xE56C, 0x45FB, 0x04D5, 0xA442,
0x4143, 0xE1D4, 0xA0FA, 0x006D, 0x22A6, 0x8231, 0xC31F, 0x6388,
0xA98A, 0x091D, 0x4833, 0xE8A4, 0xCA6F, 0x6AF8, 0x2BD6, 0x8B41,
0x6E40, 0xCED7, 0x8FF9, 0x2F6E, 0x0DA5, 0xAD32, 0xEC1C, 0x4C8B,
0x6483, 0xC414, 0x853A, 0x25AD, 0x0766, 0xA7F1, 0xE6DF, 0x4648,
0xA349, 0x03DE, 0x42F0, 0xE267, 0xC0AC, 0x603B, 0x2115, 0x8182,
0x4B80, 0xEB17, 0xAA39, 0x0AAE, 0x2865, 0x88F2, 0xC9DC, 0x694B,
0x8C4A, 0x2CDD, 0x6DF3, 0xCD64, 0xEFAF, 0x4F38, 0x0E16, 0xAE81,
0x3A85, 0x9A12, 0xDB3C, 0x7BAB, 0x5960, 0xF9F7, 0xB8D9, 0x184E,
0xFD4F, 0x5DD8, 0x1CF6, 0xBC61, 0x9EAA, 0x3E3D, 0x7F13, 0xDF84,
0x1586, 0xB511, 0xF43F, 0x54A8, 0x7663, 0xD6F4, 0x97DA, 0x374D,
0xD24C, 0x72DB, 0x33F5, 0x9362, 0xB1A9, 0x113E, 0x5010, 0xF087
};

const uint32_t crc_tab24[] =
{
0x000000, 0x5D6DCB, 0xBADB96, 0xE7B65D, 0x28DAE7, 0x75B72C, 0x920171, 0xCF6CBA, 
0x51B5CE, 0x0CD805, 0xEB6E58, 0xB60393, 0x796F29, 0x2402E2, 0xC3B4BF, 0x9ED974, 
0xA36B9C, 0xFE0657, 0x19B00A, 0x44DDC1, 0x8BB17B, 0xD6DCB0, 0x316AED, 0x6C0726, 
0xF2DE52, 0xAFB399, 0x4805C4, 0x15680F, 0xDA04B5, 0x87697E, 0x60DF23, 0x3DB2E8, 
0x1BBAF3, 0x46D738, 0xA16165, 0xFC0CAE, 0x336014, 0x6E0DDF, 0x89BB82, 0xD4D649, 
0x4A0F3D, 0x1762F6, 0xF0D4AB, 0xADB960, 0x62D5DA, 0x3FB811, 0xD80E4C, 0x856387, 
0xB8D16F, 0xE5BCA4, 0x020AF9, 0x5F6732, 0x900B88, 0xCD6643, 0x2AD01E, 0x77BDD5, 
0xE964A1, 0xB4096A, 0x53BF37, 0x0ED2FC, 0xC1BE46, 0x9CD38D, 0x7B65D0, 0x26081B, 
0x3775E6, 0x6A182D, 0x8DAE70, 0xD0C3BB, 0x1FAF01, 0x42C2CA, 0xA57497, 0xF8195C, 
0x66C028, 0x3BADE3, 0xDC1BBE, 0x817675, 0x4E1ACF, 0x137704, 0xF4C159, 0xA9AC92,
0x941E7A, 0xC973B1, 0x2EC5EC, 0x73A827, 0xBCC49D, 0xE1A956, 0x061F0B, 0x5B72C0, 
0xC5ABB4, 0x98C67F, 0x7F7022, 0x221DE9, 0xED7153, 0xB01C98, 0x57AAC5, 0x0AC70E, 
0x2CCF15, 0x71A2DE, 0x961483, 0xCB7948, 0x0415F2, 0x597839, 0xBECE64, 0xE3A3AF, 
0x7D7ADB, 0x201710, 0xC7A14D, 0x9ACC86, 0x55A03C, 0x08CDF7, 0xEF7BAA, 0xB21661, 
0x8FA489, 0xD2C942, 0x357F1F, 0x6812D4, 0xA77E6E, 0xFA13A5, 0x1DA5F8, 0x40C833, 
0xDE1147, 0x837C8C, 0x64CAD1, 0x39A71A, 0xF6CBA0, 0xABA66B, 0x4C1036, 0x117DFD, 

0x6EEBCC, 0x338607, 0xD4305A, 0x895D91, 0x46312B, 0x1B5CE0, 0xFCEABD, 0xA18776, 
0x3F5E02, 0x6233C9, 0x858594, 0xD8E85F, 0x1784E5, 0x4AE92E, 0xAD5F73, 0xF032B8, 
0xCD8050, 0x90ED9B, 0x775BC6, 0x2A360D, 0xE55AB7, 0xB8377C, 0x5F8121, 0x02ECEA, 
0x9C359E, 0xC15855, 0x26EE08, 0x7B83C3, 0xB4EF79, 0xE982B2, 0x0E34EF, 0x535924,
0x75513F, 0x283CF4, 0xCF8AA9, 0x92E762, 0x5D8BD8, 0x00E613, 0xE7504E, 0xBA3D85, 
0x24E4F1, 0x79893A, 0x9E3F67, 0xC352AC, 0x0C3E16, 0x5153DD, 0xB6E580, 0xEB884B, 
0xD63AA3, 0x8B5768, 0x6CE135, 0x318CFE, 0xFEE044, 0xA38D8F, 0x443BD2, 0x195619, 
0x878F6D, 0xDAE2A6, 0x3D54FB, 0x603930, 0xAF558A, 0xF23841, 0x158E1C, 0x48E3D7, 
0x599E2A, 0x04F3E1, 0xE345BC, 0xBE2877, 0x7144CD, 0x2C2906, 0xCB9F5B, 0x96F290, 
0x082BE4, 0x55462F, 0xB2F072, 0xEF9DB9, 0x20F103, 0x7D9CC8, 0x9A2A95, 0xC7475E, 
0xFAF5B6, 0xA7987D, 0x402E20, 0x1D43EB, 0xD22F51, 0x8F429A, 0x68F4C7, 0x35990C, 
0xAB4078, 0xF62DB3, 0x119BEE, 0x4CF625, 0x839A9F, 0xDEF754, 0x394109, 0x642CC2, 
0x4224D9, 0x1F4912, 0xF8FF4F, 0xA59284, 0x6AFE3E, 0x3793F5, 0xD025A8, 0x8D4863, 
0x139117, 0x4EFCDC, 0xA94A81, 0xF4274A, 0x3B4BF0, 0x66263B, 0x819066, 0xDCFDAD,
0xE14F45, 0xBC228E, 0x5B94D3, 0x06F918, 0xC995A2, 0x94F869, 0x734E34, 0x2E23FF, 
0xB0FA8B, 0xED9740, 0x0A211D, 0x574CD6, 0x98206C, 0xC54DA7, 0x22FBFA, 0x7F9631
};                                                                                  
                                                                                        
const uint32_t crc_tab24r64[] =
{                                                                                       
0x000000, 0x864CFB, 0x8AD50D, 0x0C99F6, 0x93E6E1, 0x15AA1A, 0x1933EC, 0x9F7F17, 
0xA18139, 0x27CDC2, 0x2B5434, 0xAD18CF, 0x3267D8, 0xB42B23, 0xB8B2D5, 0x3EFE2E, 
0xC54E89, 0x430272, 0x4F9B84, 0xC9D77F, 0x56A868, 0xD0E493, 0xDC7D65, 0x5A319E, 
0x64CFB0, 0xE2834B, 0xEE1ABD, 0x685646, 0xF72951, 0x7165AA, 0x7DFC5C, 0xFBB0A7, 
0x0CD1E9, 0x8A9D12, 0x8604E4, 0x00481F, 0x9F3708, 0x197BF3, 0x15E205, 0x93AEFE, 
0xAD50D0, 0x2B1C2B, 0x2785DD, 0xA1C926, 0x3EB631, 0xB8FACA, 0xB4633C, 0x322FC7, 
0xC99F60, 0x4FD39B, 0x434A6D, 0xC50696, 0x5A7981, 0xDC357A, 0xD0AC8C, 0x56E077, 
0x681E59, 0xEE52A2, 0xE2CB54, 0x6487AF, 0xFBF8B8, 0x7DB443, 0x712DB5, 0xF7614E, 
0x19A3D2, 0x9FEF29, 0x9376DF, 0x153A24, 0x8A4533, 0x0C09C8, 0x00903E, 0x86DCC5, 
0xB822EB, 0x3E6E10, 0x32F7E6, 0xB4BB1D, 0x2BC40A, 0xAD88F1, 0xA11107, 0x275DFC,
0xDCED5B, 0x5AA1A0, 0x563856, 0xD074AD, 0x4F0BBA, 0xC94741, 0xC5DEB7, 0x43924C, 
0x7D6C62, 0xFB2099, 0xF7B96F, 0x71F594, 0xEE8A83, 0x68C678, 0x645F8E, 0xE21375, 
0x15723B, 0x933EC0, 0x9FA736, 0x19EBCD, 0x8694DA, 0x00D821, 0x0C41D7, 0x8A0D2C, 
0xB4F302, 0x32BFF9, 0x3E260F, 0xB86AF4, 0x2715E3, 0xA15918, 0xADC0EE, 0x2B8C15, 
0xD03CB2, 0x567049, 0x5AE9BF, 0xDCA544, 0x43DA53, 0xC596A8, 0xC90F5E, 0x4F43A5, 
0x71BD8B, 0xF7F170, 0xFB6886, 0x7D247D, 0xE25B6A, 0x641791, 0x688E67, 0xEEC29C, 

0x3347A4, 0xB50B5F, 0xB992A9, 0x3FDE52, 0xA0A145, 0x26EDBE, 0x2A7448, 0xAC38B3, 
0x92C69D, 0x148A66, 0x181390, 0x9E5F6B, 0x01207C, 0x876C87, 0x8BF571, 0x0DB98A, 
0xF6092D, 0x7045D6, 0x7CDC20, 0xFA90DB, 0x65EFCC, 0xE3A337, 0xEF3AC1, 0x69763A, 
0x578814, 0xD1C4EF, 0xDD5D19, 0x5B11E2, 0xC46EF5, 0x42220E, 0x4EBBF8, 0xC8F703,
0x3F964D, 0xB9DAB6, 0xB54340, 0x330FBB, 0xAC70AC, 0x2A3C57, 0x26A5A1, 0xA0E95A, 
0x9E1774, 0x185B8F, 0x14C279, 0x928E82, 0x0DF195, 0x8BBD6E, 0x872498, 0x016863, 
0xFAD8C4, 0x7C943F, 0x700DC9, 0xF64132, 0x693E25, 0xEF72DE, 0xE3EB28, 0x65A7D3, 
0x5B59FD, 0xDD1506, 0xD18CF0, 0x57C00B, 0xC8BF1C, 0x4EF3E7, 0x426A11, 0xC426EA, 
0x2AE476, 0xACA88D, 0xA0317B, 0x267D80, 0xB90297, 0x3F4E6C, 0x33D79A, 0xB59B61, 
0x8B654F, 0x0D29B4, 0x01B042, 0x87FCB9, 0x1883AE, 0x9ECF55, 0x9256A3, 0x141A58, 
0xEFAAFF, 0x69E604, 0x657FF2, 0xE33309, 0x7C4C1E, 0xFA00E5, 0xF69913, 0x70D5E8, 
0x4E2BC6, 0xC8673D, 0xC4FECB, 0x42B230, 0xDDCD27, 0x5B81DC, 0x57182A, 0xD154D1, 
0x26359F, 0xA07964, 0xACE092, 0x2AAC69, 0xB5D37E, 0x339F85, 0x3F0673, 0xB94A88, 
0x87B4A6, 0x01F85D, 0x0D61AB, 0x8B2D50, 0x145247, 0x921EBC, 0x9E874A, 0x18CBB1,
0xE37B16, 0x6537ED, 0x69AE1B, 0xEFE2E0, 0x709DF7, 0xF6D10C, 0xFA48FA, 0x7C0401, 
0x42FA2F, 0xC4B6D4, 0xC82F22, 0x4E63D9, 0xD11CCE, 0x575035, 0x5BC9C3, 0xDD8538
};

const uint32_t crc_tab32_reflected[] =
{
0x00000000, 0x77073096, 0xEE0E612C, 0x990951BA, 0x076DC419, 0x706AF48F, 0xE963A535, 0x9E6495A3, 
0x0EDB8832, 0x79DCB8A4, 0xE0D5E91E, 0x97D2D988, 0x09B64C2B, 0x7EB17CBD, 0xE7B82D07, 0x90BF1D91, 
0x1DB71064, 0x6AB020F2, 0xF3B97148, 0x84BE41DE, 0x1ADAD47D, 0x6DDDE4EB, 0xF4D4B551, 0x83D385C7, 
0x136C9856, 0x646BA8C0, 0xFD62F97A, 0x8A65C9EC, 0x14015C4F, 0x63066CD9, 0xFA0F3D63, 0x8D080DF5, 
0x3B6E20C8, 0x4C69105E, 0xD56041E4, 0xA2677172, 0x3C03E4D1, 0x4B04D447, 0xD20D85FD, 0xA50AB56B, 
0x35B5A8FA, 0x42B2986C, 0xDBBBC9D6, 0xACBCF940, 0x32D86CE3, 0x45DF5C75, 0xDCD60DCF, 0xABD13D59, 
0x26D930AC, 0x51DE003A, 0xC8D75180, 0xBFD06116, 0x21B4F4B5, 0x56B3C423, 0xCFBA9599, 0xB8BDA50F, 
0x2802B89E, 0x5F058808, 0xC60CD9B2, 0xB10BE924, 0x2F6F7C87, 0x58684C11, 0xC1611DAB, 0xB6662D3D, 
                                                                                              
0x76DC4190, 0x01DB7106, 0x98D220BC, 0xEFD5102A, 0x71B18589, 0x06B6B51F, 0x9FBFE4A5, 0xE8B8D433, 
0x7807C9A2, 0x0F00F934, 0x9609A88E, 0xE10E9818, 0x7F6A0DBB, 0x086D3D2D, 0x91646C97, 0xE6635C01, 
0x6B6B51F4, 0x1C6C6162, 0x856530D8, 0xF262004E, 0x6C0695ED, 0x1B01A57B, 0x8208F4C1, 0xF50FC457, 
0x65B0D9C6, 0x12B7E950, 0x8BBEB8EA, 0xFCB9887C, 0x62DD1DDF, 0x15DA2D49, 0x8CD37CF3, 0xFBD44C65, 
0x4DB26158, 0x3AB551CE, 0xA3BC0074, 0xD4BB30E2, 0x4ADFA541, 0x3DD895D7, 0xA4D1C46D, 0xD3D6F4FB, 
0x4369E96A, 0x346ED9FC, 0xAD678846, 0xDA60B8D0, 0x44042D73, 0x33031DE5, 0xAA0A4C5F, 0xDD0D7CC9, 
0x5005713C, 0x270241AA, 0xBE0B1010, 0xC90C2086, 0x5768B525, 0x206F85B3, 0xB966D409, 0xCE61E49F, 
0x5EDEF90E, 0x29D9C998, 0xB0D09822, 0xC7D7A8B4, 0x59B33D17, 0x2EB40D81, 0xB7BD5C3B, 0xC0BA6CAD, 
                                                                                              
0xEDB88320, 0x9ABFB3B6, 0x03B6E20C, 0x74B1D29A, 0xEAD54739, 0x9DD277AF, 0x04DB2615, 0x73DC1683, 
0xE3630B12, 0x94643B84, 0x0D6D6A3E, 0x7A6A5AA8, 0xE40ECF0B, 0x9309FF9D, 0x0A00AE27, 0x7D079EB1, 
0xF00F9344, 0x8708A3D2, 0x1E01F268, 0x6906C2FE, 0xF762575D, 0x806567CB, 0x196C3671, 0x6E6B06E7, 
0xFED41B76, 0x89D32BE0, 0x10DA7A5A, 0x67DD4ACC, 0xF9B9DF6F, 0x8EBEEFF9, 0x17B7BE43, 0x60B08ED5, 
0xD6D6A3E8, 0xA1D1937E, 0x38D8C2C4, 0x4FDFF252, 0xD1BB67F1, 0xA6BC5767, 0x3FB506DD, 0x48B2364B, 
0xD80D2BDA, 0xAF0A1B4C, 0x36034AF6, 0x41047A60, 0xDF60EFC3, 0xA867DF55, 0x316E8EEF, 0x4669BE79, 
0xCB61B38C, 0xBC66831A, 0x256FD2A0, 0x5268E236, 0xCC0C7795, 0xBB0B4703, 0x220216B9, 0x5505262F, 
0xC5BA3BBE, 0xB2BD0B28, 0x2BB45A92, 0x5CB36A04, 0xC2D7FFA7, 0xB5D0CF31, 0x2CD99E8B, 0x5BDEAE1D, 
                                                                                              
0x9B64C2B0, 0xEC63F226, 0x756AA39C, 0x026D930A, 0x9C0906A9, 0xEB0E363F, 0x72076785, 0x05005713, 
0x95BF4A82, 0xE2B87A14, 0x7BB12BAE, 0x0CB61B38, 0x92D28E9B, 0xE5D5BE0D, 0x7CDCEFB7, 0x0BDBDF21, 
0x86D3D2D4, 0xF1D4E242, 0x68DDB3F8, 0x1FDA836E, 0x81BE16CD, 0xF6B9265B, 0x6FB077E1, 0x18B74777, 
0x88085AE6, 0xFF0F6A70, 0x66063BCA, 0x11010B5C, 0x8F659EFF, 0xF862AE69, 0x616BFFD3, 0x166CCF45, 
0xA00AE278, 0xD70DD2EE, 0x4E048354, 0x3903B3C2, 0xA7672661, 0xD06016F7, 0x4969474D, 0x3E6E77DB, 
0xAED16A4A, 0xD9D65ADC, 0x40DF0B66, 0x37D83BF0, 0xA9BCAE53, 0xDEBB9EC5, 0x47B2CF7F, 0x30B5FFE9, 
0xBDBDF21C, 0xCABAC28A, 0x53B39330, 0x24B4A3A6, 0xBAD03605, 0xCDD70693, 0x54DE5729, 0x23D967BF, 
0xB3667A2E, 0xC4614AB8, 0x5D681B02, 0x2A6F2B94, 0xB40BBE37, 0xC30C8EA1, 0x5A05DF1B, 0x2D02EF8D
};                                                                                             
                                                                                              
const uint32_t crc_tab32_normal[] =
{                                                                                             
0x00000000, 0x04C11DB7, 0x09823B6E, 0x0D4326D9, 0x130476DC, 0x17C56B6B, 0x1A864DB2, 0x1E475005, 
0x2608EDB8, 0x22C9F00F, 0x2F8AD6D6, 0x2B4BCB61, 0x350C9B64, 0x31CD86D3, 0x3C8EA00A, 0x384FBDBD, 
0x4C11DB70, 0x48D0C6C7, 0x4593E01E, 0x4152FDA9, 0x5F15ADAC, 0x5BD4B01B, 0x569796C2, 0x52568B75, 
0x6A1936C8, 0x6ED82B7F, 0x639B0DA6, 0x675A1011, 0x791D4014, 0x7DDC5DA3, 0x709F7B7A, 0x745E66CD, 
0x9823B6E0, 0x9CE2AB57, 0x91A18D8E, 0x95609039, 0x8B27C03C, 0x8FE6DD8B, 0x82A5FB52, 0x8664E6E5, 
0xBE2B5B58, 0xBAEA46EF, 0xB7A96036, 0xB3687D81, 0xAD2F2D84, 0xA9EE3033, 0xA4AD16EA, 0xA06C0B5D, 
0xD4326D90, 0xD0F37027, 0xDDB056FE, 0xD9714B49, 0xC7361B4C, 0xC3F706FB, 0xCEB42022, 0xCA753D95, 
0xF23A8028, 0xF6FB9D9F, 0xFBB8BB46, 0xFF79A6F1, 0xE13EF6F4, 0xE5FFEB43, 0xE8BCCD9A, 0xEC7DD02D, 
0x34867077, 0x30476DC0, 0x3D044B19, 0x39C556AE, 0x278206AB, 0x23431B1C, 0x2E003DC5, 0x2AC12072, 
0x128E9DCF, 0x164F8078, 0x1B0CA6A1, 0x1FCDBB16, 0x018AEB13, 0x054BF6A4, 0x0808D07D, 0x0CC9CDCA, 
0x7897AB07, 0x7C56B6B0, 0x71159069, 0x75D48DDE, 0x6B93DDDB, 0x6F52C06C, 0x6211E6B5, 0x66D0FB02, 
0x5E9F46BF, 0x5A5E5B08, 0x571D7DD1, 0x53DC6066, 0x4D9B3063, 0x495A2DD4, 0x44190B0D, 0x40D816BA, 
0xACA5C697, 0xA864DB20, 0xA527FDF9, 0xA1E6E04E, 0xBFA1B04B, 0xBB60ADFC, 0xB6238B25, 0xB2E29692, 
0x8AAD2B2F, 0x8E6C3698, 0x832F1041, 0x87EE0DF6, 0x99A95DF3, 0x9D684044, 0x902B669D, 0x94EA7B2A, 
0xE0B41DE7, 0xE4750050, 0xE9362689, 0xEDF73B3E, 0xF3B06B3B, 0xF771768C, 0xFA325055, 0xFEF34DE2, 
0xC6BCF05F, 0xC27DEDE8, 0xCF3ECB31, 0xCBFFD686, 0xD5B88683, 0xD1799B34, 0xDC3ABDED, 0xD8FBA05A, 
                                                                                              
0x690CE0EE, 0x6DCDFD59, 0x608EDB80, 0x644FC637, 0x7A089632, 0x7EC98B85, 0x738AAD5C, 0x774BB0EB, 
0x4F040D56, 0x4BC510E1, 0x46863638, 0x42472B8F, 0x5C007B8A, 0x58C1663D, 0x558240E4, 0x51435D53, 
0x251D3B9E, 0x21DC2629, 0x2C9F00F0, 0x285E1D47, 0x36194D42, 0x32D850F5, 0x3F9B762C, 0x3B5A6B9B, 
0x0315D626, 0x07D4CB91, 0x0A97ED48, 0x0E56F0FF, 0x1011A0FA, 0x14D0BD4D, 0x19939B94, 0x1D528623, 
0xF12F560E, 0xF5EE4BB9, 0xF8AD6D60, 0xFC6C70D7, 0xE22B20D2, 0xE6EA3D65, 0xEBA91BBC, 0xEF68060B, 
0xD727BBB6, 0xD3E6A601, 0xDEA580D8, 0xDA649D6F, 0xC423CD6A, 0xC0E2D0DD, 0xCDA1F604, 0xC960EBB3, 
0xBD3E8D7E, 0xB9FF90C9, 0xB4BCB610, 0xB07DABA7, 0xAE3AFBA2, 0xAAFBE615, 0xA7B8C0CC, 0xA379DD7B, 
0x9B3660C6, 0x9FF77D71, 0x92B45BA8, 0x9675461F, 0x8832161A, 0x8CF30BAD, 0x81B02D74, 0x857130C3, 
0x5D8A9099, 0x594B8D2E, 0x5408ABF7, 0x50C9B640, 0x4E8EE645, 0x4A4FFBF2, 0x470CDD2B, 0x43CDC09C, 
0x7B827D21, 0x7F436096, 0x7200464F, 0x76C15BF8, 0x68860BFD, 0x6C47164A, 0x61043093, 0x65C52D24, 
0x119B4BE9, 0x155A565E, 0x18197087, 0x1CD86D30, 0x029F3D35, 0x065E2082, 0x0B1D065B, 0x0FDC1BEC, 
0x3793A651, 0x3352BBE6, 0x3E119D3F, 0x3AD08088, 0x2497D08D, 0x2056CD3A, 0x2D15EBE3, 0x29D4F654, 
0xC5A92679, 0xC1683BCE, 0xCC2B1D17, 0xC8EA00A0, 0xD6AD50A5, 0xD26C4D12, 0xDF2F6BCB, 0xDBEE767C, 
0xE3A1CBC1, 0xE760D676, 0xEA23F0AF, 0xEEE2ED18, 0xF0A5BD1D, 0xF464A0AA, 0xF9278673, 0xFDE69BC4, 
0x89B8FD09, 0x8D79E0BE, 0x803AC667, 0x84FBDBD0, 0x9ABC8BD5, 0x9E7D9662, 0x933EB0BB, 0x97FFAD0C, 
0xAFB010B1, 0xAB710D06, 0xA6322BDF, 0xA2F33668, 0xBCB4666D, 0xB8757BDA, 0xB5365D03, 0xB1F740B4
};
                                                                                              
const uint32_t crc_tabxfer_normal[] =
{                                                                                             
0x00000000, 0x000000AF, 0x0000015E, 0x000001F1, 0x000002BC, 0x00000213, 0x000003E2, 0x0000034D, 
0x00000578, 0x000005D7, 0x00000426, 0x00000489, 0x000007C4, 0x0000076B, 0x0000069A, 0x00000635, 
0x00000AF0, 0x00000A5F, 0x00000BAE, 0x00000B01, 0x0000084C, 0x000008E3, 0x00000912, 0x000009BD, 
0x00000F88, 0x00000F27, 0x00000ED6, 0x00000E79, 0x00000D34, 0x00000D9B, 0x00000C6A, 0x00000CC5, 
0x000015E0, 0x0000154F, 0x000014BE, 0x00001411, 0x0000175C, 0x000017F3, 0x00001602, 0x000016AD, 
0x00001098, 0x00001037, 0x000011C6, 0x00001169, 0x00001224, 0x0000128B, 0x0000137A, 0x000013D5, 
0x00001F10, 0x00001FBF, 0x00001E4E, 0x00001EE1, 0x00001DAC, 0x00001D03, 0x00001CF2, 0x00001C5D, 
0x00001A68, 0x00001AC7, 0x00001B36, 0x00001B99, 0x000018D4, 0x0000187B, 0x0000198A, 0x00001925, 
0x00002BC0, 0x00002B6F, 0x00002A9E, 0x00002A31, 0x0000297C, 0x000029D3, 0x00002822, 0x0000288D, 
0x00002EB8, 0x00002E17, 0x00002FE6, 0x00002F49, 0x00002C04, 0x00002CAB, 0x00002D5A, 0x00002DF5, 
0x00002130, 0x0000219F, 0x0000206E, 0x000020C1, 0x0000238C, 0x00002323, 0x000022D2, 0x0000227D, 
0x00002448, 0x000024E7, 0x00002516, 0x000025B9, 0x000026F4, 0x0000265B, 0x000027AA, 0x00002705, 
0x00003E20, 0x00003E8F, 0x00003F7E, 0x00003FD1, 0x00003C9C, 0x00003C33, 0x00003DC2, 0x00003D6D, 
0x00003B58, 0x00003BF7, 0x00003A06, 0x00003AA9, 0x000039E4, 0x0000394B, 0x000038BA, 0x00003815, 
0x000034D0, 0x0000347F, 0x0000358E, 0x00003521, 0x0000366C, 0x000036C3, 0x00003732, 0x0000379D, 
0x000031A8, 0x00003107, 0x000030F6, 0x00003059, 0x00003314, 0x000033BB, 0x0000324A, 0x000032E5, 
                                                                                              
0x00005780, 0x0000572F, 0x000056DE, 0x00005671, 0x0000553C, 0x00005593, 0x00005462, 0x000054CD, 
0x000052F8, 0x00005257, 0x000053A6, 0x00005309, 0x00005044, 0x000050EB, 0x0000511A, 0x000051B5, 
0x00005D70, 0x00005DDF, 0x00005C2E, 0x00005C81, 0x00005FCC, 0x00005F63, 0x00005E92, 0x00005E3D, 
0x00005808, 0x000058A7, 0x00005956, 0x000059F9, 0x00005AB4, 0x00005A1B, 0x00005BEA, 0x00005B45, 
0x00004260, 0x000042CF, 0x0000433E, 0x00004391, 0x000040DC, 0x00004073, 0x00004182, 0x0000412D, 
0x00004718, 0x000047B7, 0x00004646, 0x000046E9, 0x000045A4, 0x0000450B, 0x000044FA, 0x00004455, 
0x00004890, 0x0000483F, 0x000049CE, 0x00004961, 0x00004A2C, 0x00004A83, 0x00004B72, 0x00004BDD, 
0x00004DE8, 0x00004D47, 0x00004CB6, 0x00004C19, 0x00004F54, 0x00004FFB, 0x00004E0A, 0x00004EA5, 
0x00007C40, 0x00007CEF, 0x00007D1E, 0x00007DB1, 0x00007EFC, 0x00007E53, 0x00007FA2, 0x00007F0D, 
0x00007938, 0x00007997, 0x00007866, 0x000078C9, 0x00007B84, 0x00007B2B, 0x00007ADA, 0x00007A75, 
0x000076B0, 0x0000761F, 0x000077EE, 0x00007741, 0x0000740C, 0x000074A3, 0x00007552, 0x000075FD, 
0x000073C8, 0x00007367, 0x00007296, 0x00007239, 0x00007174, 0x000071DB, 0x0000702A, 0x00007085, 
0x000069A0, 0x0000690F, 0x000068FE, 0x00006851, 0x00006B1C, 0x00006BB3, 0x00006A42, 0x00006AED, 
0x00006CD8, 0x00006C77, 0x00006D86, 0x00006D29, 0x00006E64, 0x00006ECB, 0x00006F3A, 0x00006F95, 
0x00006350, 0x000063FF, 0x0000620E, 0x000062A1, 0x000061EC, 0x00006143, 0x000060B2, 0x0000601D, 
0x00006628, 0x00006687, 0x00006776, 0x000067D9, 0x00006494, 0x0000643B, 0x000065CA, 0x00006565
};
                                                                                              
const uint32_t crc_tab32C[] =
{                                                                                             
0x00000000, 0xF26B8303, 0xE13B70F7, 0x1350F3F4, 0xC79A971F, 0x35F1141C, 0x26A1E7E8, 0xD4CA64EB, 
0x8AD958CF, 0x78B2DBCC, 0x6BE22838, 0x9989AB3B, 0x4D43CFD0, 0xBF284CD3, 0xAC78BF27, 0x5E133C24, 
0x105EC76F, 0xE235446C, 0xF165B798, 0x030E349B, 0xD7C45070, 0x25AFD373, 0x36FF2087, 0xC494A384, 
0x9A879FA0, 0x68EC1CA3, 0x7BBCEF57, 0x89D76C54, 0x5D1D08BF, 0xAF768BBC, 0xBC267848, 0x4E4DFB4B, 
0x20BD8EDE, 0xD2D60DDD, 0xC186FE29, 0x33ED7D2A, 0xE72719C1, 0x154C9AC2, 0x061C6936, 0xF477EA35, 
0xAA64D611, 0x580F5512, 0x4B5FA6E6, 0xB93425E5, 0x6DFE410E, 0x9F95C20D, 0x8CC531F9, 0x7EAEB2FA, 
0x30E349B1, 0xC288CAB2, 0xD1D83946, 0x23B3BA45, 0xF779DEAE, 0x05125DAD, 0x1642AE59, 0xE4292D5A, 
0xBA3A117E, 0x4851927D, 0x5B016189, 0xA96AE28A, 0x7DA08661, 0x8FCB0562, 0x9C9BF696, 0x6EF07595, 
0x417B1DBC, 0xB3109EBF, 0xA0406D4B, 0x522BEE48, 0x86E18AA3, 0x748A09A0, 0x67DAFA54, 0x95B17957, 
0xCBA24573, 0x39C9C670, 0x2A993584, 0xD8F2B687, 0x0C38D26C, 0xFE53516F, 0xED03A29B, 0x1F682198, 
0x5125DAD3, 0xA34E59D0, 0xB01EAA24, 0x42752927, 0x96BF4DCC, 0x64D4CECF, 0x77843D3B, 0x85EFBE38, 
0xDBFC821C, 0x2997011F, 0x3AC7F2EB, 0xC8AC71E8, 0x1C661503, 0xEE0D9600, 0xFD5D65F4, 0x0F36E6F7, 
0x61C69362, 0x93AD1061, 0x80FDE395, 0x72966096, 0xA65C047D, 0x5437877E, 0x4767748A, 0xB50CF789, 
0xEB1FCBAD, 0x197448AE, 0x0A24BB5A, 0xF84F3859, 0x2C855CB2, 0xDEEEDFB1, 0xCDBE2C45, 0x3FD5AF46, 
0x7198540D, 0x83F3D70E, 0x90A324FA, 0x62C8A7F9, 0xB602C312, 0x44694011, 0x5739B3E5, 0xA55230E6, 
0xFB410CC2, 0x092A8FC1, 0x1A7A7C35, 0xE811FF36, 0x3CDB9BDD, 0xCEB018DE, 0xDDE0EB2A, 0x2F8B6829, 
                                                                                              
0x82F63B78, 0x709DB87B, 0x63CD4B8F, 0x91A6C88C, 0x456CAC67, 0xB7072F64, 0xA457DC90, 0x563C5F93, 
0x082F63B7, 0xFA44E0B4, 0xE9141340, 0x1B7F9043, 0xCFB5F4A8, 0x3DDE77AB, 0x2E8E845F, 0xDCE5075C, 
0x92A8FC17, 0x60C37F14, 0x73938CE0, 0x81F80FE3, 0x55326B08, 0xA759E80B, 0xB4091BFF, 0x466298FC, 
0x1871A4D8, 0xEA1A27DB, 0xF94AD42F, 0x0B21572C, 0xDFEB33C7, 0x2D80B0C4, 0x3ED04330, 0xCCBBC033, 
0xA24BB5A6, 0x502036A5, 0x4370C551, 0xB11B4652, 0x65D122B9, 0x97BAA1BA, 0x84EA524E, 0x7681D14D, 
0x2892ED69, 0xDAF96E6A, 0xC9A99D9E, 0x3BC21E9D, 0xEF087A76, 0x1D63F975, 0x0E330A81, 0xFC588982, 
0xB21572C9, 0x407EF1CA, 0x532E023E, 0xA145813D, 0x758FE5D6, 0x87E466D5, 0x94B49521, 0x66DF1622, 
0x38CC2A06, 0xCAA7A905, 0xD9F75AF1, 0x2B9CD9F2, 0xFF56BD19, 0x0D3D3E1A, 0x1E6DCDEE, 0xEC064EED, 
0xC38D26C4, 0x31E6A5C7, 0x22B65633, 0xD0DDD530, 0x0417B1DB, 0xF67C32D8, 0xE52CC12C, 0x1747422F, 
0x49547E0B, 0xBB3FFD08, 0xA86F0EFC, 0x5A048DFF, 0x8ECEE914, 0x7CA56A17, 0x6FF599E3, 0x9D9E1AE0, 
0xD3D3E1AB, 0x21B862A8, 0x32E8915C, 0xC083125F, 0x144976B4, 0xE622F5B7, 0xF5720643, 0x07198540, 
0x590AB964, 0xAB613A67, 0xB831C993, 0x4A5A4A90, 0x9E902E7B, 0x6CFBAD78, 0x7FAB5E8C, 0x8DC0DD8F, 
0xE330A81A, 0x115B2B19, 0x020BD8ED, 0xF0605BEE, 0x24AA3F05, 0xD6C1BC06, 0xC5914FF2, 0x37FACCF1, 
0x69E9F0D5, 0x9B8273D6, 0x88D28022, 0x7AB90321, 0xAE7367CA, 0x5C18E4C9, 0x4F48173D, 0xBD23943E, 
0xF36E6F75, 0x0105EC76, 0x12551F82, 0xE03E9C81, 0x34F4F86A, 0xC69F7B69, 0xD5CF889D, 0x27A40B9E, 
0x79B737BA, 0x8BDCB4B9, 0x988C474D, 0x6AE7C44E, 0xBE2DA0A5, 0x4C4623A6, 0x5F16D052, 0xAD7D5351
};
                                                                                              
const uint32_t crc_tab32D[] =
{                                                                                             
0x00000000, 0x2BDDD04F, 0x57BBA09E, 0x7C6670D1, 0xAF77413C, 0x84AA9173, 0xF8CCE1A2, 0xD31131ED, 
0xF6DD1A53, 0xDD00CA1C, 0xA166BACD, 0x8ABB6A82, 0x59AA5B6F, 0x72778B20, 0x0E11FBF1, 0x25CC2BBE, 
0x4589AC8D, 0x6E547CC2, 0x12320C13, 0x39EFDC5C, 0xEAFEEDB1, 0xC1233DFE, 0xBD454D2F, 0x96989D60, 
0xB354B6DE, 0x98896691, 0xE4EF1640, 0xCF32C60F, 0x1C23F7E2, 0x37FE27AD, 0x4B98577C, 0x60458733, 
0x8B13591A, 0xA0CE8955, 0xDCA8F984, 0xF77529CB, 0x24641826, 0x0FB9C869, 0x73DFB8B8, 0x580268F7, 
0x7DCE4349, 0x56139306, 0x2A75E3D7, 0x01A83398, 0xD2B90275, 0xF964D23A, 0x8502A2EB, 0xAEDF72A4, 
0xCE9AF597, 0xE54725D8, 0x99215509, 0xB2FC8546, 0x61EDB4AB, 0x4A3064E4, 0x36561435, 0x1D8BC47A, 
0x3847EFC4, 0x139A3F8B, 0x6FFC4F5A, 0x44219F15, 0x9730AEF8, 0xBCED7EB7, 0xC08B0E66, 0xEB56DE29, 
0xBE152A1F, 0x95C8FA50, 0xE9AE8A81, 0xC2735ACE, 0x11626B23, 0x3ABFBB6C, 0x46D9CBBD, 0x6D041BF2, 
0x48C8304C, 0x6315E003, 0x1F7390D2, 0x34AE409D, 0xE7BF7170, 0xCC62A13F, 0xB004D1EE, 0x9BD901A1, 
0xFB9C8692, 0xD04156DD, 0xAC27260C, 0x87FAF643, 0x54EBC7AE, 0x7F3617E1, 0x03506730, 0x288DB77F, 
0x0D419CC1, 0x269C4C8E, 0x5AFA3C5F, 0x7127EC10, 0xA236DDFD, 0x89EB0DB2, 0xF58D7D63, 0xDE50AD2C, 
0x35067305, 0x1EDBA34A, 0x62BDD39B, 0x496003D4, 0x9A713239, 0xB1ACE276, 0xCDCA92A7, 0xE61742E8, 
0xC3DB6956, 0xE806B919, 0x9460C9C8, 0xBFBD1987, 0x6CAC286A, 0x4771F825, 0x3B1788F4, 0x10CA58BB, 
0x708FDF88, 0x5B520FC7, 0x27347F16, 0x0CE9AF59, 0xDFF89EB4, 0xF4254EFB, 0x88433E2A, 0xA39EEE65, 
0x8652C5DB, 0xAD8F1594, 0xD1E96545, 0xFA34B50A, 0x292584E7, 0x02F854A8, 0x7E9E2479, 0x5543F436, 
                                                                                              
0xD419CC15, 0xFFC41C5A, 0x83A26C8B, 0xA87FBCC4, 0x7B6E8D29, 0x50B35D66, 0x2CD52DB7, 0x0708FDF8, 
0x22C4D646, 0x09190609, 0x757F76D8, 0x5EA2A697, 0x8DB3977A, 0xA66E4735, 0xDA0837E4, 0xF1D5E7AB, 
0x91906098, 0xBA4DB0D7, 0xC62BC006, 0xEDF61049, 0x3EE721A4, 0x153AF1EB, 0x695C813A, 0x42815175, 
0x674D7ACB, 0x4C90AA84, 0x30F6DA55, 0x1B2B0A1A, 0xC83A3BF7, 0xE3E7EBB8, 0x9F819B69, 0xB45C4B26, 
0x5F0A950F, 0x74D74540, 0x08B13591, 0x236CE5DE, 0xF07DD433, 0xDBA0047C, 0xA7C674AD, 0x8C1BA4E2, 
0xA9D78F5C, 0x820A5F13, 0xFE6C2FC2, 0xD5B1FF8D, 0x06A0CE60, 0x2D7D1E2F, 0x511B6EFE, 0x7AC6BEB1, 
0x1A833982, 0x315EE9CD, 0x4D38991C, 0x66E54953, 0xB5F478BE, 0x9E29A8F1, 0xE24FD820, 0xC992086F, 
0xEC5E23D1, 0xC783F39E, 0xBBE5834F, 0x90385300, 0x432962ED, 0x68F4B2A2, 0x1492C273, 0x3F4F123C, 
0x6A0CE60A, 0x41D13645, 0x3DB74694, 0x166A96DB, 0xC57BA736, 0xEEA67779, 0x92C007A8, 0xB91DD7E7, 
0x9CD1FC59, 0xB70C2C16, 0xCB6A5CC7, 0xE0B78C88, 0x33A6BD65, 0x187B6D2A, 0x641D1DFB, 0x4FC0CDB4, 
0x2F854A87, 0x04589AC8, 0x783EEA19, 0x53E33A56, 0x80F20BBB, 0xAB2FDBF4, 0xD749AB25, 0xFC947B6A, 
0xD95850D4, 0xF285809B, 0x8EE3F04A, 0xA53E2005, 0x762F11E8, 0x5DF2C1A7, 0x2194B176, 0x0A496139, 
0xE11FBF10, 0xCAC26F5F, 0xB6A41F8E, 0x9D79CFC1, 0x4E68FE2C, 0x65B52E63, 0x19D35EB2, 0x320E8EFD, 
0x17C2A543, 0x3C1F750C, 0x407905DD, 0x6BA4D592, 0xB8B5E47F, 0x93683430, 0xEF0E44E1, 0xC4D394AE, 
0xA496139D, 0x8F4BC3D2, 0xF32DB303, 0xD8F0634C, 0x0BE152A1, 0x203C82EE, 0x5C5AF23F, 0x77872270, 
0x524B09CE, 0x7996D981, 0x05F0A950, 0x2E2D791F, 0xFD3C48F2, 0xD6E198BD, 0xAA87E86C, 0x815A3823
};
                                                                                              
const uint32_t crc_tab32K[] =
{                                                                                             
0x00000000, 0x741B8CD7, 0xE83719AE, 0x9C2C9579, 0xA475BF8B, 0xD06E335C, 0x4C42A625, 0x38592AF2, 
0x3CF0F3C1, 0x48EB7F16, 0xD4C7EA6F, 0xA0DC66B8, 0x98854C4A, 0xEC9EC09D, 0x70B255E4, 0x04A9D933, 
0x79E1E782, 0x0DFA6B55, 0x91D6FE2C, 0xE5CD72FB, 0xDD945809, 0xA98FD4DE, 0x35A341A7, 0x41B8CD70, 
0x45111443, 0x310A9894, 0xAD260DED, 0xD93D813A, 0xE164ABC8, 0x957F271F, 0x0953B266, 0x7D483EB1, 
0xF3C3CF04, 0x87D843D3, 0x1BF4D6AA, 0x6FEF5A7D, 0x57B6708F, 0x23ADFC58, 0xBF816921, 0xCB9AE5F6, 
0xCF333CC5, 0xBB28B012, 0x2704256B, 0x531FA9BC, 0x6B46834E, 0x1F5D0F99, 0x83719AE0, 0xF76A1637, 
0x8A222886, 0xFE39A451, 0x62153128, 0x160EBDFF, 0x2E57970D, 0x5A4C1BDA, 0xC6608EA3, 0xB27B0274, 
0xB6D2DB47, 0xC2C95790, 0x5EE5C2E9, 0x2AFE4E3E, 0x12A764CC, 0x66BCE81B, 0xFA907D62, 0x8E8BF1B5, 
0x939C12DF, 0xE7879E08, 0x7BAB0B71, 0x0FB087A6, 0x37E9AD54, 0x43F22183, 0xDFDEB4FA, 0xABC5382D, 
0xAF6CE11E, 0xDB776DC9, 0x475BF8B0, 0x33407467, 0x0B195E95, 0x7F02D242, 0xE32E473B, 0x9735CBEC, 
0xEA7DF55D, 0x9E66798A, 0x024AECF3, 0x76516024, 0x4E084AD6, 0x3A13C601, 0xA63F5378, 0xD224DFAF, 
0xD68D069C, 0xA2968A4B, 0x3EBA1F32, 0x4AA193E5, 0x72F8B917, 0x06E335C0, 0x9ACFA0B9, 0xEED42C6E, 
0x605FDDDB, 0x1444510C, 0x8868C475, 0xFC7348A2, 0xC42A6250, 0xB031EE87, 0x2C1D7BFE, 0x5806F729, 
0x5CAF2E1A, 0x28B4A2CD, 0xB49837B4, 0xC083BB63, 0xF8DA9191, 0x8CC11D46, 0x10ED883F, 0x64F604E8, 
0x19BE3A59, 0x6DA5B68E, 0xF18923F7, 0x8592AF20, 0xBDCB85D2, 0xC9D00905, 0x55FC9C7C, 0x21E710AB, 
0x254EC998, 0x5155454F, 0xCD79D036, 0xB9625CE1, 0x813B7613, 0xF520FAC4, 0x690C6FBD, 0x1D17E36A, 
                                                                                              
0x5323A969, 0x273825BE, 0xBB14B0C7, 0xCF0F3C10, 0xF75616E2, 0x834D9A35, 0x1F610F4C, 0x6B7A839B, 
0x6FD35AA8, 0x1BC8D67F, 0x87E44306, 0xF3FFCFD1, 0xCBA6E523, 0xBFBD69F4, 0x2391FC8D, 0x578A705A, 
0x2AC24EEB, 0x5ED9C23C, 0xC2F55745, 0xB6EEDB92, 0x8EB7F160, 0xFAAC7DB7, 0x6680E8CE, 0x129B6419, 
0x1632BD2A, 0x622931FD, 0xFE05A484, 0x8A1E2853, 0xB24702A1, 0xC65C8E76, 0x5A701B0F, 0x2E6B97D8, 
0xA0E0666D, 0xD4FBEABA, 0x48D77FC3, 0x3CCCF314, 0x0495D9E6, 0x708E5531, 0xECA2C048, 0x98B94C9F, 
0x9C1095AC, 0xE80B197B, 0x74278C02, 0x003C00D5, 0x38652A27, 0x4C7EA6F0, 0xD0523389, 0xA449BF5E, 
0xD90181EF, 0xAD1A0D38, 0x31369841, 0x452D1496, 0x7D743E64, 0x096FB2B3, 0x954327CA, 0xE158AB1D, 
0xE5F1722E, 0x91EAFEF9, 0x0DC66B80, 0x79DDE757, 0x4184CDA5, 0x359F4172, 0xA9B3D40B, 0xDDA858DC, 
0xC0BFBBB6, 0xB4A43761, 0x2888A218, 0x5C932ECF, 0x64CA043D, 0x10D188EA, 0x8CFD1D93, 0xF8E69144, 
0xFC4F4877, 0x8854C4A0, 0x147851D9, 0x6063DD0E, 0x583AF7FC, 0x2C217B2B, 0xB00DEE52, 0xC4166285, 
0xB95E5C34, 0xCD45D0E3, 0x5169459A, 0x2572C94D, 0x1D2BE3BF, 0x69306F68, 0xF51CFA11, 0x810776C6, 
0x85AEAFF5, 0xF1B52322, 0x6D99B65B, 0x19823A8C, 0x21DB107E, 0x55C09CA9, 0xC9EC09D0, 0xBDF78507, 
0x337C74B2, 0x4767F865, 0xDB4B6D1C, 0xAF50E1CB, 0x9709CB39, 0xE31247EE, 0x7F3ED297, 0x0B255E40, 
0x0F8C8773, 0x7B970BA4, 0xE7BB9EDD, 0x93A0120A, 0xABF938F8, 0xDFE2B42F, 0x43CE2156, 0x37D5AD81, 
0x4A9D9330, 0x3E861FE7, 0xA2AA8A9E, 0xD6B10649, 0xEEE82CBB, 0x9AF3A06C, 0x06DF3515, 0x72C4B9C2, 
0x766D60F1, 0x0276EC26, 0x9E5A795F, 0xEA41F588, 0xD218DF7A, 0xA60353AD, 0x3A2FC6D4, 0x4E344A03
};
                                                                                              
const uint32_t crc_tab32Q[] =
{                                                                                             
0x00000000, 0x814141AB, 0x83C3C2FD, 0x02828356, 0x86C6C451, 0x078785FA, 0x050506AC, 0x84444707, 
0x8CCCC909, 0x0D8D88A2, 0x0F0F0BF4, 0x8E4E4A5F, 0x0A0A0D58, 0x8B4B4CF3, 0x89C9CFA5, 0x08888E0E, 
0x98D8D3B9, 0x19999212, 0x1B1B1144, 0x9A5A50EF, 0x1E1E17E8, 0x9F5F5643, 0x9DDDD515, 0x1C9C94BE, 
0x14141AB0, 0x95555B1B, 0x97D7D84D, 0x169699E6, 0x92D2DEE1, 0x13939F4A, 0x11111C1C, 0x90505DB7, 
0xB0F0E6D9, 0x31B1A772, 0x33332424, 0xB272658F, 0x36362288, 0xB7776323, 0xB5F5E075, 0x34B4A1DE, 
0x3C3C2FD0, 0xBD7D6E7B, 0xBFFFED2D, 0x3EBEAC86, 0xBAFAEB81, 0x3BBBAA2A, 0x3939297C, 0xB87868D7, 
0x28283560, 0xA96974CB, 0xABEBF79D, 0x2AAAB636, 0xAEEEF131, 0x2FAFB09A, 0x2D2D33CC, 0xAC6C7267, 
0xA4E4FC69, 0x25A5BDC2, 0x27273E94, 0xA6667F3F, 0x22223838, 0xA3637993, 0xA1E1FAC5, 0x20A0BB6E, 
0xE0A08C19, 0x61E1CDB2, 0x63634EE4, 0xE2220F4F, 0x66664848, 0xE72709E3, 0xE5A58AB5, 0x64E4CB1E, 
0x6C6C4510, 0xED2D04BB, 0xEFAF87ED, 0x6EEEC646, 0xEAAA8141, 0x6BEBC0EA, 0x696943BC, 0xE8280217, 
0x78785FA0, 0xF9391E0B, 0xFBBB9D5D, 0x7AFADCF6, 0xFEBE9BF1, 0x7FFFDA5A, 0x7D7D590C, 0xFC3C18A7, 
0xF4B496A9, 0x75F5D702, 0x77775454, 0xF63615FF, 0x727252F8, 0xF3331353, 0xF1B19005, 0x70F0D1AE, 
0x50506AC0, 0xD1112B6B, 0xD393A83D, 0x52D2E996, 0xD696AE91, 0x57D7EF3A, 0x55556C6C, 0xD4142DC7, 
0xDC9CA3C9, 0x5DDDE262, 0x5F5F6134, 0xDE1E209F, 0x5A5A6798, 0xDB1B2633, 0xD999A565, 0x58D8E4CE, 
0xC888B979, 0x49C9F8D2, 0x4B4B7B84, 0xCA0A3A2F, 0x4E4E7D28, 0xCF0F3C83, 0xCD8DBFD5, 0x4CCCFE7E, 
0x44447070, 0xC50531DB, 0xC787B28D, 0x46C6F326, 0xC282B421, 0x43C3F58A, 0x414176DC, 0xC0003777, 
                                                                                              
0x40005999, 0xC1411832, 0xC3C39B64, 0x4282DACF, 0xC6C69DC8, 0x4787DC63, 0x45055F35, 0xC4441E9E, 
0xCCCC9090, 0x4D8DD13B, 0x4F0F526D, 0xCE4E13C6, 0x4A0A54C1, 0xCB4B156A, 0xC9C9963C, 0x4888D797, 
0xD8D88A20, 0x5999CB8B, 0x5B1B48DD, 0xDA5A0976, 0x5E1E4E71, 0xDF5F0FDA, 0xDDDD8C8C, 0x5C9CCD27, 
0x54144329, 0xD5550282, 0xD7D781D4, 0x5696C07F, 0xD2D28778, 0x5393C6D3, 0x51114585, 0xD050042E, 
0xF0F0BF40, 0x71B1FEEB, 0x73337DBD, 0xF2723C16, 0x76367B11, 0xF7773ABA, 0xF5F5B9EC, 0x74B4F847, 
0x7C3C7649, 0xFD7D37E2, 0xFFFFB4B4, 0x7EBEF51F, 0xFAFAB218, 0x7BBBF3B3, 0x793970E5, 0xF878314E, 
0x68286CF9, 0xE9692D52, 0xEBEBAE04, 0x6AAAEFAF, 0xEEEEA8A8, 0x6FAFE903, 0x6D2D6A55, 0xEC6C2BFE, 
0xE4E4A5F0, 0x65A5E45B, 0x6727670D, 0xE66626A6, 0x622261A1, 0xE363200A, 0xE1E1A35C, 0x60A0E2F7, 
0xA0A0D580, 0x21E1942B, 0x2363177D, 0xA22256D6, 0x266611D1, 0xA727507A, 0xA5A5D32C, 0x24E49287, 
0x2C6C1C89, 0xAD2D5D22, 0xAFAFDE74, 0x2EEE9FDF, 0xAAAAD8D8, 0x2BEB9973, 0x29691A25, 0xA8285B8E, 
0x38780639, 0xB9394792, 0xBBBBC4C4, 0x3AFA856F, 0xBEBEC268, 0x3FFF83C3, 0x3D7D0095, 0xBC3C413E, 
0xB4B4CF30, 0x35F58E9B, 0x37770DCD, 0xB6364C66, 0x32720B61, 0xB3334ACA, 0xB1B1C99C, 0x30F08837, 
0x10503359, 0x911172F2, 0x9393F1A4, 0x12D2B00F, 0x9696F708, 0x17D7B6A3, 0x155535F5, 0x9414745E, 
0x9C9CFA50, 0x1DDDBBFB, 0x1F5F38AD, 0x9E1E7906, 0x1A5A3E01, 0x9B1B7FAA, 0x9999FCFC, 0x18D8BD57, 
0x8888E0E0, 0x09C9A14B, 0x0B4B221D, 0x8A0A63B6, 0x0E4E24B1, 0x8F0F651A, 0x8D8DE64C, 0x0CCCA7E7, 
0x044429E9, 0x85056842, 0x8787EB14, 0x06C6AABF, 0x8282EDB8, 0x03C3AC13, 0x01412F45, 0x80006EEE
};

const uint64_t crc_tab40gsm_normal[] =
{
0x0000000000, 0x0004820009, 0x0009040012, 0x000D86001B, 
0x0012080024, 0x00168A002D, 0x001B0C0036, 0x001F8E003F, 
0x0024100048, 0x0020920041, 0x002D14005A, 0x0029960053, 
0x003618006C, 0x00329A0065, 0x003F1C007E, 0x003B9E0077, 
0x0048200090, 0x004CA20099, 0x0041240082, 0x0045A6008B, 
0x005A2800B4, 0x005EAA00BD, 0x00532C00A6, 0x0057AE00AF, 
0x006C3000D8, 0x0068B200D1, 0x00653400CA, 0x0061B600C3, 
0x007E3800FC, 0x007ABA00F5, 0x00773C00EE, 0x0073BE00E7, 
0x0090400120, 0x0094C20129, 0x0099440132, 0x009DC6013B, 
0x0082480104, 0x0086CA010D, 0x008B4C0116, 0x008FCE011F, 
0x00B4500168, 0x00B0D20161, 0x00BD54017A, 0x00B9D60173, 
0x00A658014C, 0x00A2DA0145, 0x00AF5C015E, 0x00ABDE0157, 
0x00D86001B0, 0x00DCE201B9, 0x00D16401A2, 0x00D5E601AB, 
0x00CA680194, 0x00CEEA019D, 0x00C36C0186, 0x00C7EE018F, 
0x00FC7001F8, 0x00F8F201F1, 0x00F57401EA, 0x00F1F601E3, 
0x00EE7801DC, 0x00EAFA01D5, 0x00E77C01CE, 0x00E3FE01C7, 

0x0120800240, 0x0124020249, 0x0129840252, 0x012D06025B, 
0x0132880264, 0x01360A026D, 0x013B8C0276, 0x013F0E027F, 
0x0104900208, 0x0100120201, 0x010D94021A, 0x0109160213, 
0x011698022C, 0x01121A0225, 0x011F9C023E, 0x011B1E0237, 
0x0168A002D0, 0x016C2202D9, 0x0161A402C2, 0x01652602CB, 
0x017AA802F4, 0x017E2A02FD, 0x0173AC02E6, 0x01772E02EF, 
0x014CB00298, 0x0148320291, 0x0145B4028A, 0x0141360283, 
0x015EB802BC, 0x015A3A02B5, 0x0157BC02AE, 0x01533E02A7, 
0x01B0C00360, 0x01B4420369, 0x01B9C40372, 0x01BD46037B, 
0x01A2C80344, 0x01A64A034D, 0x01ABCC0356, 0x01AF4E035F, 
0x0194D00328, 0x0190520321, 0x019DD4033A, 0x0199560333, 
0x0186D8030C, 0x01825A0305, 0x018FDC031E, 0x018B5E0317, 
0x01F8E003F0, 0x01FC6203F9, 0x01F1E403E2, 0x01F56603EB, 
0x01EAE803D4, 0x01EE6A03DD, 0x01E3EC03C6, 0x01E76E03CF, 
0x01DCF003B8, 0x01D87203B1, 0x01D5F403AA, 0x01D17603A3, 
0x01CEF8039C, 0x01CA7A0395, 0x01C7FC038E, 0x01C37E0387, 

0x0241000480, 0x0245820489, 0x0248040492, 0x024C86049B, 
0x02530804A4, 0x02578A04AD, 0x025A0C04B6, 0x025E8E04BF, 
0x02651004C8, 0x02619204C1, 0x026C1404DA, 0x02689604D3, 
0x02771804EC, 0x02739A04E5, 0x027E1C04FE, 0x027A9E04F7, 
0x0209200410, 0x020DA20419, 0x0200240402, 0x0204A6040B, 
0x021B280434, 0x021FAA043D, 0x02122C0426, 0x0216AE042F, 
0x022D300458, 0x0229B20451, 0x022434044A, 0x0220B60443, 
0x023F38047C, 0x023BBA0475, 0x02363C046E, 0x0232BE0467, 
0x02D14005A0, 0x02D5C205A9, 0x02D84405B2, 0x02DCC605BB, 
0x02C3480584, 0x02C7CA058D, 0x02CA4C0596, 0x02CECE059F, 
0x02F55005E8, 0x02F1D205E1, 0x02FC5405FA, 0x02F8D605F3, 
0x02E75805CC, 0x02E3DA05C5, 0x02EE5C05DE, 0x02EADE05D7, 
0x0299600530, 0x029DE20539, 0x0290640522, 0x0294E6052B, 
0x028B680514, 0x028FEA051D, 0x02826C0506, 0x0286EE050F, 
0x02BD700578, 0x02B9F20571, 0x02B474056A, 0x02B0F60563, 
0x02AF78055C, 0x02ABFA0555, 0x02A67C054E, 0x02A2FE0547, 

0x03618006C0, 0x03650206C9, 0x03688406D2, 0x036C0606DB, 
0x03738806E4, 0x03770A06ED, 0x037A8C06F6, 0x037E0E06FF, 
0x0345900688, 0x0341120681, 0x034C94069A, 0x0348160693, 
0x03579806AC, 0x03531A06A5, 0x035E9C06BE, 0x035A1E06B7, 
0x0329A00650, 0x032D220659, 0x0320A40642, 0x032426064B, 
0x033BA80674, 0x033F2A067D, 0x0332AC0666, 0x03362E066F, 
0x030DB00618, 0x0309320611, 0x0304B4060A, 0x0300360603, 
0x031FB8063C, 0x031B3A0635, 0x0316BC062E, 0x03123E0627, 
0x03F1C007E0, 0x03F54207E9, 0x03F8C407F2, 0x03FC4607FB, 
0x03E3C807C4, 0x03E74A07CD, 0x03EACC07D6, 0x03EE4E07DF, 
0x03D5D007A8, 0x03D15207A1, 0x03DCD407BA, 0x03D85607B3, 
0x03C7D8078C, 0x03C35A0785, 0x03CEDC079E, 0x03CA5E0797, 
0x03B9E00770, 0x03BD620779, 0x03B0E40762, 0x03B466076B, 
0x03ABE80754, 0x03AF6A075D, 0x03A2EC0746, 0x03A66E074F, 
0x039DF00738, 0x0399720731, 0x0394F4072A, 0x0390760723, 
0x038FF8071C, 0x038B7A0715, 0x0386FC070E, 0x03827E0707
};

const uint64_t crc_tab64_normal[] =
{
0x0000000000000000, 0x42F0E1EBA9EA3693, 0x85E1C3D753D46D26, 0xC711223CFA3E5BB5, 
0x493366450E42ECDF, 0x0BC387AEA7A8DA4C, 0xCCD2A5925D9681F9, 0x8E224479F47CB76A, 
0x9266CC8A1C85D9BE, 0xD0962D61B56FEF2D, 0x17870F5D4F51B498, 0x5577EEB6E6BB820B, 
0xDB55AACF12C73561, 0x99A54B24BB2D03F2, 0x5EB4691841135847, 0x1C4488F3E8F96ED4, 
0x663D78FF90E185EF, 0x24CD9914390BB37C, 0xE3DCBB28C335E8C9, 0xA12C5AC36ADFDE5A, 
0x2F0E1EBA9EA36930, 0x6DFEFF5137495FA3, 0xAAEFDD6DCD770416, 0xE81F3C86649D3285, 
0xF45BB4758C645C51, 0xB6AB559E258E6AC2, 0x71BA77A2DFB03177, 0x334A9649765A07E4, 
0xBD68D2308226B08E, 0xFF9833DB2BCC861D, 0x388911E7D1F2DDA8, 0x7A79F00C7818EB3B, 
0xCC7AF1FF21C30BDE, 0x8E8A101488293D4D, 0x499B3228721766F8, 0x0B6BD3C3DBFD506B, 
0x854997BA2F81E701, 0xC7B97651866BD192, 0x00A8546D7C558A27, 0x4258B586D5BFBCB4, 
0x5E1C3D753D46D260, 0x1CECDC9E94ACE4F3, 0xDBFDFEA26E92BF46, 0x990D1F49C77889D5, 
0x172F5B3033043EBF, 0x55DFBADB9AEE082C, 0x92CE98E760D05399, 0xD03E790CC93A650A, 
0xAA478900B1228E31, 0xE8B768EB18C8B8A2, 0x2FA64AD7E2F6E317, 0x6D56AB3C4B1CD584, 
0xE374EF45BF6062EE, 0xA1840EAE168A547D, 0x66952C92ECB40FC8, 0x2465CD79455E395B, 
0x3821458AADA7578F, 0x7AD1A461044D611C, 0xBDC0865DFE733AA9, 0xFF3067B657990C3A, 
0x711223CFA3E5BB50, 0x33E2C2240A0F8DC3, 0xF4F3E018F031D676, 0xB60301F359DBE0E5, 
                                                                                           
0xDA050215EA6C212F, 0x98F5E3FE438617BC, 0x5FE4C1C2B9B84C09, 0x1D14202910527A9A, 
0x93366450E42ECDF0, 0xD1C685BB4DC4FB63, 0x16D7A787B7FAA0D6, 0x5427466C1E109645, 
0x4863CE9FF6E9F891, 0x0A932F745F03CE02, 0xCD820D48A53D95B7, 0x8F72ECA30CD7A324, 
0x0150A8DAF8AB144E, 0x43A04931514122DD, 0x84B16B0DAB7F7968, 0xC6418AE602954FFB, 
0xBC387AEA7A8DA4C0, 0xFEC89B01D3679253, 0x39D9B93D2959C9E6, 0x7B2958D680B3FF75, 
0xF50B1CAF74CF481F, 0xB7FBFD44DD257E8C, 0x70EADF78271B2539, 0x321A3E938EF113AA, 
0x2E5EB66066087D7E, 0x6CAE578BCFE24BED, 0xABBF75B735DC1058, 0xE94F945C9C3626CB, 
0x676DD025684A91A1, 0x259D31CEC1A0A732, 0xE28C13F23B9EFC87, 0xA07CF2199274CA14, 
0x167FF3EACBAF2AF1, 0x548F120162451C62, 0x939E303D987B47D7, 0xD16ED1D631917144, 
0x5F4C95AFC5EDC62E, 0x1DBC74446C07F0BD, 0xDAAD56789639AB08, 0x985DB7933FD39D9B, 
0x84193F60D72AF34F, 0xC6E9DE8B7EC0C5DC, 0x01F8FCB784FE9E69, 0x43081D5C2D14A8FA, 
0xCD2A5925D9681F90, 0x8FDAB8CE70822903, 0x48CB9AF28ABC72B6, 0x0A3B7B1923564425, 
0x70428B155B4EAF1E, 0x32B26AFEF2A4998D, 0xF5A348C2089AC238, 0xB753A929A170F4AB, 
0x3971ED50550C43C1, 0x7B810CBBFCE67552, 0xBC902E8706D82EE7, 0xFE60CF6CAF321874, 
0xE224479F47CB76A0, 0xA0D4A674EE214033, 0x67C58448141F1B86, 0x253565A3BDF52D15, 
0xAB1721DA49899A7F, 0xE9E7C031E063ACEC, 0x2EF6E20D1A5DF759, 0x6C0603E6B3B7C1CA, 
                                                                                           
0xF6FAE5C07D3274CD, 0xB40A042BD4D8425E, 0x731B26172EE619EB, 0x31EBC7FC870C2F78, 
0xBFC9838573709812, 0xFD39626EDA9AAE81, 0x3A28405220A4F534, 0x78D8A1B9894EC3A7, 
0x649C294A61B7AD73, 0x266CC8A1C85D9BE0, 0xE17DEA9D3263C055, 0xA38D0B769B89F6C6, 
0x2DAF4F0F6FF541AC, 0x6F5FAEE4C61F773F, 0xA84E8CD83C212C8A, 0xEABE6D3395CB1A19, 
0x90C79D3FEDD3F122, 0xD2377CD44439C7B1, 0x15265EE8BE079C04, 0x57D6BF0317EDAA97, 
0xD9F4FB7AE3911DFD, 0x9B041A914A7B2B6E, 0x5C1538ADB04570DB, 0x1EE5D94619AF4648, 
0x02A151B5F156289C, 0x4051B05E58BC1E0F, 0x87409262A28245BA, 0xC5B073890B687329, 
0x4B9237F0FF14C443, 0x0962D61B56FEF2D0, 0xCE73F427ACC0A965, 0x8C8315CC052A9FF6, 
0x3A80143F5CF17F13, 0x7870F5D4F51B4980, 0xBF61D7E80F251235, 0xFD913603A6CF24A6, 
0x73B3727A52B393CC, 0x31439391FB59A55F, 0xF652B1AD0167FEEA, 0xB4A25046A88DC879, 
0xA8E6D8B54074A6AD, 0xEA16395EE99E903E, 0x2D071B6213A0CB8B, 0x6FF7FA89BA4AFD18, 
0xE1D5BEF04E364A72, 0xA3255F1BE7DC7CE1, 0x64347D271DE22754, 0x26C49CCCB40811C7, 
0x5CBD6CC0CC10FAFC, 0x1E4D8D2B65FACC6F, 0xD95CAF179FC497DA, 0x9BAC4EFC362EA149, 
0x158E0A85C2521623, 0x577EEB6E6BB820B0, 0x906FC95291867B05, 0xD29F28B9386C4D96, 
0xCEDBA04AD0952342, 0x8C2B41A1797F15D1, 0x4B3A639D83414E64, 0x09CA82762AAB78F7, 
0x87E8C60FDED7CF9D, 0xC51827E4773DF90E, 0x020905D88D03A2BB, 0x40F9E43324E99428, 
                                                                                           
0x2CFFE7D5975E55E2, 0x6E0F063E3EB46371, 0xA91E2402C48A38C4, 0xEBEEC5E96D600E57, 
0x65CC8190991CB93D, 0x273C607B30F68FAE, 0xE02D4247CAC8D41B, 0xA2DDA3AC6322E288, 
0xBE992B5F8BDB8C5C, 0xFC69CAB42231BACF, 0x3B78E888D80FE17A, 0x7988096371E5D7E9, 
0xF7AA4D1A85996083, 0xB55AACF12C735610, 0x724B8ECDD64D0DA5, 0x30BB6F267FA73B36, 
0x4AC29F2A07BFD00D, 0x08327EC1AE55E69E, 0xCF235CFD546BBD2B, 0x8DD3BD16FD818BB8, 
0x03F1F96F09FD3CD2, 0x41011884A0170A41, 0x86103AB85A2951F4, 0xC4E0DB53F3C36767, 
0xD8A453A01B3A09B3, 0x9A54B24BB2D03F20, 0x5D45907748EE6495, 0x1FB5719CE1045206, 
0x919735E51578E56C, 0xD367D40EBC92D3FF, 0x1476F63246AC884A, 0x568617D9EF46BED9, 
0xE085162AB69D5E3C, 0xA275F7C11F7768AF, 0x6564D5FDE549331A, 0x279434164CA30589, 
0xA9B6706FB8DFB2E3, 0xEB46918411358470, 0x2C57B3B8EB0BDFC5, 0x6EA7525342E1E956, 
0x72E3DAA0AA188782, 0x30133B4B03F2B111, 0xF7021977F9CCEAA4, 0xB5F2F89C5026DC37, 
0x3BD0BCE5A45A6B5D, 0x79205D0E0DB05DCE, 0xBE317F32F78E067B, 0xFCC19ED95E6430E8, 
0x86B86ED5267CDBD3, 0xC4488F3E8F96ED40, 0x0359AD0275A8B6F5, 0x41A94CE9DC428066, 
0xCF8B0890283E370C, 0x8D7BE97B81D4019F, 0x4A6ACB477BEA5A2A, 0x089A2AACD2006CB9, 
0x14DEA25F3AF9026D, 0x562E43B4931334FE, 0x913F6188692D6F4B, 0xD3CF8063C0C759D8, 
0x5DEDC41A34BBEEB2, 0x1F1D25F19D51D821, 0xD80C07CD676F8394, 0x9AFCE626CE85B507 
};                                                                                          
                                                                                           
const uint64_t crc_tab641b_reflected[] =
{                                                                                          
0x0000000000000000, 0x01B0000000000000, 0x0360000000000000, 0x02D0000000000000, 
0x06C0000000000000, 0x0770000000000000, 0x05A0000000000000, 0x0410000000000000, 
0x0D80000000000000, 0x0C30000000000000, 0x0EE0000000000000, 0x0F50000000000000, 
0x0B40000000000000, 0x0AF0000000000000, 0x0820000000000000, 0x0990000000000000, 
0x1B00000000000000, 0x1AB0000000000000, 0x1860000000000000, 0x19D0000000000000, 
0x1DC0000000000000, 0x1C70000000000000, 0x1EA0000000000000, 0x1F10000000000000, 
0x1680000000000000, 0x1730000000000000, 0x15E0000000000000, 0x1450000000000000, 
0x1040000000000000, 0x11F0000000000000, 0x1320000000000000, 0x1290000000000000, 
0x3600000000000000, 0x37B0000000000000, 0x3560000000000000, 0x34D0000000000000, 
0x30C0000000000000, 0x3170000000000000, 0x33A0000000000000, 0x3210000000000000, 
0x3B80000000000000, 0x3A30000000000000, 0x38E0000000000000, 0x3950000000000000, 
0x3D40000000000000, 0x3CF0000000000000, 0x3E20000000000000, 0x3F90000000000000, 
0x2D00000000000000, 0x2CB0000000000000, 0x2E60000000000000, 0x2FD0000000000000, 
0x2BC0000000000000, 0x2A70000000000000, 0x28A0000000000000, 0x2910000000000000, 
0x2080000000000000, 0x2130000000000000, 0x23E0000000000000, 0x2250000000000000, 
0x2640000000000000, 0x27F0000000000000, 0x2520000000000000, 0x2490000000000000, 
                                                                                           
0x6C00000000000000, 0x6DB0000000000000, 0x6F60000000000000, 0x6ED0000000000000, 
0x6AC0000000000000, 0x6B70000000000000, 0x69A0000000000000, 0x6810000000000000, 
0x6180000000000000, 0x6030000000000000, 0x62E0000000000000, 0x6350000000000000, 
0x6740000000000000, 0x66F0000000000000, 0x6420000000000000, 0x6590000000000000, 
0x7700000000000000, 0x76B0000000000000, 0x7460000000000000, 0x75D0000000000000, 
0x71C0000000000000, 0x7070000000000000, 0x72A0000000000000, 0x7310000000000000, 
0x7A80000000000000, 0x7B30000000000000, 0x79E0000000000000, 0x7850000000000000, 
0x7C40000000000000, 0x7DF0000000000000, 0x7F20000000000000, 0x7E90000000000000, 
0x5A00000000000000, 0x5BB0000000000000, 0x5960000000000000, 0x58D0000000000000, 
0x5CC0000000000000, 0x5D70000000000000, 0x5FA0000000000000, 0x5E10000000000000, 
0x5780000000000000, 0x5630000000000000, 0x54E0000000000000, 0x5550000000000000, 
0x5140000000000000, 0x50F0000000000000, 0x5220000000000000, 0x5390000000000000, 
0x4100000000000000, 0x40B0000000000000, 0x4260000000000000, 0x43D0000000000000, 
0x47C0000000000000, 0x4670000000000000, 0x44A0000000000000, 0x4510000000000000, 
0x4C80000000000000, 0x4D30000000000000, 0x4FE0000000000000, 0x4E50000000000000, 
0x4A40000000000000, 0x4BF0000000000000, 0x4920000000000000, 0x4890000000000000, 
                                                                                           
0xD800000000000000, 0xD9B0000000000000, 0xDB60000000000000, 0xDAD0000000000000, 
0xDEC0000000000000, 0xDF70000000000000, 0xDDA0000000000000, 0xDC10000000000000, 
0xD580000000000000, 0xD430000000000000, 0xD6E0000000000000, 0xD750000000000000, 
0xD340000000000000, 0xD2F0000000000000, 0xD020000000000000, 0xD190000000000000, 
0xC300000000000000, 0xC2B0000000000000, 0xC060000000000000, 0xC1D0000000000000, 
0xC5C0000000000000, 0xC470000000000000, 0xC6A0000000000000, 0xC710000000000000, 
0xCE80000000000000, 0xCF30000000000000, 0xCDE0000000000000, 0xCC50000000000000, 
0xC840000000000000, 0xC9F0000000000000, 0xCB20000000000000, 0xCA90000000000000, 
0xEE00000000000000, 0xEFB0000000000000, 0xED60000000000000, 0xECD0000000000000, 
0xE8C0000000000000, 0xE970000000000000, 0xEBA0000000000000, 0xEA10000000000000, 
0xE380000000000000, 0xE230000000000000, 0xE0E0000000000000, 0xE150000000000000, 
0xE540000000000000, 0xE4F0000000000000, 0xE620000000000000, 0xE790000000000000, 
0xF500000000000000, 0xF4B0000000000000, 0xF660000000000000, 0xF7D0000000000000, 
0xF3C0000000000000, 0xF270000000000000, 0xF0A0000000000000, 0xF110000000000000, 
0xF880000000000000, 0xF930000000000000, 0xFBE0000000000000, 0xFA50000000000000, 
0xFE40000000000000, 0xFFF0000000000000, 0xFD20000000000000, 0xFC90000000000000, 
                                                                                           
0xB400000000000000, 0xB5B0000000000000, 0xB760000000000000, 0xB6D0000000000000, 
0xB2C0000000000000, 0xB370000000000000, 0xB1A0000000000000, 0xB010000000000000, 
0xB980000000000000, 0xB830000000000000, 0xBAE0000000000000, 0xBB50000000000000, 
0xBF40000000000000, 0xBEF0000000000000, 0xBC20000000000000, 0xBD90000000000000, 
0xAF00000000000000, 0xAEB0000000000000, 0xAC60000000000000, 0xADD0000000000000, 
0xA9C0000000000000, 0xA870000000000000, 0xAAA0000000000000, 0xAB10000000000000, 
0xA280000000000000, 0xA330000000000000, 0xA1E0000000000000, 0xA050000000000000, 
0xA440000000000000, 0xA5F0000000000000, 0xA720000000000000, 0xA690000000000000, 
0x8200000000000000, 0x83B0000000000000, 0x8160000000000000, 0x80D0000000000000, 
0x84C0000000000000, 0x8570000000000000, 0x87A0000000000000, 0x8610000000000000, 
0x8F80000000000000, 0x8E30000000000000, 0x8CE0000000000000, 0x8D50000000000000, 
0x8940000000000000, 0x88F0000000000000, 0x8A20000000000000, 0x8B90000000000000, 
0x9900000000000000, 0x98B0000000000000, 0x9A60000000000000, 0x9BD0000000000000, 
0x9FC0000000000000, 0x9E70000000000000, 0x9CA0000000000000, 0x9D10000000000000, 
0x9480000000000000, 0x9530000000000000, 0x97E0000000000000, 0x9650000000000000, 
0x9240000000000000, 0x93F0000000000000, 0x9120000000000000, 0x9090000000000000 
};                                                                                          
                                                                                           
const uint64_t crc_tab64jones_reflected[] =
{                                                                                          
0x0000000000000000, 0x7AD870C830358979, 0xF5B0E190606B12F2, 0x8F689158505E9B8B, 
0xC038E5739841B68F, 0xBAE095BBA8743FF6, 0x358804E3F82AA47D, 0x4F50742BC81F2D04, 
0xAB28ECB46814FE75, 0xD1F09C7C5821770C, 0x5E980D24087FEC87, 0x24407DEC384A65FE, 
0x6B1009C7F05548FA, 0x11C8790FC060C183, 0x9EA0E857903E5A08, 0xE478989FA00BD371, 
0x7D08FF3B88BE6F81, 0x07D08FF3B88BE6F8, 0x88B81EABE8D57D73, 0xF2606E63D8E0F40A, 
0xBD301A4810FFD90E, 0xC7E86A8020CA5077, 0x4880FBD87094CBFC, 0x32588B1040A14285, 
0xD620138FE0AA91F4, 0xACF86347D09F188D, 0x2390F21F80C18306, 0x594882D7B0F40A7F, 
0x1618F6FC78EB277B, 0x6CC0863448DEAE02, 0xE3A8176C18803589, 0x997067A428B5BCF0, 
0xFA11FE77117CDF02, 0x80C98EBF2149567B, 0x0FA11FE77117CDF0, 0x75796F2F41224489, 
0x3A291B04893D698D, 0x40F16BCCB908E0F4, 0xCF99FA94E9567B7F, 0xB5418A5CD963F206, 
0x513912C379682177, 0x2BE1620B495DA80E, 0xA489F35319033385, 0xDE51839B2936BAFC, 
0x9101F7B0E12997F8, 0xEBD98778D11C1E81, 0x64B116208142850A, 0x1E6966E8B1770C73, 
0x8719014C99C2B083, 0xFDC17184A9F739FA, 0x72A9E0DCF9A9A271, 0x08719014C99C2B08, 
0x4721E43F0183060C, 0x3DF994F731B68F75, 0xB29105AF61E814FE, 0xC849756751DD9D87, 
0x2C31EDF8F1D64EF6, 0x56E99D30C1E3C78F, 0xD9810C6891BD5C04, 0xA3597CA0A188D57D, 
0xEC09088B6997F879, 0x96D1784359A27100, 0x19B9E91B09FCEA8B, 0x636199D339C963F2, 
                                                                                           
0xDF7ADABD7A6E2D6F, 0xA5A2AA754A5BA416, 0x2ACA3B2D1A053F9D, 0x50124BE52A30B6E4, 
0x1F423FCEE22F9BE0, 0x659A4F06D21A1299, 0xEAF2DE5E82448912, 0x902AAE96B271006B, 
0x74523609127AD31A, 0x0E8A46C1224F5A63, 0x81E2D7997211C1E8, 0xFB3AA75142244891, 
0xB46AD37A8A3B6595, 0xCEB2A3B2BA0EECEC, 0x41DA32EAEA507767, 0x3B024222DA65FE1E, 
0xA2722586F2D042EE, 0xD8AA554EC2E5CB97, 0x57C2C41692BB501C, 0x2D1AB4DEA28ED965, 
0x624AC0F56A91F461, 0x1892B03D5AA47D18, 0x97FA21650AFAE693, 0xED2251AD3ACF6FEA, 
0x095AC9329AC4BC9B, 0x7382B9FAAAF135E2, 0xFCEA28A2FAAFAE69, 0x8632586ACA9A2710, 
0xC9622C4102850A14, 0xB3BA5C8932B0836D, 0x3CD2CDD162EE18E6, 0x460ABD1952DB919F, 
0x256B24CA6B12F26D, 0x5FB354025B277B14, 0xD0DBC55A0B79E09F, 0xAA03B5923B4C69E6, 
0xE553C1B9F35344E2, 0x9F8BB171C366CD9B, 0x10E3202993385610, 0x6A3B50E1A30DDF69, 
0x8E43C87E03060C18, 0xF49BB8B633338561, 0x7BF329EE636D1EEA, 0x012B592653589793, 
0x4E7B2D0D9B47BA97, 0x34A35DC5AB7233EE, 0xBBCBCC9DFB2CA865, 0xC113BC55CB19211C, 
0x5863DBF1E3AC9DEC, 0x22BBAB39D3991495, 0xADD33A6183C78F1E, 0xD70B4AA9B3F20667, 
0x985B3E827BED2B63, 0xE2834E4A4BD8A21A, 0x6DEBDF121B863991, 0x1733AFDA2BB3B0E8, 
0xF34B37458BB86399, 0x8993478DBB8DEAE0, 0x06FBD6D5EBD3716B, 0x7C23A61DDBE6F812, 
0x3373D23613F9D516, 0x49ABA2FE23CC5C6F, 0xC6C333A67392C7E4, 0xBC1B436E43A74E9D, 
                                                                                           
0x95AC9329AC4BC9B5, 0xEF74E3E19C7E40CC, 0x601C72B9CC20DB47, 0x1AC40271FC15523E, 
0x5594765A340A7F3A, 0x2F4C0692043FF643, 0xA02497CA54616DC8, 0xDAFCE7026454E4B1, 
0x3E847F9DC45F37C0, 0x445C0F55F46ABEB9, 0xCB349E0DA4342532, 0xB1ECEEC59401AC4B, 
0xFEBC9AEE5C1E814F, 0x8464EA266C2B0836, 0x0B0C7B7E3C7593BD, 0x71D40BB60C401AC4, 
0xE8A46C1224F5A634, 0x927C1CDA14C02F4D, 0x1D148D82449EB4C6, 0x67CCFD4A74AB3DBF, 
0x289C8961BCB410BB, 0x5244F9A98C8199C2, 0xDD2C68F1DCDF0249, 0xA7F41839ECEA8B30, 
0x438C80A64CE15841, 0x3954F06E7CD4D138, 0xB63C61362C8A4AB3, 0xCCE411FE1CBFC3CA, 
0x83B465D5D4A0EECE, 0xF96C151DE49567B7, 0x76048445B4CBFC3C, 0x0CDCF48D84FE7545, 
0x6FBD6D5EBD3716B7, 0x15651D968D029FCE, 0x9A0D8CCEDD5C0445, 0xE0D5FC06ED698D3C, 
0xAF85882D2576A038, 0xD55DF8E515432941, 0x5A3569BD451DB2CA, 0x20ED197575283BB3, 
0xC49581EAD523E8C2, 0xBE4DF122E51661BB, 0x3125607AB548FA30, 0x4BFD10B2857D7349, 
0x04AD64994D625E4D, 0x7E7514517D57D734, 0xF11D85092D094CBF, 0x8BC5F5C11D3CC5C6, 
0x12B5926535897936, 0x686DE2AD05BCF04F, 0xE70573F555E26BC4, 0x9DDD033D65D7E2BD, 
0xD28D7716ADC8CFB9, 0xA85507DE9DFD46C0, 0x273D9686CDA3DD4B, 0x5DE5E64EFD965432, 
0xB99D7ED15D9D8743, 0xC3450E196DA80E3A, 0x4C2D9F413DF695B1, 0x36F5EF890DC31CC8, 
0x79A59BA2C5DC31CC, 0x037DEB6AF5E9B8B5, 0x8C157A32A5B7233E, 0xF6CD0AFA9582AA47, 
                                                                                           
0x4AD64994D625E4DA, 0x300E395CE6106DA3, 0xBF66A804B64EF628, 0xC5BED8CC867B7F51, 
0x8AEEACE74E645255, 0xF036DC2F7E51DB2C, 0x7F5E4D772E0F40A7, 0x05863DBF1E3AC9DE, 
0xE1FEA520BE311AAF, 0x9B26D5E88E0493D6, 0x144E44B0DE5A085D, 0x6E963478EE6F8124, 
0x21C640532670AC20, 0x5B1E309B16452559, 0xD476A1C3461BBED2, 0xAEAED10B762E37AB, 
0x37DEB6AF5E9B8B5B, 0x4D06C6676EAE0222, 0xC26E573F3EF099A9, 0xB8B627F70EC510D0, 
0xF7E653DCC6DA3DD4, 0x8D3E2314F6EFB4AD, 0x0256B24CA6B12F26, 0x788EC2849684A65F, 
0x9CF65A1B368F752E, 0xE62E2AD306BAFC57, 0x6946BB8B56E467DC, 0x139ECB4366D1EEA5, 
0x5CCEBF68AECEC3A1, 0x2616CFA09EFB4AD8, 0xA97E5EF8CEA5D153, 0xD3A62E30FE90582A, 
0xB0C7B7E3C7593BD8, 0xCA1FC72BF76CB2A1, 0x45775673A732292A, 0x3FAF26BB9707A053, 
0x70FF52905F188D57, 0x0A2722586F2D042E, 0x854FB3003F739FA5, 0xFF97C3C80F4616DC, 
0x1BEF5B57AF4DC5AD, 0x61372B9F9F784CD4, 0xEE5FBAC7CF26D75F, 0x9487CA0FFF135E26, 
0xDBD7BE24370C7322, 0xA10FCEEC0739FA5B, 0x2E675FB4576761D0, 0x54BF2F7C6752E8A9, 
0xCDCF48D84FE75459, 0xB71738107FD2DD20, 0x387FA9482F8C46AB, 0x42A7D9801FB9CFD2, 
0x0DF7ADABD7A6E2D6, 0x772FDD63E7936BAF, 0xF8474C3BB7CDF024, 0x829F3CF387F8795D, 
0x66E7A46C27F3AA2C, 0x1C3FD4A417C62355, 0x935745FC4798B8DE, 0xE98F353477AD31A7, 
0xA6DF411FBFB21CA3, 0xDC0731D78F8795DA, 0x536FA08FDFD90E51, 0x29B7D047EFEC8728
};


#define                 P_SICK      0x8005

uint8_t update_crc8_atm( uint8_t crc, uint8_t c )
{
    return (crc8_table_atm[crc ^ c]);
}

uint8_t update_crc8_ccitt( uint8_t crc, uint8_t c )
{
    return (crc8_table_ccitt[crc ^ c]);
}

uint8_t update_crc8_maxim( uint8_t crc, uint8_t c )
{
    return (crc8_table_maxim[crc ^ c]);
}

uint8_t update_crc8( uint8_t crc, uint8_t c )
{
    return (crc8_table[crc ^ c]);
}

uint8_t update_crc8_j1850( uint8_t crc, uint8_t c )
{
    return (crc8_table_j1850[crc ^ c]);
}

uint8_t update_crc8_wcdma( uint8_t crc, uint8_t c )
{
    return (crc8_table_wcdma[crc ^ c]);
}

uint8_t update_crc8_rohc( uint8_t crc, uint8_t c )
{
    return (crc8_table_rohc[crc ^ c]);
}

uint8_t update_crc8_darc( uint8_t crc, uint8_t c )
{
    return (crc8_table_darc[crc ^ c]);
}

/* Common routines for calculations */
uint16_t update_crc16_normal(const uint16_t *table, uint16_t crc, char c )
{
    uint16_t short_c;

    short_c  = 0x00ff & (uint16_t) c;

    /* Normal form */
    return (crc << 8) ^ table[(crc >> 8) ^ short_c];
}

uint16_t update_crc16_reflected(const uint16_t *table, uint16_t crc, char c )
{
    uint16_t short_c;

    short_c  = 0x00ff & (uint16_t) c;

    /* Reflected form */
    return (crc >> 8) ^ table[(crc ^ short_c) & 0xff];
}

uint32_t update_crc24_normal(const uint32_t *table, uint32_t crc, char c )
{
    uint32_t long_c;

    long_c = 0x000000ffL & (uint32_t) c;

    return (crc << 8) ^ table[((crc >> 16) ^ long_c) & 0xff];
}

uint32_t update_crc32_normal(const uint32_t *table, uint32_t crc, char c )
{
    uint32_t long_c;

    long_c = 0x000000ffL & (uint32_t) c;

    return (crc << 8) ^ table[((crc >> 24) ^ long_c) & 0xff];
}

uint32_t update_crc32_reflected(const uint32_t *table, uint32_t crc, char c )
{
    uint32_t long_c;

    long_c = 0x000000ffL & (uint32_t) c;

    return (crc >> 8) ^ table[(crc ^ long_c) & 0xff];
}

uint64_t update_crc40_normal(const uint64_t *table, uint64_t crc, char c )
{
    uint64_t long64_c;

    long64_c = 0x00000000000000ffULL & (uint64_t) c;

    return (crc << 8) ^ table[((crc >> 32) ^ long64_c) & 0xff];
}

uint64_t update_crc64_normal(const uint64_t *table, uint64_t crc, char c )
{
    uint64_t long64_c;

    long64_c = 0x00000000000000ffULL & (uint64_t) c;

    return (crc << 8) ^ table[((crc >> 56) ^ long64_c) & 0xff];
}

uint64_t update_crc64_reflected(const uint64_t *table, uint64_t crc, char c )
{
    uint64_t long64_c;

    long64_c = 0x00000000000000ffULL & (uint64_t) c;

    return (crc >> 8) ^ table[(crc ^ long64_c) & 0xff];
}

uint16_t update_crc_sick( uint16_t crc, char c, char prev_byte )
{
    uint16_t short_c, short_p;

    short_c  =   0x00ff & (uint16_t) c;
    short_p  = ( 0x00ff & (uint16_t) prev_byte ) << 8;

    if ( crc & 0x8000 ) crc = ( crc << 1 ) ^ P_SICK;
    else                crc =   crc << 1;

    crc &= 0xffff;
    crc ^= ( short_c | short_p );

    return crc;
}

uint16_t update_crc16_8005( uint16_t crc, char c )
{
    return update_crc16_normal(crc_tab_8005_normal,crc,c);
}
uint16_t update_crc16_A001( uint16_t crc, char c )
{
    return update_crc16_reflected(crc_tab_8005_reflected,crc,c);
}
uint16_t update_crc16_1021( uint16_t crc, char c )
{
    return update_crc16_normal(crc_tab_1021_normal,crc,c);
}
uint16_t update_crc16_8408( uint16_t crc, char c )
{
    return update_crc16_reflected(crc_tab_1021_reflected,crc,c);
}
uint16_t update_crc16_3D65( uint16_t crc, char c )
{
    return update_crc16_normal(crc_tab_3d65_normal,crc,c);
}
uint16_t update_crc16_dnp( uint16_t crc, char c )
{
    return update_crc16_reflected(crc_tab_3d65_reflected,crc,c);
}
uint16_t update_crc16_t10_dif( uint16_t crc, char c )
{
    return update_crc16_normal(crc_tabt10dif,crc,c);
}
uint16_t update_crc16_0589( uint16_t crc, char c )
{
    return update_crc16_normal(crc_tabdect,crc,c);
}
uint16_t update_crc16_teledisk( uint16_t crc, char c )
{
    return update_crc16_normal(crc_tabteledisk,crc,c);
}
uint32_t update_crc24( uint32_t crc, char c )
{
    return update_crc24_normal(crc_tab24,crc,c);
}
uint32_t update_crc24_r64( uint32_t crc, char c )
{
    return update_crc24_normal(crc_tab24r64,crc,c);
}
uint32_t update_crc32_refl( uint32_t crc, char c )
{
    return update_crc32_reflected(crc_tab32_reflected,crc,c);
}
uint32_t update_crc32_norm( uint32_t crc, char c )
{
    return update_crc32_normal(crc_tab32_normal,crc,c);
}
uint32_t update_crc32_xfer( uint32_t crc, char c )
{
    return update_crc32_normal(crc_tabxfer_normal,crc,c);
}
uint32_t update_crc32_c( uint32_t crc, char c )
{
  return update_crc32_reflected(crc_tab32C,crc,c);
}
uint32_t update_crc32_d( uint32_t crc, char c )
{
  return update_crc32_reflected(crc_tab32D,crc,c);
}
uint32_t update_crc32_k( uint32_t crc, char c )
{
    return update_crc32_normal(crc_tab32K,crc,c);
}
uint32_t update_crc32_q( uint32_t crc, char c )
{
    return update_crc32_normal(crc_tab32Q,crc,c);
}

uint64_t update_crc40_gsm(uint64_t crc, char c)
{
    return update_crc40_normal(crc_tab40gsm_normal,crc,c);
}

uint64_t update_crc64(uint64_t crc, char c)
{
    return update_crc64_normal(crc_tab64_normal,crc,c);
}

uint64_t update_crc64_1B(uint64_t crc, char c)
{
    return update_crc64_reflected(crc_tab641b_reflected,crc,c);
}

uint64_t update_crc64_jones(uint64_t crc, char c)
{
    return update_crc64_reflected(crc_tab64jones_reflected,crc,c);
}

uint8_t calculate_crc8_itu(char *p, unsigned int length)
{
uint8_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc8_atm(crc,*p++);
      }
    return crc ^ 0x55;
}

uint8_t calculate_crc8_atm(char *p, unsigned int length)
{
uint8_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc8_atm(crc,*p++);
      }
    return crc;
}

uint8_t calculate_crc8_ccitt(char *p, unsigned int length)
{
uint8_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc8_ccitt(crc,*p++);
      }
    return crc;
}

uint8_t calculate_crc8_maxim(char *p, unsigned int length)
{
uint8_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc8_maxim(crc,*p++);
      }
    return crc;
}

uint8_t calculate_crc8(char *p, unsigned int length)
{
uint8_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc8(crc,*p++);
      }
    return crc;
}

uint8_t calculate_crc8_icode(char *p, unsigned int length)
{
uint8_t crc;
unsigned int i;

    crc = 0xFD;

    for (i=0; i < length; i++)
      {
        crc = update_crc8_j1850(crc,*p++);
      }
    return crc;
}

uint8_t calculate_crc8_j1850(char *p, unsigned int length)
{
uint8_t crc;
unsigned int i;

    crc = 0xFF;

    for (i=0; i < length; i++)
      {
        crc = update_crc8_j1850(crc,*p++);
      }
    return ~crc;
}

uint8_t calculate_crc8_wcdma(char *p, unsigned int length)
{
uint8_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc8_wcdma(crc,*p++);
      }
    return crc;
}

uint8_t calculate_crc8_rohc(char *p, unsigned int length)
{
uint8_t crc;
unsigned int i;

    crc = 0xFF;

    for (i=0; i < length; i++)
      {
        crc = update_crc8_rohc(crc,*p++);
      }
    return crc;
}

uint8_t calculate_crc8_darc(char *p, unsigned int length)
{
uint8_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc8_darc(crc,*p++);
      }
    return crc;
}

uint16_t calculate_crc16_Buypass(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_8005(crc,*p++);
      }
    return crc;
}

uint16_t calculate_crc16_DDS_110(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0x800D;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_8005(crc,*p++);
      }
    return crc;
}

uint16_t calculate_crc16_EN_13757(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_3D65(crc,*p++);
      }
    return ~crc;
}


uint16_t calculate_crc16_Teledisk(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_teledisk(crc,*p++);
      }
    return crc;
}

uint16_t calculate_crc16(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_A001(crc,*p++);
      }
    return crc;
}

uint16_t calculate_crc16_Modbus(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0xFFFF;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_A001(crc,*p++);
      }
    return crc;
}

uint16_t calculate_crc16_Maxim(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_A001(crc,*p++);
      }
    return ~crc;
}

uint16_t calculate_crc16_USB(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0xFFFF;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_A001(crc,*p++);
      }
    return ~crc;
}

uint16_t calculate_crc16_T10_DIF(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_t10_dif(crc,*p++);
      }
    return crc;
}

uint16_t calculate_crc16_Dect_X(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_0589(crc,*p++);
      }
    return crc;
}

uint16_t calculate_crc16_Dect_R(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_0589(crc,*p++);
      }
    return crc ^ 0x0001;
}

uint16_t calculate_crc16_sick(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;
char Prev_Byte = 0;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc_sick(crc,*p,Prev_Byte);
        Prev_Byte = *p++;
      }
    /* Reverse endian */
    return (crc >> 8) | ((crc & 0xFF) << 8);
}

uint16_t calculate_crc16_DNP(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_dnp(crc,*p++);
      }
    crc = ~crc;
    /* Reverse endian */
    return (crc >> 8) | ((crc & 0xFF) << 8);
}

uint16_t calculate_crc16_Ccitt_Xmodem(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_1021(crc,*p++);
      }
    return crc;
}

uint16_t calculate_crc16_Ccitt_FFFF(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0xFFFF;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_1021(crc,*p++);
      }
    return crc;
}

uint16_t calculate_crc16_Ccitt_1D0F(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0x1D0F;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_1021(crc,*p++);
      }
    return crc;
}

uint16_t calculate_crc16_Genibus(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0xFFFF;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_1021(crc,*p++);
      }
    return ~crc;
}

uint16_t calculate_crc16_Kermit(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_8408(crc,*p++);
      }
    /* Reverse endian */
    return (crc >> 8) | ((crc & 0xFF) << 8);
}

uint16_t calculate_crc16_X25(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0xFFFF;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_8408(crc,*p++);
      }
    return ~crc;
}

uint16_t calculate_crc16_MCRF4XX(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0xFFFF;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_8408(crc,*p++);
      }
    return crc;
}

uint16_t calculate_crc16_Riello(char *p, unsigned int length)
{
uint16_t crc;
unsigned int i;

    crc = 0x554D;

    for (i=0; i < length; i++)
      {
        crc = update_crc16_8408(crc,*p++);
      }
    return crc;
}

uint16_t calculate_chk16_Fletcher(char *p, unsigned int length)
{
uint16_t check, check_fletcher;
unsigned int i;

    check = 0;
    check_fletcher = 0;

    for (i=0; i < length; i++)
      {
        check += (*p++);
        check_fletcher += check;
      }
    return ((check_fletcher & 0xFF) << 8) | (check & 0xFF);
}

uint32_t calculate_crc24_flexray_a(char *p, unsigned int length)
{
uint32_t crc32;
unsigned int i;

    crc32 = 0x00FEDCBAL;

    for (i=0; i < length; i++)
      {
          crc32 = update_crc24(crc32,*p++);
      }

    return crc32;
}

uint32_t calculate_crc24_flexray_b(char *p, unsigned int length)
{
uint32_t crc32;
unsigned int i;

    crc32 = 0x00ABCDEFL;

    for (i=0; i < length; i++)
      {
          crc32 = update_crc24(crc32,*p++);
      }

    return crc32;
}

uint32_t calculate_crc24_r64(char *p, unsigned int length)
{
uint32_t crc32;
unsigned int i;

    crc32 = 0x00B704CEL;

    for (i=0; i < length; i++)
      {
          crc32 = update_crc24_r64(crc32,*p++);
      }

    return crc32;
}

uint32_t calculate_crc32(char *p, unsigned int length)
{
uint32_t crc32;
unsigned int i;

    crc32 = 0xFFFFFFFFL;

    for (i=0; i < length; i++)
      {
          crc32 = update_crc32_refl(crc32,*p++);
      }

    /* One's complement = Xor with FFFFFFFF */
    return ~crc32;
}

uint32_t calculate_crc32_jamcrc(char *p, unsigned int length)
{
uint32_t crc32;
unsigned int i;

    crc32 = 0xFFFFFFFFL;

    for (i=0; i < length; i++)
      {
          crc32 = update_crc32_refl(crc32,*p++);
      }

    return crc32;
}

uint32_t calculate_crc32_c(char *p, unsigned int length)
{
uint32_t crc32;
unsigned int i;

    crc32 = 0xFFFFFFFFL;

    for (i=0; i < length; i++)
      {
          crc32 = update_crc32_c(crc32,*p++);
      }

    /* One's complement = Xor with FFFFFFFF */
    return ~crc32;
}

uint32_t calculate_crc32_d(char *p, unsigned int length)
{
uint32_t crc32;
unsigned int i;

    crc32 = 0xFFFFFFFFL;

    for (i=0; i < length; i++)
      {
          crc32 = update_crc32_d(crc32,*p++);
      }

    /* One's complement = Xor with FFFFFFFF */
    return ~crc32;
}

uint32_t calculate_crc32_bzip2(char *p, unsigned int length)
{
uint32_t crc32;
unsigned int i;

    crc32 = 0xFFFFFFFFL;

    for (i=0; i < length; i++)
      {
          crc32 = update_crc32_norm(crc32,*p++);
      }

    /* One's complement = Xor with FFFFFFFF */
    return ~crc32;
}

uint32_t calculate_crc32_mpeg2(char *p, unsigned int length)
{
uint32_t crc32;
unsigned int i;

    crc32 = 0xFFFFFFFFL;

    for (i=0; i < length; i++)
      {
          crc32 = update_crc32_norm(crc32,*p++);
      }

    return crc32;
}

uint32_t calculate_crc32_posix(char *p, unsigned int length)
{
uint32_t crc32;
unsigned int i;

    crc32 = 0L;

    for (i=0; i < length; i++)
      {
          crc32 = update_crc32_norm(crc32,*p++);
      }

    /* One's complement = Xor with FFFFFFFF */
    return ~crc32;
}

uint32_t calculate_crc32_k(char *p, unsigned int length)
{
uint32_t crc32;
unsigned int i;

    crc32 = 0L;

    for (i=0; i < length; i++)
      {
          crc32 = update_crc32_k(crc32,*p++);
      }

    return crc32;
}

uint32_t calculate_crc32_q(char *p, unsigned int length)
{
uint32_t crc32;
unsigned int i;

    crc32 = 0L;

    for (i=0; i < length; i++)
      {
          crc32 = update_crc32_q(crc32,*p++);
      }

    return crc32;
}

uint32_t calculate_crc32_xfer(char *p, unsigned int length)
{
uint32_t crc32;
unsigned int i;

    crc32 = 0L;

    for (i=0; i < length; i++)
      {
          crc32 = update_crc32_xfer(crc32,*p++);
      }

    return crc32;
}

uint64_t calculate_crc40_gsm(char *p, unsigned int length)
{
uint64_t crc64;
unsigned int i;

    crc64 = 0ULL;

    for (i=0; i < length; i++)
      {
          crc64 = update_crc40_gsm(crc64,*p++);
      }

    return crc64;
}

uint64_t calculate_crc64(char *p, unsigned int length)
{
uint64_t crc64;
unsigned int i;

    crc64 = 0ULL;

    for (i=0; i < length; i++)
      {
        crc64 = update_crc64(crc64,*p++);
      }

    return crc64;
}

uint64_t calculate_crc64_1b(char *p, unsigned int length)
{
uint64_t crc64;
unsigned int i;

    crc64 = 0ULL;

    for (i=0; i < length; i++)
      {
        crc64 = update_crc64_1B(crc64,*p++);
      }

    return crc64;
}

uint64_t calculate_crc64_we(char *p, unsigned int length)
{
uint64_t crc64;
unsigned int i;

    crc64 = 0xFFFFFFFFFFFFFFFFULL;

    for (i=0; i < length; i++)
      {
        crc64 = update_crc64(crc64,*p++);
      }

    /* One's complement = Xor with FFFFFFFFFFFFFFFF */
    return ~crc64;
}

uint64_t calculate_crc64_jones(char *p, unsigned int length)
{
uint64_t crc64;
unsigned int i;

    crc64 = 0xFFFFFFFFFFFFFFFFULL;

    for (i=0; i < length; i++)
      {
          crc64 = update_crc64_jones(crc64,*p++);
      }

    return crc64;
}

