#include "AP1017.h"

/******************** Constructors & Destructors ****************************/

// Default constructor
AP1017::AP1017(DigitalOut* A, DigitalOut* B, I2C* M) : motorOn(false), dutyCycle(0.0)
{
     i2cMotor = M;
     inA = A;
     inB = B;
     
     // Instantiate TCA9554A
     motor = new TCA9554A(i2cMotor, TCA9554A::SLAVE_ADDRESS_38H);  // 38H->Port 0->RSV
     
     // Initialize RSV as output
     if (motor->configurePort(TCA9554A::PORT_0, TCA9554A::DIR_OUTPUT)!= TCA9554A::SUCCESS) {
         MSG("#AP1017: Error configuring TCA9554A port.\r\n");
     }
     motor->setPortLevel(TCA9554A::PORT_0, TCA9554A::LOW);  // Turn motor off
}

// Default destructor
AP1017::~AP1017(void)
{
    stop();
    delete inA, inB, motor, i2cMotor;
}

/*********************** Member Functions ***********************************/

AP1017::Status AP1017::setDirection(AP1017::Rotation dir)
{
    direction = dir;
    
    switch(direction){
        case DIRECTION_CW:                              // direction = 0x00
            if(isMotorOn())
            {
                MSG("#Error: Cannot change direction while motor is running.\r\n");
                return ERROR_MOTORON;
            }else
            {
                inA->write(1);
                inB->write(0);
                MSG("#Direction: CCW\r\n");
            }
            break;
        case DIRECTION_CCW:                                 // direction = 0x01
            if(isMotorOn())
            {
                MSG("#Error: Cannot change direction while motor is running.\r\n");
                return ERROR_MOTORON;
            }else
            {
                inA->write(0);
                inB->write(1);
                MSG("#Direction: CW\r\n");
            }
            break;
        case DIRECTION_BRAKE:                              // direction = 0x03
            inA->write(1);
            inB->write(1);
            MSG("#Direction: Brake\r\n");
            break;
        case DIRECTION_COAST:                              // direction = 0x04
            inA->write(0);
            inB->write(0);
            motorOn = false;
            MSG("#Direction: Coast\r\n");
            break;
        default:
            return ERROR_DIRECTION;
    }

    return SUCCESS;
}


AP1017::Rotation AP1017::getDirection(void)
{
    MSG("#Direction: ");
    switch(direction){
        case DIRECTION_CW:
            MSG("CW\r\n");
            break;
        case DIRECTION_CCW:
            MSG("CCW\r\n");
            break;
        case DIRECTION_COAST:
            MSG("Coast\r\n");
            break;
        case DIRECTION_BRAKE:
            MSG("Brake\r\n");
            break;
        default:
            MSG("Error: Invalid direction\r\n");
    }
    return direction;
}


AP1017::Status AP1017::setSpeed(double dc)
{
    if((dc <= 100.0) && (dc >= 0.0))
    {
        dutyCycle = dc/100.0;

        if(motorOn == true){
            MSG("Speed setting not yet implemented.\r\n");
            //MSG("#Changed running motor speed: %.1f%%.\r\n", dc);
        }
    }
    else
    {
        dutyCycle = 0.0;
        return ERROR_DUTY_CYCLE;
    }

    return SUCCESS;
}


double AP1017::getSpeed(void)
{
    MSG("Speed: %.1f%%\r\n.", dutyCycle*100.0);
    return dutyCycle*100.0;
}


AP1017::Status AP1017::start(void)
{
    motor->setPortLevel(TCA9554A::PORT_0, TCA9554A::HIGH);        // set RSV high
    motorOn = true;                         // Set ON flag

    return SUCCESS;
}


AP1017::Status AP1017::stop(void)
{
    motor->setPortLevel(TCA9554A::PORT_0, TCA9554A::LOW);        // set RSV low
    motorOn = false;                        // Set OFF flag

    return SUCCESS;
}

AP1017::Status  AP1017::brake(void)
{
    setDirection(DIRECTION_BRAKE);
    return SUCCESS;
}

AP1017::Status  AP1017::coast(void)
{
    setDirection(DIRECTION_COAST);
    return SUCCESS;
}

bool AP1017::isMotorOn(void)
{
    return motorOn;
}