#ifndef __AK9752_H__
#define __AK9752_H__

#include "mbed.h"

/**
 * This is a device driver of AK9752.
 *
 * @note AK9752 IR sensor device manufactured by AKM.
 * Example:
 * @code
 * #include "mbed.h"
 * #include "ak9750.h"
 * 
 * #define I2C_SPEED_100KHZ    100000
 * #define I2C_SPEED_400KHZ    400000
 * 
 * int main() {
 *     // Creates an instance of I2C
 *     I2C connection(I2C_SDA0, I2C_SCL0);
 *     connection.frequency(I2C_SPEED_100KHZ);
 *
 *     // TBD
 *
 *     while(true) {
 *              // TBD    
 *         }
 *     }
 * }
 * @endcode
*/
class AK9752
{
public:
    /**
     * Enum for return status.
     */
    typedef enum {
        SUCCESS,                 /**< Success */
        ERROR,                   /**< Error */
        ERROR_I2C_WRITE,         /**< I2C write error */
        ERROR_I2C_READ,          /**< I2C read error */
        ERROR_ARG_OUT_OF_BOUNDS, /**< An argument is out of bounds */
    } Status;
    
    /**
     * Enum for slave address of AK9752.
     */
    typedef enum {
        SLAVE_ADDR_1 = 0x64,   /**< CAD1=0, CAD0=0 */
    } SlaveAddress;
    
    /**
     * Enum for operation mode of measurement.
     */
    typedef enum {
        MODE_STANDBY = 0x00,
        MODE_CONTINUOUS = 0x01,
        MODE_SINGLE_SHOT = 0x02,
    } OperationMode;

    /**
     * Enum for Cut-off frequency for tempearature sensor setting.
     */    
    typedef enum {
        FCTMP_NOFILTER = 0x00,      /**< No Filter */
        FCTMP_2P5HZ = 0x01,         /**< Fc = 2.5 Hz */
        FCTMP_0P9HZ = 0x02,         /**< Fc = 0.9 Hz */
        FCTMP_0P45HZ = 0x03,        /**< Fc = 0.45 Hz */
        FCTMP_0P22HZ = 0x04,        /**< Fc = 0.22 Hz */
    } FcTmp;
    
    /**
     * Enum for Cut-off frequency for IR sensor setting.
     */    
    typedef enum {
        FCIR_NOFILTER = 0x00,      /**< No Filter */
        FCIR_2P5HZ = 0x01,         /**< Fc = 2.5 Hz */
        FCIR_0P9HZ = 0x02,         /**< Fc = 0.9 Hz */
        FCIR_0P45HZ = 0x03,        /**< Fc = 0.45 Hz */
    } FcIr;
    
    /**
     * Structure for intterupt status.
     */
    typedef struct {
        bool irh;
        bool irl;
        bool tmph;
        bool tmpl;
        bool dr;
    } InterruptStatus;

    /**
     * Structure for threshold.
     */    
    typedef struct {
        int16_t thirh;        /**< High Threshold level of IR. */
        int16_t thirl;        /**< Low Threshold level of IR. */
        int16_t thtmph;       /**< High Threshold level of temperature sensor. */
        int16_t thtmpl;       /**< Low Threshold level of temperature sensor. */
    } Threshold;
    
    /**
     * Structure for measurement data.
     */
    typedef struct {
        InterruptStatus intStatus;
        int16_t ir;
        int16_t temperature;
        bool dor;
    } SensorData;
    
    /**
     * Constructor.
     *
     * @param conn instance of I2C
     * @param addr slave address of the device
     */
//    AK9752(I2C *conn, SlaveAddress addr);
    AK9752();

    void init(I2C *conn, SlaveAddress addr);

    /**
     * Checks AK9752 connection.
     * @return Returns SUCCESS if connection is confirmed, otherwise returns other value.
     */
    Status checkConnection();

    /**
     * Gets interrupt enable/disable status.
     * @param intStatus interrupt status
     * @return SUCCESS if the interrupt status is obtained successfully, otherwise returns other value.
     */
    Status getInterruptEnable(InterruptStatus *intStatus);
       
    /**
     * Sets interrupt enable/disable status.
     * @param intStatus interrupt status
     * @return SUCCESS if the interrupt status is set successfully, otherwise returns other value.
     */
    Status setInterruptEnable(const InterruptStatus *intStatus);
    
    /**
     * Gets sensor operation mode.
     * @param mode Pointer to the operation mode.
     * @param fc_tmp Pointer to the tempearture sensor filter setting.
     * @param fc_ir Pointer to the IR sensor filter setting.
     * @return SUCCESS if operation mode is set successfully, otherwise returns other value.
     */
    Status getOperationMode(OperationMode *mode, FcTmp *fc_tmp, FcIr *fc_ir);
       
    /**
     * Sets sensor operation mode.
     * @param mode operation mode to be set.
     * @param fc_tmp filter cut-off frequency setting for temperature sensor
     * @param fc_ir filter cut-off frequency setting for ir sensor
     * @return SUCCESS if operation mode is set successfully, otherwise returns other value.
     */
    Status setOperationMode(OperationMode mode, FcTmp fc_tmp = FCTMP_NOFILTER, FcIr fc_ir = FCIR_NOFILTER);
       
    /**
     * Sets threshold.
     * @param th Pointer to the threshold structure to be set.
     * @return SUCCESS if threshold is set successfully, otherwise returns other value.
     */
    Status setThreshold(const Threshold *th);
    
    /**
     * Gets threshold.
     * @param th Pointer to the threshold structure to store the read data.
     * @return SUCCESS if threshold is read successfully, otherwise returns other value.
     */
    Status getThreshold(Threshold *th);
    
    /**
     * Resets the device.
     * @return SUCCESS if the device is reset successfully, otherwise returns other value.
     */
    Status reset();
    
    /**
     * Gets sensor data.
     * @param data Pointer to the SensorData structure object to store the read data.
     * @return SUCCESS if data is obtained successfully, otherwise returns other value.
     */
    Status getSensorData(SensorData *data);
    
    /**
     * Reads register(s).
     * @param registerAddress Register address to be read.
     * @param buf Buffer to store the read data.
     * @param length Length in bytes to be read.
     * @return SUCCESS if data is read successfully, otherwise returns other value.
     */
    Status read(char registerAddress, char *buf, int length);
    
    /**
     * Writes data into register(s).
     * @param registerAddress Register address to be written.
     * @param buf Data to be written.
     * @param length Length in bytes to be written.
     * @return SUCCESS if data is written successfully, otherwise returns other value.
     */
    Status write(char registerAddress, const char *buf, int length);
    
private:
    /**
     * Holds a pointer to an I2C object. 
     */
    I2C *connection;
    
    /**
     * Holds the slave address. 
     */
    SlaveAddress slaveAddress;

//    const static uint8_t IR_DATA_LEN = 7;    /**<! Data length of IR sensor. From ST1 to ST2. */
    
    /**
     * Gets sensor measurement data, from ST1 to ST2.
     * @param buf Buffer to store the read data.
     * @return SUCCESS if data is read successfully, otherwise returns other value.
     */
    Status getData(char *buf);
};

#endif // __AK9752_H__