#ifndef SLCAN_H_INCLUDED
#define SLCAN_H_INCLUDED

#include <mbed.h>
#include <USBSerial.h>

class SLCANBase {
public:
    bool update();

protected:
    SLCANBase();
    virtual ~SLCANBase();
    
    // To be implemented by subclasses
    virtual bool setBaudrate(int baudrate) = 0;
    virtual bool setMode(CAN::Mode mode) = 0;
    virtual bool transmitMessage(CANMessage& msg) = 0;
    
    virtual bool processCommands() = 0;
    virtual bool processCANMessages() = 0;
    virtual bool flush() = 0;
    
    virtual uint8_t getFirmwareVersion();
    virtual uint8_t getHardwareVersion();
    virtual const char* getSerialString();
    
    // Shared amongst subclasses
    static size_t formatCANMessage(const CANMessage& msg, char* buf, size_t max_len);
    static size_t formattedCANMessageLength(const CANMessage& msg);
    static size_t commandResponseLength(const char* command);
    bool execCommand(const char* command, char* response=NULL);
    
private:
    bool execConfigCommand(const char* command);
    bool execTransmitCommand(const char* command, char* response);
    bool execDiagnosticCommand(const char *command, char* response);
};

class USBSLCAN : public SLCANBase {
public:
    USBSLCAN(USBSerial& stream, CAN& can);
protected:
    virtual bool setBaudrate(int baudrate);
    virtual bool setMode(CAN::Mode mode);
    virtual bool transmitMessage(CANMessage& msg);
    
    virtual bool processCommands();
    virtual bool processCANMessages();
    virtual bool flush();
    
    bool getNextCANMessage(CANMessage& msg);
private:
    USBSerial& stream;
    CAN& can;
    CANMessage queuedMessage;
    bool messageQueued;
    bool commandQueued;
    bool commandOverflow;
    char inputCommandBuffer[32];
    char outputPacketBuffer[64];
    size_t inputCommandLen;
    size_t outputPacketLen;
};

class SerialSLCAN : public SLCANBase {
public:
    SerialSLCAN(Serial& stream, CAN& can);
protected:
    virtual bool setBaudrate(int baudrate);
    virtual bool setMode(CAN::Mode mode);
    virtual bool transmitMessage(CANMessage& msg);
    
    virtual bool processCommands();
    virtual bool processCANMessages();
    virtual bool flush();
    
    bool getNextCANMessage(CANMessage& msg);
private:
    Serial& stream;
    CAN& can;
    bool commandQueued;
    bool commandOverflow;
    char inputCommandBuffer[32];
    size_t inputCommandLen;
};

#endif