/**
 * @example websocket/example.cpp
 *
 * This is an example of using the wiconnect API to read data
 * from a webpage via WebSocket
 *
 * It works as follows:
 * 1. Instantiate the WiConnect Library
 * 2. Initiate Communication with WiFi Module
 * 3. Set the network credentials and other settings
 * 4. Open:
 *    http://wiconnect.ack.me/_media/2.1/http_server_ws_simple/http_server_ws_simple.html
 *    In your web browser
 *    This program detects your browser is connected and displays the data
 * 5. That's It!
 *
 *
 */


/******************************************************************************
 * Example Variables
 */

// This is the name of your WiFi network
// Look for this name in your WiFi settings
// (e.g. your phone's list of WiFi networks in the WiFi settings menu)
// tip: add double-quotes around SSID to add spaces to name
#define NETWORK_SSID "\"<YOUR NETWORK NAME HERE>\""


// This is the password of your WiFi network
// Leave as empty string (e.g "") to connect to OPEN network
#define NETWORK_PASSWORD "\"<YOUR NETWORK PASSWORD HERE>\""


/******************************************************************************
 * Includes
 */

// include C library headers
#include <stdio.h> // needed for printf

// include target specific defines
#include "target_config.h"
// include the Wiconnect Host Library API header
#include "Wiconnect.h"



/******************************************************************************
 * Global Defines
 */


// Serial used for printfs to terminal (i.e. NOT used for WiConnect)
static Serial consoleSerial(STDIO_UART_TX, STDIO_UART_RX);
uint8_t rxBuffer[128], txBuffer[128];


/******************************************************************************
 * Starting point of application
 */
int main(int argc, char **argv)
{
    WiconnectResult result;
    
    consoleSerial.baud(115200); // console terminal to 115200 baud

    //-------------------------------------------------------------------------
    // STEP 1: Instantiate WiConnect Library
    //-------------------------------------------------------------------------

    // Setup  wiconnect serial interface configuration
    // Here we only specify the rx buffer size and not rx buffer pointer, this means
    // The serial RX buffer will be dynamically allocated
    SerialConfig serialConfig(WICONNECT_RX_PIN, WICONNECT_TX_PIN, 256, NULL);

    // Instantiate the Wiconnect library
    // Here we only specify the buffer size and not buffer pointer, this means
    // The internal buffer will be dynamically allocated
    Wiconnect wiconnect(serialConfig, 256, NULL, WICONNECT_RESET_PIN);

    //-------------------------------------------------------------------------
    // STEP 2: Initiate Communication with WiFi Module
    //-------------------------------------------------------------------------

    printf("Initializing WiConnect Library...\r\n");

    // Initialize communication with WiFi module
    if(WICONNECT_FAILED(result, wiconnect.init(false)))
    {
        printf("Failed to initialize communication with WiFi module!\r\n"
                "Make sure the wires are connected correctly\r\n");
        for(;;); // infinite loop
    }


    //-------------------------------------------------------------------------
    // STEP 3: Set the network credentials
    // Note: We manually set the network parameters to ensure SDK backward compatibility
    //-------------------------------------------------------------------------

    printf("Setting network SSID: %s\r\n", NETWORK_SSID);
    if(WICONNECT_FAILED(result, wiconnect.setSetting("wlan.ssid", NETWORK_SSID)))
    {
        printf("Failed to set wlan.ssid setting\r\n");
        for(;;); // infinite loop
    }
    
    printf("Setting network password\r\n");
    if(WICONNECT_FAILED(result, wiconnect.setSetting("wlan.passkey", NETWORK_PASSWORD)))
    {
        printf("Failed to set wlan.passkey setting\r\n");
        for(;;); // infinite loop
    }

    printf("Enabling mDNS\r\n");
    if(WICONNECT_FAILED(result, wiconnect.setSetting("mdns.enabled", 1)))
    {
        printf("Failed to enable mDNS\r\n");
        for(;;); // infinite loop
    }

    printf("Setting mDNS name to: mymodule.local\r\n");
    if(WICONNECT_FAILED(result, wiconnect.setSetting("mdns.name", "mymodule.local")))
    {
        printf("Failed to set mDNS name\r\n");
        for(;;); // infinite loop
    }

    printf("Enabling the HTTP Server\r\n");
    if(WICONNECT_FAILED(result, wiconnect.setSetting("http.server.enabled", 1)))
    {
        printf("Failed to set HTTP server enabled\r\n");
        for(;;); // infinite loop
    }

    printf("Saving settings to Non-volatile Memory\r\n");
    if(WICONNECT_FAILED(result, wiconnect.saveSettings()))
    {
        printf("Failed save settings\r\n");
        for(;;); // infinite loop
    }
    
    printf("Brining the network up\r\n");
    if(WICONNECT_FAILED(result, wiconnect.join()))
    {
        printf("Failed to join network: %s\r\n", NETWORK_SSID);
        for(;;); // infinite loop
    }

    //-------------------------------------------------------------------------
    // STEP 4: Wait for a websocket to connect
    //-------------------------------------------------------------------------

    printf("Using your web browser, go to:\r\n");
    printf("   http://wiconnect.ack.me/_media/2.1/http_server_ws_simple/http_server_ws_simple.html\r\n");
    printf("and click the 'Open WebSocket' button.\r\n");
    printf("Once you do that you'll be able to stream data between the webpage and host micro.\r\n");

    WiconnectSocket websocket(sizeof(rxBuffer), rxBuffer, sizeof(txBuffer), txBuffer);
    
    // wait for a client to connect
    while(true)
    {
        printf("\r\nWaiting for websockets to connect...\r\n");
        if(WICONNECT_FAILED(result, wiconnect.httpAcceptWebSocket(websocket)))
        {
            printf("Failed to accept client websocket: %d\r\n", result);
            continue;
        }

        printf("Websocket connected!\r\n");
        printf("Address: %s:%d\r\n", websocket.getHost(), websocket.getRemotePort());
        printf("Anything typed into the console will be set to the webpage and visa versa\r\n");

        while(true)
        {
            bool dataAvailable;
            websocket.poll(&dataAvailable, true);

            if(!websocket.isConnected())
            {
                printf("Websocket disconnect!\r\n");
                break;
            }

            if(dataAvailable)
            {
                uint8_t *buffer;
                uint16_t bytesRead;
                websocket.read(&buffer, &bytesRead);
                buffer[bytesRead] = 0;
                printf("%s\r\n", buffer);
            }

            while(consoleSerial.readable())
            {
                websocket.putc(consoleSerial.getc(), false);
            }
            // flush the buffer
            websocket.flushTxBuffer();
        }
    }

    //-------------------------------------------------------------------------
    // STEP 5: Done!
    //-------------------------------------------------------------------------

    while(true){} // infinite loop
}

