/**
 * @example web_setup/example.cpp
 *
 * This is an example of using the web setup network API to
 * configure the module network credentials remotely.
 *
 * It works as follows:
 * 1. Instantiate the WiConnect Library
 * 2. Initiate Communication with WiFi Module
 * 3. Start WebSetup with specified parameters
 * 4. Wait for web setup webpage to complete
 * 5. That's it!
 *
 *
 */


/******************************************************************************
 * Example Variables
 */

// This is the name of the WebSetup network
// Look for this name in your WiFi settings
// (e.g. your phone's list of WiFi networks in the WiFi settings menu)
// tip: add double-quotes around SSID to add spaces to name
#define WEB_SETUP_SSID "\"WiConnect WebSetup Example\""

// This is the password for the WebSetup network
// Leave as empty string (e.g "") to create OPEN network
#define WEB_SETUP_PASSWORD "password"




/******************************************************************************
 * Includes
 */

// include C library headers
#include <stdio.h> // needed for printf

// include target specific defines
#include "target_config.h"
// include the Wiconnect Host Library API header
#include "Wiconnect.h"




/******************************************************************************
 * Global Defines
 */


// Serial used for printfs to terminal (i.e. NOT used for WiConnect)
static Serial consoleSerial(STDIO_UART_TX, STDIO_UART_RX);

// Buffer used internally by WiConnect library, note that this is optional
static uint8_t wiconnectInternalBuffer[256];



/******************************************************************************
 * Starting point of application
 */
int main(int argc, char **argv)
{
    WiconnectResult result;
    consoleSerial.baud(115200); // console terminal to 115200 baud

    //-------------------------------------------------------------------------
    // STEP 1: Instantiate WiConnect Library
    //-------------------------------------------------------------------------

    // Setup  wiconnect serial interface configuration
    // Here we only specify the rx buffer size and not rx buffer pointer, this means
    // The serial RX buffer will be dynamically allocated
    SerialConfig serialConfig(WICONNECT_RX_PIN, WICONNECT_TX_PIN, 256, NULL);

    // Instantiate the Wiconnect library
    // Here we specify the buffer size AND buffer pointer, this means we're using static allocation
    Wiconnect wiconnect(serialConfig, sizeof(wiconnectInternalBuffer), wiconnectInternalBuffer, WICONNECT_RESET_PIN);


    //-------------------------------------------------------------------------
    // STEP 2: Initiate Communication with WiFi Module
    //-------------------------------------------------------------------------

    printf("Initializing WiConnect Library...\r\n");

    // Initialize communication with WiFi module
    if(WICONNECT_FAILED(result, wiconnect.init()))
    {
        if(result == WICONNECT_FIRMWARE_OUTDATED)
        {
            printf("*** The WiFi firmware is not supported. Run the ota example to update the firmware:\r\n");
            printf("https://developer.mbed.org/teams/ACKme/code/wiconnect-ota_example\r\n\r\n");
        }
        else
        {
            printf("Failed to initialize communication with WiFi module!\r\n"
                   "Make sure the wires are connected correctly\r\n");
        }
        for(;;); // infinite loop
    }


    //-------------------------------------------------------------------------
    // STEP 3: Start WebSetup with specified parameters
    //-------------------------------------------------------------------------

    printf("Starting Websetup...\r\n");

    // Start web setup
    if(wiconnect.startWebSetup(WEB_SETUP_SSID, WEB_SETUP_PASSWORD) != WICONNECT_SUCCESS)
    {
        printf("Failed to start web setup\r\n");
        for(;;); // infinite loop
    }

    //-------------------------------------------------------------------------
    // STEP 4: Wait for web setup webpage to complete
    //-------------------------------------------------------------------------

    printf("Web setup has started.\r\n\r\n");
    printf("1. Using your phone (or PC, Mac, Linux, etc.)\r\n   connect to the WiFi network: %s\r\n", WEB_SETUP_SSID);
    printf("2. The password is: %s\r\n", WEB_SETUP_PASSWORD);
    printf("3. Once connected, open your browser and enter the URL: http://setup.com\r\n");
    printf("4. This will bringup a setup page, enter your router's credentials.\r\n");
    printf("5. Click the 'Save & Exit' button at the bottom of the webpage\r\n\r\n");

    for(;;)
    {
        bool isRunning;
        
        if(wiconnect.isWebSetupRunning(&isRunning) != WICONNECT_SUCCESS)
        {
            printf("Failed to get websetup status\r\n");
            for(;;);
        }
        delayMs(100);
        if(!isRunning)
            break;
    }

    //-------------------------------------------------------------------------
    // STEP 5: Done!
    //-------------------------------------------------------------------------

    printf("IP Address: %s\r\n", wiconnect.getIpAddress());
    printf("Web setup example has completed!\r\n");

    while(true){} // infinite loop
}


