/**
 * ACKme WiConnect Host Library is licensed under the BSD licence: 
 * 
 * Copyright (c)2014 ACKme Networks.
 * All rights reserved. 
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met: 
 * 
 * 1. Redistributions of source code must retain the above copyright notice, 
 * this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 * this list of conditions and the following disclaimer in the documentation 
 * and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products 
 * derived from this software without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
 * OF SUCH DAMAGE.
 */

#include "tests/Tests.h"
#include "Wiconnect.h"



static void printScanResults(const ScanResultList &scanResultList);




/*************************************************************************************************/
WiconnectResult networkScanCommand(int argc, char **argv)
{
    WiconnectResult result = WICONNECT_BAD_ARG;
    Wiconnect *wiconnect = Wiconnect::getInstance();
    ScanResultList scanResultList(TEST_BUFFER_LENGTH, TEST_BUFFER);
    uint8_t channelList[16];
    const char* ssid = NULL;
    uint8_t *channelListPtr = NULL;

    if(argc > 0)
    {
        if(strcmp(argv[0], "all") != 0)
        {
            char *ch_tok;
            char *chListPtr = argv[0];
            uint8_t ch_count = 0;

            while((ch_tok = strtok(chListPtr, ","))  != NULL)
            {
                intmax_t x;

                if(!StringUtil::parseInt(ch_tok, &x, 1, 14))
                {
                    return WICONNECT_BAD_ARG;
                }
                channelList[ch_count++] = (uint8_t)x;
                chListPtr = NULL;
            }
            channelList[ch_count] = 0;
            channelListPtr = channelList;
        }
        --argc;
        ++argv;
    }

    if(argc > 0)
    {
        ssid = argv[0];
    }

    if(!WICONNECT_FAILED(result, wiconnect->scan(scanResultList, channelListPtr, ssid)))
    {
        printScanResults(scanResultList);
    }

    return result;
}



/*************************************************************************************************/
static void printScanResults(const ScanResultList &scanResultList)
{
    SsidStrBuffer ssidBuffer;
    MacAddressStrBuffer macBuffer;
    char rateBuffer[16];
    int i = 1;

    LOG_INFO("Scan result count: %d", scanResultList.getCount());
    for(const ScanResult *res = scanResultList.getListHead(); res != NULL; res = res->getNext(), ++i)
    {
        LOG_INFO("------------------------\r\n"
                "%d: %s\r\n"
                 "\tChannel: %d\r\n"
                 "\tSignal: %s\r\n"
                 "\tSecurity: %s\r\n"
                 "\tRate: %s\r\n"
                 "\tBSSID: %s",
                 i, Wiconnect::ssidToStr(res->getSsid(), ssidBuffer),
                 res->getChannel(),
                 Wiconnect::signalStrengthToStr(res->getSignalStrength()),
                 Wiconnect::networkSecurityToStr(res->getSecurityType()),
                 res->getRateStr(rateBuffer),
                 Wiconnect::macAddressToStr(res->getMacAddress(), macBuffer));
    }

}
