/**
 * ACKme WiConnect Host Library is licensed under the BSD licence: 
 * 
 * Copyright (c)2014 ACKme Networks.
 * All rights reserved. 
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met: 
 * 
 * 1. Redistributions of source code must retain the above copyright notice, 
 * this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 * this list of conditions and the following disclaimer in the documentation 
 * and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products 
 * derived from this software without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
 * OF SUCH DAMAGE.
 */
#pragma once

#include <ctype.h>

#include "Wiconnect.h"
#include "Console.h"
#include "StringUtil.h"
#include "util/log/log.h"



#define ADD_HEADER(header) { header, NULL, NULL, NULL}
#define ADD_CMD(key, func, desc, ext) { key, func ## Command, desc, ext }
#define CMD_LIST_TERMINATOR { NULL, NULL, NULL, NULL }
#define CMD_HELP_ENTRY { "?", NULL, "Print list of commands. Add '-v' option to print verbosely", NULL }, \
                       { "help", NULL, "Print list of commands. Add '-v' option to print verbosely", NULL }

typedef WiconnectResult (*CommandProcessorFunc)(int, char**);

typedef struct
{
    const char *key;
    CommandProcessorFunc func;
    const char *desc;
    const char *extendedDesc;
} CommandListEntry;


/*************************************************************************************************/
class Command
{
public:
    Command()
    {
        argc = -1;
        func = NULL;
    }

    void init(int argc_, CommandProcessorFunc func_)
    {
        argc = argc_;
        func = func_;
    }

    char** getArgvBuffer()
    {
        return argv;
    }

    WiconnectResult execute()
    {
        return func(argc, argv);
    }

private:
    int argc;
    char *argv[DEFAULT_COMMAND_MAX_ARGV];
    CommandProcessorFunc func;
};



/*************************************************************************************************/
class CommandProcessor
{
private:
    const CommandListEntry *commandList;
    Console console;
    ConsoleSerial *serial;
    int commandListSize;

public:
    /*************************************************************************************************/
    CommandProcessor(ConsoleSerial *serial_, const CommandListEntry *commandList_) :
        commandList(commandList_), console(serial_), serial(serial_), commandListSize(0)
    {
        for(const CommandListEntry *cmd = commandList; cmd->key != NULL; ++cmd)
        {
            ++commandListSize;
        }
    }

    /*************************************************************************************************/
    ~CommandProcessor()
    {
    }

    /*************************************************************************************************/
    void waitForCommand(Command *cmdPtr)
    {
        for(;;)
        {
            char *line;
            char **argv;
            intmax_t index;
            int lineLength;
            const CommandListEntry *foundCmd = NULL;

            console.readLine(&line, &lineLength);

            if(lineLength == 0)
            {
                continue;
            }

            argv = cmdPtr->getArgvBuffer();
            int argc = parseArgs(line, DEFAULT_COMMAND_MAX_ARGV, argv);
            if(argc == -1)
            {
                LOG_ERROR("Failed to parse commandline");
                continue;
            }

            if(argv[0][0] == '?' || strcmp(argv[0], "help") == 0)
            {
                bool verbose = (argc > 1 && argv[1][0] == '-' && argv[1][1] == 'v');
                printHelp(verbose);
                continue;
            }

            for(const CommandListEntry *cmd = commandList; cmd->key != NULL; ++cmd)
            {
                if(cmd->desc != NULL && strcmp(cmd->key, argv[0]) == 0)
                {
                    foundCmd = cmd;
                    break;
                }
            }

            if(foundCmd == NULL)
            {
                if(StringUtil::parseInt(argv[0], &index, 0, commandListSize))
                {
                    foundCmd = &commandList[(int)index];
                }
                else
                {
                    LOG_ERROR("Unknown command. Enter: 'help' to list available commands");
                    continue;
                }
            }

            --argc;
            memmove(argv, &argv[1], sizeof(char*)*argc);

            if(argc == 1 && (argv[0][0] == '?' || (strstr(argv[0], "help") != NULL)))
            {
                printCommandHelp(foundCmd, true, -1);
                continue;
            }
            else
            {
                cmdPtr->init(argc, foundCmd->func);
            }
            break;
        }
    }

protected:

    /*************************************************************************************************/
    void printHelp(bool verbose)
    {
        int i = 0;
        for(const CommandListEntry *cmd = commandList; cmd->key != NULL; ++cmd, ++i)
        {
            if(cmd->desc == NULL)
            {
                --i;
                serial->printf("\r\n--------------------------------\r\n"
                        "%s\r\n", cmd->key);
                continue;
            }
            printCommandHelp(cmd, verbose, i);
        }
    }


    /*************************************************************************************************/
    void printCommandHelp(const CommandListEntry *cmd, bool verbose, int i)
    {
        if(i != -1)
            serial->printf("%2d: %10s : %s\r\n", i, cmd->key, cmd->desc);
        else
            serial->printf("    %10s : %s\r\n", cmd->key, cmd->desc);

        if(verbose)
        {
            if(cmd->extendedDesc != NULL)
            {
                const char *newline, *ptr = cmd->extendedDesc;

                print_extended_help:
                paddHelpSpaces();
                newline = strchr(ptr, '\n');
                if(newline == NULL)
                {
                    puts(ptr);
                    return;
                }
                else
                {
                    while(ptr < newline)
                        serial->write(*ptr++);
                    serial->write('\r');
                    serial->write('\n');
                    ++ptr;
                    goto print_extended_help;
                }
            }
        }
    }

    /*************************************************************************************************/
    void paddHelpSpaces()
    {
        int spaces = 17;
        while(spaces--)
            serial->write(' ');
    }


    /*************************************************************************************************/
    int parseArgs(char *line, int max_argc, char **argv)
    {
        // TODO: \x00 style escaping
        char *p;
        unsigned char c;
        int argc = 0;
        char *tokenstart = NULL;
        enum states
        {
            INITIAL,
            WORD,
            STRING
        } state = INITIAL;

        for (p = line; *p != '\0'; p++)
        {
            c = (unsigned char) * p;
            /* One less because word at end-of-line case increments again */
            if (argc >= max_argc - 1)
            {
                return argc;
            }
            switch (state)
            {
            case INITIAL:
                if (isspace(c))
                {
                    continue;
                }
                if (c == '"')
                {
                    state = STRING;
                    tokenstart = p + 1;
                    continue;
                }
                tokenstart = p;
                state = WORD;
                continue;
            case STRING:
                if (c == '"')
                {
                    state = INITIAL;
                    *p = 0;
                    argv[argc++] = tokenstart;
                }
                continue;
            case WORD:
                if (c == '"')
                {
                    state = STRING;
                    *p = 0;
                    argv[argc++] = tokenstart;
                    tokenstart = p + 1;
                }
                else if (isspace(c))
                {
                    state = INITIAL;
                    *p = 0;
                    argv[argc++] = tokenstart;
                }
                continue;
            }
        }
        if (state == WORD)
        {
            *p = 0;
            argv[argc++] = tokenstart;
            argv[argc] = NULL;
        }
        else if (state == STRING)
        {
            argc = -1; /* Unterminated string */
        }
        else
        {
            argv[argc] = NULL;
        }

        return argc;
    }

};
