/**
 * ACKme WiConnect Host Library is licensed under the BSD licence: 
 * 
 * Copyright (c)2014 ACKme Networks.
 * All rights reserved. 
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met: 
 * 
 * 1. Redistributions of source code must retain the above copyright notice, 
 * this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 * this list of conditions and the following disclaimer in the documentation 
 * and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products 
 * derived from this software without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
 * OF SUCH DAMAGE.
 */
#include "Wiconnect.h"
#include "util/log/log.h"
#include "util/CommandProcessor/CommandProcessor.h"
#include "tests/Tests.h"
#include "target_config.h"


#if TEST_NONBLOCKING_API
#error 'TEST_NONBLOCKING_API = true' NOT currently supported for the mbed sdk
#endif


int wiconnectLogDebug(const char *str);
int wiconnectLogAssert(const char *str);


static const CommandListEntry commandList[] =
{
    CMD_HELP_ENTRY,
    WICONNECT_TEST_CMD_LIST,
    NETWORK_TEST_CMD_LIST,
    SOCKET_TEST_CMD_LIST,
    FILE_TEST_CMD_LIST,
    CMD_LIST_TERMINATOR
};

char testBuffer[TEST_BUFFER_LENGTH];
static uint8_t wiconnectInternalBuffer[WICONNECT_INTERNAL_BUFFER_SIZE];
static uint8_t wiconnectSerialRxBuffer[WICONNECT_SERIAL_RX_BUFFER_SIZE];

static const SerialConfig serialConfig(WICONNECT_RX_PIN, WICONNECT_TX_PIN, WICONNECT_CTS_PIN, WICONNECT_RTS_PIN, WICONNECT_DEFAULT_BAUD, WICONNECT_SERIAL_RX_BUFFER_SIZE, wiconnectSerialRxBuffer);
static Wiconnect wiconnectIfc(serialConfig, WICONNECT_INTERNAL_BUFFER_SIZE, wiconnectInternalBuffer, WICONNECT_RESET_PIN, WICONNECT_WAKE_PIN, TEST_NONBLOCKING_API);

ConsoleSerial consoleSerial(SERIAL_TX, SERIAL_RX);
CommandProcessor cmdProcessor(&consoleSerial, commandList);



/*************************************************************************************************/
int main(int argc, char **argv)
{
    WiconnectResult result = WICONNECT_SUCCESS;
    consoleSerial.setBaud(CONSOLE_BAUD);

    wiconnectIfc.setAssertLogger(LogFunc(wiconnectLogAssert));
    wiconnectIfc.setCommandDefaultTimeout(5000);
#ifdef ENABLE_WICONNECT_DEBUG
    wiconnectIfc.setDebugLogger(LogFunc(wiconnectLogDebug));
#endif

    initialize_loop:
    {
        LOG_INFO("\r\n\r\nInitializing WiConnect...");
        if(WICONNECT_FAILED(result, wiconnectIfc.init(true)))
        {
            LOG_WICONNECT_ERROR(result, "Failed to initialize Wiconnect");
            LOG_INFO("Press any key to retry initialization...");
            int c = consoleSerial.getc();
            goto initialize_loop;
        }
    }

    {
        if(!WICONNECT_FAILED(result, wiconnectIfc.getVersion()))
        {
            LOG_INFO("Version: %s", wiconnectIfc.getResponseBuffer());
        }
    }

    LOG_INFO("WiConnect test app ready...");

    for(;;)
    {
        WiconnectResult result;
        Command cmd;
        cmdProcessor.waitForCommand(&cmd);
        if(WICONNECT_FAILED(result, cmd.execute()))
        {
            LOG_WICONNECT_ERROR(result, "Failed to execute command");
            if(result == WICONNECT_CMD_RESPONSE_ERROR)
            {
                LOG_ERROR("WiConnect command response code: %s", wiconnectIfc.getLastCommandResponseCodeStr());
            }
        }
    }

    return 0;
}


/*************************************************************************************************/
int wiconnectLogDebug(const char *str)
{
    logDebug(str);
    return 0;
}

/*************************************************************************************************/
int wiconnectLogAssert(const char *str)
{
    logWrite("\r\n\r\n[ASSERT] ", 13);
    logWrite(str, strlen(str));
    return 0;
}
