/**
 * ACKme WiConnect Host Library is licensed under the BSD licence: 
 * 
 * Copyright (c)2014 ACKme Networks.
 * All rights reserved. 
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met: 
 * 
 * 1. Redistributions of source code must retain the above copyright notice, 
 * this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 * this list of conditions and the following disclaimer in the documentation 
 * and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products 
 * derived from this software without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
 * OF SUCH DAMAGE.
 */
#pragma once

#include <string.h>

#include "ConsoleSerial.h"
#include "target_config.h"


typedef struct
{
    int index;
    int count;
    const char **choices;
    char cmd_buffer[32];
    char *cmd_buffer_ptr;
    char cmp_offset;
} console_tab_info_t;

typedef struct context
{
    char line[DEFAULT_COMMAND_LINE_LENGTH_MAX];
    int line_length;
    //int max_line_length;
    volatile int cursor_position;
    char history[DEFAULT_COMMAND_MAX_HISTORY][DEFAULT_COMMAND_LINE_LENGTH_MAX];
    int history_count;
    int history_index;
    //int hisory_length;
    char CR_char;
    console_tab_info_t tab_info;
} console_context_t;


class Console
{
private:
    ConsoleSerial *serial;
    console_context_t context;
public:

    /*************************************************************************************************/
    Console(ConsoleSerial *serial_) : serial(serial_)
    {
        memset((void*)&context, 0, sizeof(console_context_t));
    }

    /*************************************************************************************************/
    ~Console()
    {
    }

    /*************************************************************************************************/
    int readLine(char **line, int *lineLength)
    {
        bool in_escape_sequence = false;
        context.line[0] = 0;
        context.line_length = 0;
        context.cursor_position = 0;
        context.history_index = -1;

        serial->write(DEFAULT_CMD_PROMPT_STR);

        while(context.cursor_position < (DEFAULT_COMMAND_LINE_LENGTH_MAX-1))
        {
            int c;

            c = serial->read();
            if(c == -1)
            {
                continue;
            }

            if(c == '\r' || c == '\n')
            {
                context.CR_char = c;
            }

            if(in_escape_sequence)
            {
                in_escape_sequence = process_escape_sequence(c, true);
            }
            else
            {
                switch((unsigned char)c)
                {
                case '\t' : // tab
                    //process_tab();
                    break;
                case '\r': // newline
                case '\n': // linefeed
                    if(context.CR_char == c)
                    {
                        if(c == '\r')
                        {
                            // retrieve \n as well
                            serial->read();
                        }
                        serial->write('\r');
                        serial->write('\n');;
                        goto console_readline_finish;
                    }
                    break;
                case 27: // escape character
                case 0xE0:
                    in_escape_sequence = true;
                    break;
                case '\b':  // backspace
                case '\x7F':// delete
                    do_backspace(true);
                    break;
                default: {
                    if((c >= 32) && (c < 127))
                    {
                        add_char(c, true);
                    }
                } break;
                }
                *lineLength = context.line_length;
            }

//            if(c != '\t')
//            {
//                clear_tab_choices();
//            }

        }

        return -1;

        console_readline_finish:
        {
            *line = context.line;
            *lineLength = context.line_length;

            if(context.line_length > 0)
            {
                update_history(context.line);
            }
        }

        return 0;
    }

protected:

    /*************************************************************************************************/
    bool process_escape_sequence(char c, bool echo)
    {
        static char previous_char = 0;
        bool still_in_esc_seq = false;

    #ifdef __WIN32__
    #define UP_ARROW 'H'
    //#define RIGHT_ARROW 'M'
    #define DOWN_ARROW 'P'
    //#define LEFT_ARROW 'K'
    //#define HOME 'G'
    //#define END 'O'
    #else
    #define UP_ARROW 'A'
    #define RIGHT_ARROW 'C'
    #define DOWN_ARROW 'B'
    #define LEFT_ARROW 'D'
    #define HOME 'H'
    #define END 'F'
    #endif

        switch((unsigned char)c)
        {
    #ifndef __WIN32__
        case 'O':
    #endif
        case ';':
        case '[':
        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
        case 0xE0:
            still_in_esc_seq = true;
            break;
        case UP_ARROW: /* up arrow */
            cycle_history(false);
            break;
        case DOWN_ARROW: /* down arrow */
            cycle_history(true);
            break;
    #ifdef RIGHT_ARROW
        case RIGHT_ARROW: /* right arrow */
            update_cursor_position(true);
            break;
    #endif
    #ifdef LEFT_ARROW
        case LEFT_ARROW: /* left arrow */
            update_cursor_position(false);
            break;
    #endif
    #ifdef END
        case END: /* end */
            move_cursor_position(true);
            break;
    #endif
    #ifdef HOME
        case HOME: /* home */
            move_cursor_position(false);
            break;
    #endif
        case '~':  {/* vt320 style control codes */
            switch(previous_char)
            {
            case '1': /* home */
                move_cursor_position(false);
                break;
            case '4': /* end */
                move_cursor_position(true);
                break;
            default:
                break;
            }
        } break;
        default:
            break;
        }

        previous_char = c;

        return still_in_esc_seq;
    }

    /*************************************************************************************************/
    void update_history(char *line)
    {
        int copy_start_index = DEFAULT_COMMAND_MAX_HISTORY - 2;

        for(int i = 0; i < context.history_count; ++i)
        {
            if(strcmp(context.history[i], line) == 0)
            {
                copy_start_index = i - 1;
                break;
            }
        }

        if(context.history_count < DEFAULT_COMMAND_MAX_HISTORY)
        {
            ++context.history_count;
        }

        for(int j = copy_start_index; j >= 0; --j)
        {
            strcpy(context.history[j+1], context.history[j]);
        }
        strcpy(context.history[0], line);
    }

    /*************************************************************************************************/
    void cycle_history(bool cycle_down)
    {
        if(cycle_down)
        {
            --context.history_index;

            if(context.history_index <= -1)
            {
                context.history_index = -1;
                clear_line();
                return;
            }
        }
        else
        {
            if(context.history_index == context.history_count-1)
            {
                return;
            }
            else
            {
                ++context.history_index;
            }
        }

        const char *line = context.history[context.history_index];

        clear_line();
        add_string(line);
    }

    /*************************************************************************************************/
    void do_backspace( bool echo )
    {
        if ( context.cursor_position > 0 )
        {
            --context.cursor_position;
            if(echo)
            {
                serial->write('\b');
            }
            remove_char(echo);
        }
    }

    /*************************************************************************************************/
    void add_string(const char *s)
    {
        while(*s != 0)
        {
            add_char(*s++, true);
        }
    }

    /*************************************************************************************************/
    void add_char(char c, bool echo)
    {
        /* move the end of the line out to make space */
        for (int i = context.line_length + 1; i > context.cursor_position; --i )
        {
            context.line[i] = context.line[i - 1];
        }

        /* insert the character */
        ++context.line_length;
        context.line[context.cursor_position] = c;

        if(echo)
        {
            /* print out the modified part of the ConsoleBuffer */
            serial->write(&context.line[context.cursor_position]);
        }

        /* move the cursor back to where it's supposed to be */
        ++context.cursor_position;
        if(echo)
        {
            for (int i = context.line_length; i > context.cursor_position; --i )
            {
                serial->write('\b');
            }
        }

        context.line[context.line_length] = 0;
    }

    /*************************************************************************************************/
    void remove_char(bool echo)
    {
        /* back the rest of the line up a character */
        for (int i = context.cursor_position; i < context.line_length; ++i )
        {
            context.line[i] = context.line[i + 1];
        }

        --context.line_length;

        if(echo)
        {
            /* print out the modified part of the ConsoleBuffer */
            serial->write( &context.line[context.cursor_position] );
            /* overwrite the extra character at the end */
            serial->write(" \b");


            /* move the cursor back to where it's supposed to be */
            for (int i = context.line_length; i > context.cursor_position; --i )
            {
                serial->write( '\b' );
            }
        }
    }

    /*************************************************************************************************/
    bool update_cursor_position(bool move_right)
    {
        bool updated = false;

        if(move_right)
        {
            if(context.cursor_position < context.line_length)
            {
                ++context.cursor_position;
                updated = true;
                serial->write( "\x1B[C");
            }
        }
        else
        {
            if(context.cursor_position > 0)
            {
                --context.cursor_position;
                updated = true;
                serial->write("\x1B[D");
            }
        }

        return updated;
    }

    /*************************************************************************************************/
    void clear_line()
    {
        int len = context.line_length - context.cursor_position;
        while(len-- > 0)
        {
            serial->write(' ');
        }
        len = context.line_length;
        while(len-- > 0)
        {
            serial->write("\b \b");
        }
        context.cursor_position = 0;
        context.line_length = 0;
        context.line[0] = 0;
    }

    /*************************************************************************************************/
    void move_cursor_position(bool far_right)
    {
        while(update_cursor_position(far_right))
        {
        }
    }

    /*************************************************************************************************/
    void clear_tab_choices(void)
    {
        if(context.tab_info.choices != NULL)
        {
            free(context.tab_info.choices);
            context.tab_info.choices = NULL;
        }
    }

};
