/**
 * ACKme WiConnect Host Library is licensed under the BSD licence: 
 * 
 * Copyright (c)2014 ACKme Networks.
 * All rights reserved. 
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met: 
 * 
 * 1. Redistributions of source code must retain the above copyright notice, 
 * this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 * this list of conditions and the following disclaimer in the documentation 
 * and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products 
 * derived from this software without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
 * OF SUCH DAMAGE.
 */
#include "tests/Tests.h"
#include "Wiconnect.h"


typedef struct
{
    uint32_t bytesRemaining;
} fileInfo_t;


static WiconnectResult fileReadCallback(void *user, void *data, int maxReadSize, int *bytesRead);





/*************************************************************************************************/
WiconnectResult fileCreateCommand(int argc, char **argv)
{
    WiconnectResult result;
    Wiconnect *wiconnect = Wiconnect::getInstance();
    const char *name;
    uint32_t size;
    uint32_t version = 0;
    uint32_t type = FILE_TYPE_ANY;
    const int savedTimeOut = wiconnect->getCommandDefaultTimeout();
    fileInfo_t fileInfo;


    if(argc < 2)
    {
        LOG_ERROR("must specify file name and size");
        return WICONNECT_BAD_ARG;
    }

    name = argv[0];
    if(!StringUtil::strToUint32(argv[1], &size))
    {
        LOG_ERROR("invalid file size");
        return WICONNECT_BAD_ARG;
    }

    if(argc > 2)
    {
        if(!Wiconnect::fileVersionStrToInt(argv[2], &version))
        {
            LOG_ERROR("invalid file version");
            return WICONNECT_BAD_ARG;
        }
    }
    if(argc > 3)
    {
        if(!StringUtil::strHexToUint32(argv[3], &type))
        {
            LOG_ERROR("invalid file type");
            return WICONNECT_BAD_ARG;
        }
    }

    fileInfo.bytesRemaining = size;
    wiconnect->setCommandDefaultTimeout(30000); // increase the timeout so the user can enter data

    if(!WICONNECT_FAILED(result, wiconnect->createFile(ReaderFunc(fileReadCallback), &fileInfo, name, size, version, (FileType)type)))
    {
        LOG_INFO("File created");
    }
    wiconnect->setCommandDefaultTimeout(savedTimeOut);

    return result;
}

/*************************************************************************************************/
static WiconnectResult fileReadCallback(void *user, void *data, int maxReadSize, int *bytesReadPtr)
{
    fileInfo_t *info = (fileInfo_t*)user;
    int bytesToRead = MIN(maxReadSize, info->bytesRemaining);

    if(info->bytesRemaining == 0)
    {
        LOG_INFO("All data written");
        *bytesReadPtr = EOF;
        return WICONNECT_SUCCESS;
    }

    LOG_INFO("Enter up to %d bytes (%d bytes remaining,\r\n       Issue $$$ terminate current write):", bytesToRead, info->bytesRemaining);

    uint8_t *ptr = (uint8_t*)data;
    int bytesRead = 0;
    int terminateCount = 0;

    while(bytesToRead > 0)
    {
        int c = consoleSerial.getc();
        consoleSerial.putc(c);
        terminateCount = (c == '$') ? terminateCount+1 : 0;
        if(terminateCount >= 3)
        {
            break;
        }
        *ptr++ = (uint8_t)c;
        ++bytesRead;
        --bytesToRead;
    }

    // remove '$$' from data if we terminated
    if(terminateCount == 3)
    {
        bytesRead -= 2;
    }

    info->bytesRemaining -= bytesRead;
    *bytesReadPtr = bytesRead;
    consoleSerial.write("\r\n");

    return WICONNECT_SUCCESS;
}
