/**
 * ACKme WiConnect Host Library is licensed under the BSD licence: 
 * 
 * Copyright (c)2014 ACKme Networks.
 * All rights reserved. 
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met: 
 * 
 * 1. Redistributions of source code must retain the above copyright notice, 
 * this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 * this list of conditions and the following disclaimer in the documentation 
 * and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products 
 * derived from this software without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
 * OF SUCH DAMAGE.
 */

#include "tests/Tests.h"
#include "Wiconnect.h"




static WiconnectResult processData(WiconnectSocket &socket, uint32_t count, uint32_t delay, uint32_t size);



/*************************************************************************************************/
WiconnectResult socketUdpClientEchoCommand(int argc, char **argv)
{
    WiconnectResult result;
    Wiconnect *wiconnect = Wiconnect::getInstance();
    WiconnectSocket socket(TEST_BUFFER_LENGTH/2, TEST_BUFFER, TEST_BUFFER_LENGTH/2, TEST_BUFFER + TEST_BUFFER_LENGTH/2);
    const char *host;
    uint32_t port;
    uint32_t delay = 5;
    uint32_t count = 1000;
    uint32_t size = 256;

    if(argc < 2)
    {
        LOG_ERROR("Must specify host and port");
        return WICONNECT_BAD_ARG;
    }
    host = argv[0];

    if(!StringUtil::strToUint32((const char*)argv[1], &port))
    {
        LOG_ERROR("Invalid port");
        return WICONNECT_BAD_ARG;
    }
    if(argc > 2 && !StringUtil::strToUint32((const char*)argv[2], &count))
    {
        LOG_ERROR("Invalid packet count");
        return WICONNECT_BAD_ARG;
    }
    if(argc > 3 && !StringUtil::strToUint32((const char*)argv[3], &delay))
    {
        LOG_ERROR("Invalid packet delay");
        return WICONNECT_BAD_ARG;
    }
    if(argc > 4 && (!StringUtil::strToUint32((const char*)argv[4], &size) || size > sizeof(testBuffer)/2))
    {
        LOG_ERROR("Invalid packet size");
        return WICONNECT_BAD_ARG;
    }

    if(WICONNECT_FAILED(result, wiconnect->udpConnect(socket, host, port)))
    {
        LOG_ERROR("Failed to connect to server");
    }
    else
    {
        result = processData(socket, count, delay, size);
    }

    return result;
}


/*************************************************************************************************/
static WiconnectResult processData(WiconnectSocket &socket, uint32_t count, uint32_t delay, uint32_t size)
{
    WiconnectResult result = WICONNECT_ERROR;
    uint8_t *txBuffer = socket.getTxBuffer();
    const int txBufferLen = socket.getTxBufferSize();
    uint8_t writeCounter = 32;
    uint8_t readCounter = 32;
    uint32_t bytesSent = 0;
    uint32_t bytesReceived = 0;

    LOG_INFO("\r\n------------------\r\n"
             "Starting UDP Echo Test:\r\n"
            "Packet count: %d\r\n"
            "Packet size: %d\r\n"
            "Packet delay: %d\r\n"
            "(Press any key to terminate)",
            count, size, delay);

    for(int i = 0; i < count || bytesReceived < bytesSent; ++i)
    {
        if(i < count)
        {
            uint8_t *ptr = txBuffer;
            int l = size;
            while(l--)
            {
                *ptr++ = writeCounter++;
                if(writeCounter == 128)
                    writeCounter = 32;
            }

            if(WICONNECT_FAILED(result, socket.write(size)))
            {
                break;
            }
            bytesSent += size;
            consoleSerial.putc('.');
        }

        if(consoleSerial.readable())
        {
            int c = consoleSerial.getc();
            break;
        }

        delayMs(delay);

        uint8_t *readPtr;
        uint16_t readLen;
        if(WICONNECT_FAILED(result, socket.read(&readPtr, &readLen)))
        {
            break;
        }
        bytesReceived += readLen;

        while(readLen--)
        {
            if(*readPtr != readCounter)
            {
                LOG_ERROR("Invalid: %02X != %02X", *readPtr, readCounter);
            }
            ++readPtr;
            ++readCounter;
            if(readCounter == 128)
                readCounter = 32;
        }
    }

    consoleSerial.write("\r\n");
    LOG_INFO("------------------\r\n"
             "UDP Echo Test Complete:\r\n"
            "Bytes sent: %d\r\n"
            "Bytes received: %d\r\n",
            bytesSent, bytesReceived);

    return result;
}
