/**
 * ACKme WiConnect Host Library is licensed under the BSD licence: 
 * 
 * Copyright (c)2014 ACKme Networks.
 * All rights reserved. 
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met: 
 * 
 * 1. Redistributions of source code must retain the above copyright notice, 
 * this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 * this list of conditions and the following disclaimer in the documentation 
 * and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products 
 * derived from this software without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
 * OF SUCH DAMAGE.
 */

#include "tests/Tests.h"
#include "Wiconnect.h"



static void printFileList(const FileList &fileList);



/*************************************************************************************************/
WiconnectResult fileListCommand(int argc, char **argv)
{
    WiconnectResult result;
    Wiconnect *wiconnect = Wiconnect::getInstance();
    FileList fileList(TEST_BUFFER_LENGTH, TEST_BUFFER);
    uint32_t type = FILE_TYPE_ANY;
    const char *name = NULL;
    uint32_t version = 0;

    while(argc > 0)
    {
        if(argv[0][0] != '-' || argc < 2)
        {
            LOG_ERROR("Invalid command line argument");
            return WICONNECT_BAD_ARG;
        }

        switch(argv[0][1])
        {
        case 'v': {
            if(!Wiconnect::fileVersionStrToInt(argv[1], &version))
            {
                LOG_ERROR("Invalid file version");
                return WICONNECT_BAD_ARG;
            }
        } break;
        case 'n':
            name = argv[1];
        break;
        case 't': {
            if(!StringUtil::strHexToUint32((const char*)argv[1], &type))
            {
                LOG_ERROR("Invalid file type");
                return WICONNECT_BAD_ARG;
            }
        } break;
        default:
            LOG_ERROR("Unknown command option: %c", argv[0][1]);
            return WICONNECT_BAD_ARG;
        }

        argc -= 2;
        argv += 2;
    }


    if(!WICONNECT_FAILED(result, wiconnect->listFiles(fileList, name, (FileType)type, version)))
    {
        printFileList(fileList);
    }

    return result;

}

/*************************************************************************************************/
static void printFileList(const FileList &fileList)
{
    int i = 1;

    LOG_INFO("File count: %d", fileList.getCount());
    for(const WiconnectFile *file = fileList.getListHead(); file != NULL; file = file->getNext(), ++i)
    {
        LOG_INFO("------------------------\r\n"
                 "%d: %s\r\n"
                "\tVersion: %s\r\n"
                "\tSize: %d\r\n"
                "\tType: %s\r\n"
                "\tFlags: %s\r\n",
                i, file->getName(),
                file->getVersionStr(),
                file->getSize(),
                Wiconnect::fileTypeToStr(file->getType()),
                Wiconnect::fileFlagsToStr(file->getFlags()));
    }
}
