/**
 * @example tcp_server/example.cpp
 *
 * This is an example of using the TCP server API to
 * send/receive data from a remote TCP client.
 *
 * This example is intended to be used with 'tcp_client.py'
 * python script in the same directory as this example.cpp file.
 *
 * It works as follows:
 * 1. Instantiate the WiConnect Library
 * 2. Initiate Communication with WiFi Module
 * 3. Join the network
 * 4. Start the TCP server
 * 5. Wait for clients to connect
 * 6. Receive data from client
 * 7. Send data to client
 * 8. Close client connection
 * 9. Goto sleep 5
 *
 */


/******************************************************************************
 * Example Variables
 */

// The port the server listens on
#define TCP_SERVER_PORT 7
// The maximum simultaneous client connections
// (note this example only supports 1)
#define TCP_SERVER_MAX_CLIENTS 1

// This is the name of your WiFi network
// Look for this name in your WiFi settings
// (e.g. your phone's list of WiFi networks in the WiFi settings menu)
// tip: add double-quotes around SSID to add spaces to name
#define NETWORK_SSID "\"<YOUR NETWORK NAME HERE>\""

// This is the password of your WiFi network
// Leave as empty string (e.g "") to connect to OPEN network
#define NETWORK_PASSWORD "\"<YOUR NETWORK PASSWORD HERE>\""


/******************************************************************************
 * Includes
 */

// include C library headers
#include <stdio.h> // needed for printf

// include target specific defines
#include "target_config.h"
// include the Wiconnect Host Library API header
#include "Wiconnect.h"



/******************************************************************************
 * Local Functions
 */


/******************************************************************************
 * Global Defines
 */

// Transmit/receive buffers for the remote client socket
static uint8_t clientRxBuffer[256], clientTxBuffer[256];

// Serial used for printfs to terminal (i.e. NOT used for WiConnect)
static Serial consoleSerial(STDIO_UART_TX, STDIO_UART_RX);



//-------------------------------------------------------------------------
// STEP 1: Instantiate WiConnect Library
//-------------------------------------------------------------------------


// Setup  wiconnect serial interface configuration
// Here we only specify the rx buffer size and not rx buffer pointer, this means
// The serial RX buffer will be dynamically allocated
SerialConfig serialConfig(WICONNECT_RX_PIN, WICONNECT_TX_PIN, 256, NULL);

// Instantiate the Wiconnect library
// Here we specify the buffer size ONLY which means we're using dynmaic allocation
Wiconnect wiconnectIfc(serialConfig, 256, NULL, WICONNECT_RESET_PIN);





/******************************************************************************
 * Starting point of application
 */
int main(int argc, char **argv)
{
    WiconnectResult result;
    // Instantiate a client socket object with statically allocaed transmit/receive buffers
    // Note: this socket object isn't valid until tcpAccept() is called with in
    WiconnectSocket clientSocket(sizeof(clientRxBuffer), clientRxBuffer, sizeof(clientTxBuffer), clientTxBuffer);

    consoleSerial.baud(115200); // console terminal to 115200 baud


    //-------------------------------------------------------------------------
    // STEP 2: Initiate Communication with WiFi Module
    //-------------------------------------------------------------------------

    printf("Initializing WiConnect Library...\r\n");

    // Initialize communication with WiFi module
    if(WICONNECT_FAILED(result, wiconnectIfc.init(true)))
    {
        printf("Failed to initialize communication with WiFi module: %s\r\n"
                "Make sure the wires are connected correctly\r\n", Wiconnect::getWiconnectResultStr(result));
        if(result == WICONNECT_FIRMWARE_OUTDATED)
        {
            printf("** The WiFi firmware is not supported. Run the ota example to update the firmware:\r\n");
            printf("https://developer.mbed.org/teams/ACKme/code/wiconnect-ota_example\r\n\r\n");
        }
        for(;;); // infinite loop
    }

    //-------------------------------------------------------------------------
    // STEP 3: Join the network
    //-------------------------------------------------------------------------

    printf("Joining WiFi network: %s\r\n", NETWORK_SSID);

    // Initialize communication with WiFi module
    if(WICONNECT_FAILED(result, wiconnectIfc.join(NETWORK_SSID, NETWORK_PASSWORD)))
    {
        printf("Failed to join network: %s\r\n", Wiconnect::getWiconnectResultStr(result));
        for(;;); // infinite loop
    }


    //-------------------------------------------------------------------------
    // STEP 4: Start the TCP server
    //-------------------------------------------------------------------------

    printf("Starting TCP server, listening on: %s:%d\r\n", wiconnectIfc.getIpAddress(), TCP_SERVER_PORT);

    if(WICONNECT_FAILED(result, wiconnectIfc.tcpListen(TCP_SERVER_PORT, TCP_SERVER_MAX_CLIENTS)))
    {
        printf("Failed to start TCP server: %s\r\n", Wiconnect::getWiconnectResultStr(result));
        for(;;); // infinite loop
    }


    for(;;)
    {
        //-------------------------------------------------------------------------
        // STEP 5: Wait for clients to connect
        //-------------------------------------------------------------------------

        printf("Waiting for a client to connect...\r\n");

        if(WICONNECT_FAILED(result, wiconnectIfc.tcpAccept(clientSocket)))
        {
            printf("Failed to accept client: %s\r\n", Wiconnect::getWiconnectResultStr(result));
            continue;
        }
        printf("Client connected: %s:%d\r\n", clientSocket.getHost(), clientSocket.getRemotePort());

        //-------------------------------------------------------------------------
        // STEP 6: Receive data from client
        //-------------------------------------------------------------------------

        uint8_t *dataPtr;   // pointer to client socket's internal RX buffer
        uint16_t readSize;  // will contain number of bytes available in RX buffer
        if(WICONNECT_FAILED(result, clientSocket.read(&dataPtr, &readSize)))
        {
            printf("Failed to read data from client: %s\r\n", Wiconnect::getWiconnectResultStr(result));
            clientSocket.close();
            continue;
        }

        printf("From client: %s\r\n", dataPtr);


        //-------------------------------------------------------------------------
        // STEP 7: Send data to client
        //-------------------------------------------------------------------------

        if(WICONNECT_FAILED(result, clientSocket.puts("Hello client!!\r\n")))
        {
            printf("Failed to send data to client: %s\r\n", Wiconnect::getWiconnectResultStr(result));
            clientSocket.close();
            continue;
        }

        //-------------------------------------------------------------------------
        // STEP 8: Close client connection
        //-------------------------------------------------------------------------

        clientSocket.close();
    }

}

