/**
 * @example ota/example.cpp
 *
 * This is an example of using the wiconnect API to update
 * the wifi module's internal firmware to the latest version.
 *
 * It works as follows:
 * 1. Instantiate the WiConnect Library
 * 2. Initiate Communication with WiFi Module
 * 3. Set the network credentials
 * 4. Call the 'updateFirmware' wiconnect API and wait for the update to complete (takes ~60s)
 * 5. That's it!
 *
 *
 */


/******************************************************************************
 * Example Variables
 */

// This is the name of your WiFi network
// Look for this name in your WiFi settings
// (e.g. your phone's list of WiFi networks in the WiFi settings menu)
// tip: add double-quotes around SSID to add spaces to name
#define NETWORK_SSID "\"<YOUR NETWORK NAME HERE>\""


// This is the password of your WiFi network
// Leave as empty string (e.g "") to connect to OPEN network
#define NETWORK_PASSWORD "\"<YOUR NETWORK PASSWORD HERE>\""

/******************************************************************************
 * Includes
 */

// include C library headers
#include <stdio.h> // needed for printf

// include target specific defines
#include "target_config.h"
// include the Wiconnect Host Library API header
#include "Wiconnect.h"



/******************************************************************************
 * Global Defines
 */


// Serial used for printfs to terminal (i.e. NOT used for WiConnect)
static Serial consoleSerial(STDIO_UART_TX, STDIO_UART_RX);


/******************************************************************************
 * Starting point of application
 */
int main(int argc, char **argv)
{
    WiconnectResult result;
    
    consoleSerial.baud(115200); // console terminal to 115200 baud

    //-------------------------------------------------------------------------
    // STEP 1: Instantiate WiConnect Library
    //-------------------------------------------------------------------------

    // Setup  wiconnect serial interface configuration
    // Here we only specify the rx buffer size and not rx buffer pointer, this means
    // The serial RX buffer will be dynamically allocated
    SerialConfig serialConfig(WICONNECT_RX_PIN, WICONNECT_TX_PIN, 256, NULL);

    // Instantiate the Wiconnect library
    // Here we only specify the buffer size and not buffer pointer, this means
    // The internal buffer will be dynamically allocated
    Wiconnect wiconnect(serialConfig, 256, NULL, WICONNECT_RESET_PIN);

    //-------------------------------------------------------------------------
    // STEP 2: Initiate Communication with WiFi Module
    //-------------------------------------------------------------------------

    printf("Initializing WiConnect Library...\r\n");

    // Initialize communication with WiFi module
    if(WICONNECT_FAILED(result, wiconnect.init(true)))
    {
        // if the error is that the firmware is out-dated, then continue, else stop
        if(result != WICONNECT_FIRMWARE_OUTDATED)
        {
            printf("Failed to initialize communication with WiFi module!\r\n"
                    "Make sure the wires are connected correctly\r\n");
            for(;;); // infinite loop
        }
    }


    //-------------------------------------------------------------------------
    // STEP 3: Set the network credentials
    // Note: We manually set the network parameters to ensure SDK backward compatibility
    //-------------------------------------------------------------------------

    printf("Setting network SSID: %s\r\n", NETWORK_SSID);
    if(WICONNECT_FAILED(result, wiconnect.setSetting("wlan.ssid", NETWORK_SSID)))
    {
        printf("Failed to set wlan.ssid setting\r\n");
        for(;;); // infinite loop
    }
    
    printf("Setting network password\r\n");
    if(WICONNECT_FAILED(result, wiconnect.setSetting("wlan.passkey", NETWORK_PASSWORD)))
    {
        printf("Failed to set wlan.passkey setting\r\n");
        for(;;); // infinite loop
    }

    printf("Saving settings to Non-volatile Memory\r\n");
    if(WICONNECT_FAILED(result, wiconnect.saveSettings()))
    {
        printf("Failed save settings\r\n");
        for(;;); // infinite loop
    }
    
    //-------------------------------------------------------------------------
    // STEP 4: Update the module's firmware, this takes about 60s
    //-------------------------------------------------------------------------

    printf("Updating WiFi module's internal firmware to latest (note this can take up to 60s)\r\n");

    if(wiconnect.updateFirmware() != WICONNECT_SUCCESS)
    {
        printf("Failed to update the module's firmware\r\n");
        for(;;); // infinite loop
    }

    //-------------------------------------------------------------------------
    // STEP 5: Done!
    //-------------------------------------------------------------------------

    wiconnect.getVersion();
    printf("Version: %s\r\n", wiconnect.getResponseBuffer());

    while(true){} // infinite loop
}

