/**
 * @example join/example.cpp
 *
 * This is an example of using the join network API to
 * join a WiFi network.
 *
 * It works as follows:
 * 1. Instantiate the WiConnect Library
 * 2. Initiate Communication with WiFi Module
 * 3. Join a network using the specified parameters
 * 4. That's it!
 *
 *
 */


/******************************************************************************
 * Example Variables
 */

// This is the name of your WiFi network
// Look for this name in your WiFi settings
// (e.g. your phone's list of WiFi networks in the WiFi settings menu)
// tip: add double-quotes around SSID to add spaces to name
#define NETWORK_SSID "\"<YOUR NETWORK NAME HERE>\""

// This is the password of your WiFi network
// Leave as empty string (e.g "") to connect to OPEN network
#define NETWORK_PASSWORD "\"<YOUR NETWORK PASSWORD HERE>\""


/******************************************************************************
 * Includes
 */

// include C library headers
#include <stdio.h> // needed for printf

// include target specific defines
#include "target_config.h"
// include the Wiconnect Host Library API header
#include "Wiconnect.h"



/******************************************************************************
 * Global Defines
 */


// Serial used for printfs to terminal (i.e. NOT used for WiConnect)
static Serial consoleSerial(STDIO_UART_TX, STDIO_UART_RX);


/******************************************************************************
 * Starting point of application
 */
int main(int argc, char **argv)
{
    WiconnectResult result;
    
    consoleSerial.baud(115200); // console terminal to 115200 baud

    //-------------------------------------------------------------------------
    // STEP 1: Instantiate WiConnect Library
    //-------------------------------------------------------------------------

    // Setup  wiconnect serial interface configuration
    // Here we only specify the rx buffer size and not rx buffer pointer, this means
    // The serial RX buffer will be dynamically allocated
    SerialConfig serialConfig(WICONNECT_RX_PIN, WICONNECT_TX_PIN, 256, NULL);

    // Instantiate the Wiconnect library
    // Here we only specify the buffer size and not buffer pointer, this means
    // The internal buffer will be dynamically allocated
    Wiconnect wiconnect(serialConfig, 256, NULL, WICONNECT_RESET_PIN);

    //-------------------------------------------------------------------------
    // STEP 2: Initiate Communication with WiFi Module
    //-------------------------------------------------------------------------

    printf("Initializing WiConnect Library...\r\n");

    // Initialize communication with WiFi module
    if(WICONNECT_FAILED(result, wiconnect.init(true)))
    {
        if(result == WICONNECT_FIRMWARE_OUTDATED)
        {
            printf("** The WiFi firmware is not supported. Run the ota example to update the firmware:\r\n");
            printf("https://developer.mbed.org/teams/ACKme/code/wiconnect-ota_example\r\n\r\n");
        }
        else
        {
            printf("Failed to initialize communication with WiFi module!\r\n"
                   "Make sure the wires are connected correctly\r\n");
        }
        for(;;); // infinite loop
    }


    //-------------------------------------------------------------------------
    // STEP 3: Join a network using the specified parameters
    //-------------------------------------------------------------------------

    printf("Joining network: %s....\r\n", NETWORK_SSID);

    if(wiconnect.join(NETWORK_SSID, NETWORK_PASSWORD) != WICONNECT_SUCCESS)
    {
        printf("Failed to send join command\r\n");
        for(;;); // infinite loop
    }

    //-------------------------------------------------------------------------
    // STEP 4: Done!
    //-------------------------------------------------------------------------

    printf("IP Address: %s\r\n", wiconnect.getIpAddress());
    printf("Network join example has completed!\r\n");

    while(true){} // infinite loop
}

