/**
 * @example http_get/example.cpp
 *
 * This is an example of using the http socket API to
 * download a webpage.
 *
 * It works as follows:
 * 1. Instantiate the WiConnect Library
 * 2. Initiate Communication with WiFi Module
 * 3. Issue a HTTP GET request (Note: The module must have valid network credentials See @ref web_setup/example.cpp)
 * 4. Read the HTTP page and print
 * 5. That's it!
 *
 */


/******************************************************************************
 * Example Variables
 */

// the URL of the webpage to GET
#define HTTP_URL "http://mbed.org"





/******************************************************************************
 * Includes
 */

// include C library headers
#include <stdio.h> // needed for printf

// include target specific defines
#include "target_config.h"
// include the Wiconnect Host Library API header
#include "Wiconnect.h"




/******************************************************************************
 * Global Defines
 */


// Serial used for printfs to terminal (i.e. NOT used for WiConnect)
static Serial consoleSerial(STDIO_UART_TX, STDIO_UART_RX);



/******************************************************************************
 * Starting point of application
 */
int main(int argc, char **argv)
{
    WiconnectResult result;
    
    consoleSerial.baud(115200); // console terminal to 115200 baud

    //-------------------------------------------------------------------------
    // STEP 1: Instantiate WiConnect Library
    //-------------------------------------------------------------------------

    // Setup  wiconnect serial interface configuration
    // Here we only specify the rx buffer size and not rx buffer pointer, this means
    // The serial RX buffer will be dynamically allocated
    SerialConfig serialConfig(WICONNECT_RX_PIN, WICONNECT_TX_PIN, 256, NULL);

    // Instantiate and initialize the Wiconnect library
    // Here we only specify the buffer size and not buffer pointer, this means
    // The internal buffer will be dynamically allocated
    Wiconnect wiconnect(serialConfig, 256, NULL, WICONNECT_RESET_PIN);

    // set the default timeout to 15s as some websites take awhile to respond
    wiconnect.setCommandDefaultTimeout(15000);

    //-------------------------------------------------------------------------
    // STEP 2: Initiate Communication with WiFi Module
    //-------------------------------------------------------------------------

    printf("Initializing WiConnect Library...\r\n");

    // Initialize communication with WiFi module
    if(WICONNECT_FAILED(result, wiconnect.init(true)))
    {
        if(result == WICONNECT_FIRMWARE_OUTDATED)
        {
            printf("** The WiFi firmware is not supported. Run the ota example to update the firmware:\r\n");
            printf("https://developer.mbed.org/teams/ACKme/code/wiconnect-ota_example\r\n\r\n");
        }
        else
        {
            printf("Failed to initialize communication with WiFi module!\r\n"
                   "Make sure the wires are connected correctly\r\n");
        }
        for(;;); // infinite loop
    }


    //-------------------------------------------------------------------------
    // STEP 3: Issue a HTTP GET request
    //-------------------------------------------------------------------------

    // Initiate a socket with an RX buffer of 256 bytes
    // We're not specifying the rx buffer pointer so that means it's dynamically allocated
    WiconnectSocket socket(256);


    printf("Issuing HTTP Request: %s\r\n", HTTP_URL);

    // Issue HTTP request
    // NOTE: the module must have valid network credentials
    if(wiconnect.httpGet(socket, HTTP_URL) != WICONNECT_SUCCESS)
    {
        printf("Failed to issue HTTP request. Does the module have valid network credentials?\r\n");
        printf("Use the network join example to configure the module's network credentials:\r\n");
        printf("https://developer.mbed.org/teams/ACKme/code/wiconnect-join_example\r\n");
        for(;;); // infinite loop
    }

    //-------------------------------------------------------------------------
    // STEP 4: Read the HTTP page and print
    //-------------------------------------------------------------------------

    uint8_t *dataPtr;
    uint16_t dataLength;

    // while there's data to read, retreive and print it
    while(socket.read(&dataPtr, &dataLength) == WICONNECT_SUCCESS)
    {
        fwrite(dataPtr, 1, dataLength, stdout);
    }

    //-------------------------------------------------------------------------
    // STEP 5: Done!
    //-------------------------------------------------------------------------

    printf("HTTP GET example has completed!\r\n");

    while(true){} // infinite loop
}


