/**
 * ACKme WiConnect Host Library is licensed under the BSD licence:
 *
 * Copyright (c)2014 ACKme Networks.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 * derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

#include "Wiconnect.h"
#include "api/types/Socket/TCPSocketConnection.h"



/*************************************************************************************************/
TCPSocketConnection::TCPSocketConnection(int rxBufferLen, void *rxBuffer, int txBufferLen, void *txBuffer)
: Socket(rxBufferLen, rxBuffer, txBufferLen, txBuffer)
{
}

/*************************************************************************************************/
int TCPSocketConnection::connect(const char* host, const int port)
{
    return (Wiconnect::getInstance()->tcpConnect(socket, host, (uint16_t)port) == WICONNECT_SUCCESS) ? 0 : -1;
}

/*************************************************************************************************/
bool TCPSocketConnection::is_connected(void)
{
    return socket.isConnected();
}

/*************************************************************************************************/
// currently there's no way to timeout writing data to WiConnect, so we just have to try and send it all.
// NOTE: this WILL be fixed in the next version
int TCPSocketConnection::send(char* data, int length)
{
    return send_all(data, length);
}

/*************************************************************************************************/
// -1 if unsuccessful, else number of bytes written
int TCPSocketConnection::send_all(char* data, int length)
{
    return (socket.write(data, length, true) == WICONNECT_SUCCESS) ? length : -1;
}

/*************************************************************************************************/
int TCPSocketConnection::receive(char* data, int length)
{
    uint16_t bytesRead;

    if (!_blocking)
    {
        if(socket.getRxBufferSize() == 0 || socket.getRxBufferBytesPending() == 0)
        {
            TimeoutTimer timer;

            for(;;)
            {
                bool hasData;

                if(socket.poll(&hasData) != WICONNECT_SUCCESS)
                {
                    return -1;
                }
                else if(hasData)
                {
                    break;
                }
                else if(timer.timedOut(_timeout))
                {
                    return -1;
                }
            }
        }
    }

    if(length == 1 && socket.getRxBufferSize() > 0)
    {
        return (socket.getc((uint8_t*)data) == WICONNECT_SUCCESS) ? 1 : -1;
    }
    else
    {
        for(;;)
        {
            if(socket.read(data, length, &bytesRead) != WICONNECT_SUCCESS)
            {
                return -1;
            }
            else if(bytesRead > 0)
            {
                return bytesRead;
            }
        }
    }
}

/*************************************************************************************************/
// -1 if unsuccessful, else number of bytes received
int TCPSocketConnection::receive_all(char* data, int length)
{
    char *ptr = data;
    int totReadSize = length;

    while(length > 0)
    {
        int bytesRead = receive(ptr, length);
        if(bytesRead == -1)
        {
            return -1;
        }
        ptr += bytesRead;
        length -= bytesRead;
    }

    return totReadSize;
}


