/**
 * ACKme WiConnect Host Library is licensed under the BSD licence: 
 * 
 * Copyright (c)2014 ACKme Networks.
 * All rights reserved. 
 * 
 * Redistribution and use in source and binary forms, with or without modification, 
 * are permitted provided that the following conditions are met: 
 * 
 * 1. Redistributions of source code must retain the above copyright notice, 
 * this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 * this list of conditions and the following disclaimer in the documentation 
 * and/or other materials provided with the distribution. 
 * 3. The name of the author may not be used to endorse or promote products 
 * derived from this software without specific prior written permission. 
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
 * OF SUCH DAMAGE.
 */
#pragma once


#define MBED_SDK

#include "mbed.h"




namespace wiconnect
{

/**
 * @ingroup api_core_macro
 * @brief This is the minimum supported WiConnect version. If your
 *        WiFi module's firmware is out-dated, use the updateFirmware()
 *        API call to update your module's firmware.
 */
#define WICONNECT_MINIMUM_VERSION FILE_MAKE_VERSION(2, 0, 0, 11)

/**
 * @ingroup api_core_macro
 * @brief When defined enables asynchronous command processing
 */
#define WICONNECT_ASYNC_TIMER_ENABLED
/**
 * @ingroup api_core_macro
 * @brief When defined enables user supplied dynamic memory allocation
 */
#define WICONNECT_ENABLE_MALLOC
/**
 * @ingroup api_core_macro
 * @brief When defined enables Host<->Wiconnect Module serial RX buffering
 */
#define WICONNECT_SERIAL_RX_BUFFER
/**
 * @ingroup api_core_macro
 * @brief When defined enables certain conversion API functions to use a default buffer to store string
 */
#define WICONNECT_USE_DEFAULT_STRING_BUFFERS

/**
 * @ingroup api_core_macro
 * @brief When defined enables external interrupts on specified gpios
 */
#define WICONNECT_GPIO_IRQ_ENABLED

/**
 * @ingroup api_core_macro
 * @brief When WICONNECT_ENABLE_MALLOC defined, this is the default malloc function
 */
#define WICONNECT_DEFAULT_MALLOC malloc
/**
 * @ingroup api_core_macro
 * @brief When WICONNECT_ENABLE_MALLOC defined, this is the default free function
 */
#define WICONNECT_DEFAULT_FREE free

/**
 * @ingroup api_core_macro
 * @brief The default Host<->Wiconnect Module serial BAUD rate
 */
#define WICONNECT_DEFAULT_BAUD 115200
/**
 * @ingroup api_core_macro
 * @brief The default command timeout (i.e max command executing time)
 */
#define WICONNECT_DEFAULT_TIMEOUT 3000 // ms
/**
 * @ingroup api_core_macro
 * @brief When WICONNECT_ASYNC_TIMER_ENABLED, this specifies the max number of asynchronous commands that may be queued
 */
#define WICONNECT_MAX_QUEUED_COMMANDS 8
/**
 * @ingroup api_core_macro
 * @brief When WICONNECT_ASYNC_TIMER_ENABLED, this specifies the period in milliseconds commands should be processed
 */
#define WICONNECT_DEFAULT_COMMAND_PROCESSING_PERIOD 50 // ms

/**
 * @ingroup api_core_macro
 * @brief The default blocking mode of the Library.
 */
#define WICONNECT_DEFAULT_NONBLOCKING false

/**
 * @ingroup api_core_macro
 * @brief If defined, enables low-level debugging
 */
#define WICONNECT_ENABLE_DEBUGGING


// ----------------------------------------------------------------------------

#define WICONNECT_GPIO_BASE_CLASS : DigitalOut
#define WICONNECT_SERIAL_BASE_CLASS : RawSerial
#define WICONNECT_PERIODIC_TIMER_BASE_CLASS : Ticker
#define WICONNECT_EXTERNAL_INTERRUPT_GPIO_BASE_CLASS : InterruptIn

#define WICONNECT_MAX_PIN_IRQ_HANDLERS 3


/**
 * @ingroup api_core_macro
 * @brief Default value for a pin, Not connected
 */
#define PIN_NC NC

/**
 * @ingroup api_core_types
 * @brief Pin name on HOST
 */
typedef PinName Pin;

/**
 * @ingroup api_core_types
 * @brief Host<->Wiconnect Module serial configuration
 */
class SerialConfig
{
public:
    Pin rx;
    Pin tx;
    Pin cts;
    Pin rts;
    int baud;
    void *serialRxBuffer;
    int serialRxBufferSize;

    SerialConfig(Pin rx, Pin tx, Pin cts, Pin rts, int baud, int serialRxBufferSize, void *serialRxBuffer = NULL)
    {
        this->rx =rx;
        this->tx =tx;
        this->cts =cts;
        this->rts =rts;
        this->baud = baud;
        this->serialRxBuffer =serialRxBuffer;
        this->serialRxBufferSize =serialRxBufferSize;
    }

    SerialConfig(Pin rx, Pin tx, int serialRxBufferSize, void *serialRxBuffer = NULL)
    {
        this->rx =rx;
        this->tx =tx;
        this->cts = PIN_NC;
        this->rts = PIN_NC;
        this->baud = WICONNECT_DEFAULT_BAUD;
        this->serialRxBuffer =serialRxBuffer;
        this->serialRxBufferSize =serialRxBufferSize;
    }

    SerialConfig(Pin rx, Pin tx)
    {
        this->rx =rx;
        this->tx =tx;
        this->cts =PIN_NC;
        this->rts =PIN_NC;
        this->baud = WICONNECT_DEFAULT_BAUD;
        this->serialRxBuffer =NULL;
        this->serialRxBufferSize =0;
    }

};


/**
 * @ingroup api_core_macro
 * @brief Function to stop processor for specified number of milliseconds
 */
#define delayMs(ms) wait_ms(ms)




}
