/*
Copyright (c) 2010 Peter Barrett

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

/* 
Tue Apr 26 2011 Bart Janssens: added a socket listener
*/

#ifndef SOCKET_H_INCLUDED
#define SOCKET_H_INCLUDED

#define SOCKET_HCI      1
#define SOCKET_L2CAP    2
#define SOCKET_RFCOM    3

typedef struct
{
    u8  AddressSpecific[0]; // BDADDR,psm etc
} SocketAddrHdr;



enum SocketState
{
    SocketState_Unknown,
    SocketState_Opening,
    SocketState_Open,
    SocketState_Closing,
    SocketState_Closed,
    SocketState_Listen,
    SocketState_Accept
};

typedef void (*SocketCallback)(int socket, SocketState state, const u8* data, int len, void* userData);

class SocketListener
{
    public:
    int ID;
    int Type;
    int port;
    SocketCallback Callback;
    void* userData;
};

//===========================================================================
//===========================================================================
//  Don't need to look at or use anything below this line:
//  Internal representation of socket

class SocketHandler;
class SocketInternal
{
    public:

    u8 ID;
    u8 State;
    u8 Type;
    u8 B0;
    int port;
    SocketCallback Callback;
    void* userData;
    u8  Data[0];    // Extra socket data starts here

    void Recv(const u8* data, int len)
    {
        Callback(ID,(SocketState)State,data,len,userData);
    }

    void SetState(SocketState state)
    {
        State = state;
        Callback(ID,(SocketState)State,0,0,userData);
    }
};

class SocketHandler
{
    public:
    virtual int Open(SocketInternal* sock, SocketAddrHdr* addr) = 0;
    virtual int Create(SocketInternal* sock, SocketAddrHdr* addr) = 0;
    virtual int Send(SocketInternal* sock, const u8* data, int len) = 0;
    virtual int Close(SocketInternal* sock) = 0;
    virtual int Accept(SocketInternal* sock, SocketAddrHdr* addr) = 0;
};

int RegisterSocketHandler(int type, SocketHandler* handler);
SocketInternal* GetSocketInternal(int socket);

int Socket_Open(int type, SocketAddrHdr* addr, SocketCallback callback, void* userData);   // Open a socket
int Socket_Listen(int type, int port,SocketCallback callback, void* userData);
int Socket_InUse(int type, int port);
int Socket_Accept(int socket, SocketCallback callback, void* userData);
SocketInternal* Socket_Create(int type,SocketAddrHdr* addr, int port);
int Socket_Send(int socket, const u8* data, int len);
int Socket_State(int socket);
int Socket_Close(int socket);

#define ERR_SOCKET_TYPE_NOT_FOUND -200
#define ERR_SOCKET_NOT_FOUND -201
#define ERR_SOCKET_NONE_LEFT -202
#define ERR_SOCKET_IN_USE -203

#endif // SOCKET_H_INCLUDED
