#include "KalmanFilter.h"

KalmanFilter::KalmanFilter()
{
    reset();
}

KalmanFilter::~KalmanFilter(void)
{
}

void KalmanFilter::reset()
{    
    A[0][0] = 1.0;
    A[0][1] = -0.0025;
    A[1][0] = 0.0;
    A[1][1] = 1.0;
    
    B[0][0] = 0.0025;
    B[1][0] = 0.0;
    
    C[0][0] = 1.0;
    C[0][1] = 0.0;
    
    Sz[0][0] = 1; //7.2
    
    Sw[0][0] = 0.0025;
    Sw[0][1] = 0.0025;
    Sw[1][0] = 0.0025;
    Sw[1][1] = 0.0025;
    
    xhat[0][0] = 0.0;
    xhat[1][0] = 0.0;
    
    /* Matriz P -> covariância do do erro
     * Definindo o estado inicial da matriz de covariância do erro, 
     * é aconselhável a definir L um número alto. 
     * Neste caso, sabemos o estado inicial do nosso modelo, 
     * então L é igual a ZERO.
     */
    
    P[0][0] = 0; //L
    P[0][1] = 0;
    P[1][0] = 0;
    P[1][1] = 0; //L
}

float KalmanFilter::update(float gyroRaw, float accelAngle)
{
    // Inputs.
    float u[1][1];              // Gyroscope rate.
    float y[1][1];              // Accelerometer angle.
    
    // Temp values.
    float AP[2][2];             // This is the matrix A*P
    float CT[2][1];             // This is the matrix C'
    float APCT[2][1];           // This is the matrix A*P*C'
    float CP[1][2];             // This is the matrix C*P
    float CPCT[1][1];           // This is the matrix C*P*C'
    float CPCTSz[1][1];         // This is the matrix C*P*C'+Sz
    float CPCTSzInv[1][1];      // This is the matrix inv(C*P*C'+Sz)
    float K[2][1];              // This is the Kalman gain.
    float Cxhat[1][1];          // This is the vector C*xhat
    float yCxhat[1][1];         // This is the vector y-C*xhat
    float KyCxhat[2][1];        // This is the vector K*(y-C*xhat)
    float Axhat[2][1];          // This is the vector A*xhat 
    float Bu[2][1];             // This is the vector B*u
    float AxhatBu[2][1];        // This is the vector A*xhat+B*u
    float AT[2][2];             // This is the matrix A'
    float APAT[2][2];           // This is the matrix A*P*A'
    float APATSw[2][2];         // This is the matrix A*P*A'+Sw
    float KC[2][2];             // This is the matrix K*C
    float KCP[2][2];            // This is the matrix K*C*P
    float KCPAT[2][2];          // This is the matrix K*C*P*A'
    
    // Fill in inputs.
    u[0][0] = gyroRaw;
    y[0][0] = accelAngle;
    
    // Update the state estimate by extrapolating estimate with gyroscope input.
    // xhat_est = A * xhat + B * u
    matrix_multiply((float*) A, (float*) xhat, 2, 2, 1, (float*) Axhat);
    matrix_multiply((float*) B, (float*) u, 2, 1, 1, (float*) Bu);
    matrix_addition((float*) Axhat, (float*) Bu, 2, 1, (float*) AxhatBu);
    
    // Compute the innovation.
    // Inn = y - c * xhat;
    matrix_multiply((float*) C, (float*) xhat, 1, 2, 1, (float*) Cxhat);
    matrix_subtraction((float*) y, (float*) Cxhat, 1, 1, (float*) yCxhat);
    
    // Compute the covariance of the innovation.
    // s = C * P * C' + Sz
    matrix_transpose((float*) C, 1, 2, (float*) CT);
    matrix_multiply((float*) C, (float*) P, 1, 2, 2, (float*) CP);
    matrix_multiply((float*) CP, (float*) CT, 1, 2, 1, (float*) CPCT);
    matrix_addition((float*) CPCT, (float*) Sz, 1, 1, (float*) CPCTSz);
    
    // Compute the kalman gain matrix.
    // K = A * P * C' * inv(s)
    matrix_multiply((float*) A, (float*) P, 2, 2, 2, (float*) AP);
    matrix_multiply((float*) AP, (float*) CT, 2, 2, 1, (float*) APCT);
    matrix_inversion((float*) CPCTSz, 1, (float*) CPCTSzInv);
    matrix_multiply((float*) APCT, (float*) CPCTSzInv, 2, 1, 1, (float*) K);
    
    // Update the state estimate.
    // xhat = xhat_est + K * Inn;
    matrix_multiply((float*) K, (float*) yCxhat, 2, 1, 1, (float*) KyCxhat);
    matrix_addition((float*) AxhatBu, (float*) KyCxhat, 2, 1, (float*) xhat);
    
    // Compute the new covariance of the estimation error.
    // P = A * P * A' - K * C * P * A' + Sw
    matrix_transpose((float*) A, 2, 2, (float*) AT);
    matrix_multiply((float*) AP, (float*) AT, 2, 2, 2, (float*) APAT);
    matrix_addition((float*) APAT, (float*) Sw, 2, 2, (float*) APATSw);
    matrix_multiply((float*) K, (float*) C, 2, 1, 2, (float*) KC);
    matrix_multiply((float*) KC, (float*) P, 2, 2, 2, (float*) KCP);
    matrix_multiply((float*) KCP, (float*) AT, 2, 2, 2, (float*) KCPAT);
    matrix_subtraction((float*) APATSw, (float*) KCPAT, 2, 2, (float*) P);
    
    // Return the estimate.
    return xhat[0][0];
}

float KalmanFilter::getEstimatedData()
{
    return xhat[0][0];
}

void KalmanFilter::matrix_multiply(float* A, float* B, int m, int p, int n, float* C)
// Matrix Multiplication Routine
{
    // A = input matrix (m x p)
    // B = input matrix (p x n)
    // m = number of rows in A
    // p = number of columns in A = number of rows in B
    // n = number of columns in B
    // C = output matrix = A*B (m x n)
    int i, j, k;
    for (i=0;i<m;i++)
        for(j=0;j<n;j++)
        {
          C[n*i+j]=0;
          for (k=0;k<p;k++)
            C[n*i+j]= C[n*i+j]+A[p*i+k]*B[n*k+j];
        }
}

void KalmanFilter::matrix_addition(float* A, float* B, int m, int n, float* C)
// Matrix Addition Routine
{
    // A = input matrix (m x n)
    // B = input matrix (m x n)
    // m = number of rows in A = number of rows in B
    // n = number of columns in A = number of columns in B
    // C = output matrix = A+B (m x n)
    int i, j;
    for (i=0;i<m;i++)
        for(j=0;j<n;j++)
            C[n*i+j]=A[n*i+j]+B[n*i+j];
}

void KalmanFilter::matrix_subtraction(float* A, float* B, int m, int n, float* C)
// Matrix Subtraction Routine
{
    // A = input matrix (m x n)
    // B = input matrix (m x n)
    // m = number of rows in A = number of rows in B
    // n = number of columns in A = number of columns in B
    // C = output matrix = A-B (m x n)
    int i, j;
    for (i=0;i<m;i++)
        for(j=0;j<n;j++)
            C[n*i+j]=A[n*i+j]-B[n*i+j];
}

void KalmanFilter::matrix_transpose(float* A, int m, int n, float* C)
// Matrix Transpose Routine
{
    // A = input matrix (m x n)
    // m = number of rows in A
    // n = number of columns in A
    // C = output matrix = the transpose of A (n x m)
    int i, j;
    for (i=0;i<m;i++)
        for(j=0;j<n;j++)
            C[m*j+i]=A[n*i+j];
}


int KalmanFilter::matrix_inversion(float* A, int n, float* AInverse)
// Matrix Inversion Routine
{
    // A = input matrix (n x n)
    // n = dimension of A
    // AInverse = inverted matrix (n x n)
    // This function inverts a matrix based on the Gauss Jordan method.
    // The function returns 1 on success, 0 on failure.
    int i, j, iPass, imx, icol, irow;
    float det, temp, pivot, factor;
    float* ac = (float*) calloc(n*n, sizeof(float));
    det = 1;
    for (i = 0; i < n; i++)
    {
        for (j = 0; j < n; j++)
        {
            AInverse[n*i+j] = 0;
            ac[n*i+j] = A[n*i+j];
        }
        AInverse[n*i+i] = 1;
    }

    // The current pivot row is iPass.
    // For each pass, first find the maximum element in the pivot column.
    for (iPass = 0; iPass < n; iPass++)
    {
        imx = iPass;
        for (irow = iPass; irow < n; irow++)
        {
            if (fabs(A[n*irow+iPass]) > fabs(A[n*imx+iPass])) imx = irow;
        }

        // Interchange the elements of row iPass and row imx in both A and AInverse.
        if (imx != iPass)
        {
            for (icol = 0; icol < n; icol++)
            {
                temp = AInverse[n*iPass+icol];
                AInverse[n*iPass+icol] = AInverse[n*imx+icol];
                AInverse[n*imx+icol] = temp;
                if (icol >= iPass)
                {
                    temp = A[n*iPass+icol];
                    A[n*iPass+icol] = A[n*imx+icol];
                    A[n*imx+icol] = temp;
                }
            }
        }

        // The current pivot is now A[iPass][iPass].
        // The determinant is the product of the pivot elements.
        pivot = A[n*iPass+iPass];
        det = det * pivot;
        if (det == 0)
        {
            free(ac);
            return 0;
        }

        for (icol = 0; icol < n; icol++)
        {
            // Normalize the pivot row by dividing by the pivot element.
            AInverse[n*iPass+icol] = AInverse[n*iPass+icol] / pivot;
            if (icol >= iPass) A[n*iPass+icol] = A[n*iPass+icol] / pivot;
        }

        for (irow = 0; irow < n; irow++)
        {
            // Add a multiple of the pivot row to each row.  The multiple factor
            // is chosen so that the element of A on the pivot column is 0.
            if (irow != iPass) factor = A[n*irow+iPass];
            for (icol = 0; icol < n; icol++)
            {
                if (irow != iPass)
                {
                    AInverse[n*irow+icol] -= factor * AInverse[n*iPass+icol];
                    A[n*irow+icol] -= factor * A[n*iPass+icol];
                }
            }
        }
    }

    free(ac);
    return 1;
}
