#include "mbed.h"
#include "XNucleoIKS01A2.h"
#include "string"

#define CR 0x0D

using namespace std;

Serial pc(PA_2, PA_3, 115200); // seriale pc
Serial lora(PB_6,PA_10,115200);

static XNucleoIKS01A2 *board = XNucleoIKS01A2::instance(D14,D15,D4,D5); // scheda
static LSM303AGRMagSensor *magSensor = board->magnetometer; // magnetoscopio
static LPS22HBSensor *presSensor = board->pt_sensor;  // pressione
static HTS221Sensor *tempSensor= board->ht_sensor; // temperatura e umidita'
static LSM6DSLSensor *iNemo = board->acc_gyro; // giroscopio

enum DataType {PRESSURE = 0, MAG_FIELD, TEMPERATURE, HUMIDITY, ACCELERATION, ANG_VELOCITY, N_DATA_TYPES};
uint16_t ports[N_DATA_TYPES] = {15, 16, 17, 18, 19, 20};


void connect_to_lora();
void wait4join();
void modem_at_cmd(const string&);
void at_send(uint16_t, const string&, bool);

string hex_encode(const string&);

void enable_all_sensors();
void get_pressure();
void get_magnetic_field();
void get_temperature();
void get_humidity();
void get_acceleration();
void get_gyroscope();


float pressure = 0;
float temperature = 0;
float humidity = 0;
int assi_giroscopio[3];
int assi_accelerazione[3];
int assi_magnetometro[3];

int main() {

    enable_all_sensors();
    connect_to_lora();
    
    size_t counter = 0;
    int data_type = PRESSURE;
    
    char payload[256];
    while (1) {
        
        memset(payload, 0, 256);
        
        switch (data_type) {
            case PRESSURE:
                get_pressure();
                sprintf(payload, "{\"pressure\":%.6f}", pressure);
                break;
            case MAG_FIELD:
                get_magnetic_field();
                sprintf(payload, "{\"magx\":%d, \"magy\":%d, \"magz\":%d}", assi_magnetometro[0] , assi_magnetometro[1] , assi_magnetometro[2]);
                break;
            case TEMPERATURE:
                get_temperature();
                sprintf(payload, "{\"temperature\":%.6f}", temperature);
                break;
            case HUMIDITY:
                get_humidity();
                sprintf(payload, "{\"humidity\":%.6f}", humidity);
                break;
            case ACCELERATION:
                get_acceleration();
                sprintf(payload, "{\"accx\":%d, \"accy\":%d, \"accz\":%d}", assi_accelerazione[0] , assi_accelerazione[1] , assi_accelerazione[2]);
                break;
            case ANG_VELOCITY:
                get_gyroscope();
                sprintf(payload, "{\"angvelx\":%d, \"angvely\":%d, \"angvelz\":%d}", assi_giroscopio[0] , assi_giroscopio[1] , assi_giroscopio[2]);
                break;    
        }
        
        at_send(ports[data_type], payload, false);
        
        data_type = (data_type + 1) % N_DATA_TYPES;
        ++counter;
        
        wait(5.0);
    }
}


string hex_encode(const string& input) {
    static const char hex_digits[] = "0123456789ABCDEF";
    string output;
    output.reserve(input.length() * 2);
    for (int i = 0; i < input.length(); i++)
    {
        output += hex_digits[input[i] >> 4];
        output += hex_digits[input[i] & 15];
    }
    return output;
}

void at_send(uint16_t port, const string& payload, bool ack = false) {
    //string port_msg;
    char port_pref[12] = {"AT+SEND="};
    //port_msg.reserve(5); // max port 2^16-1 = 65545
    sprintf(port_pref + 8, "%d", port);
    //pc.printf(port_pref);
    string hex_payload = hex_encode(payload);
    //string test = (string)port_pref + "," + hex_payload + "," + (ack ? '1' : '0');
    //pc.printf(&test[0]);
    modem_at_cmd((string)port_pref + "," + hex_payload + "," + (ack ? '1' : '0'));
}

void modem_at_cmd(const string& buffer) {
    for(uint16_t i = 0; i < buffer.length(); i++) {
        lora.putc(buffer[i]);
        pc.putc(buffer[i]);
    }
    lora.putc(CR);
    
    pc.putc(CR);
    pc.printf("\n");
    
    char c = 0;
    do {
        if (lora.readable()) {
            c = lora.getc();
            pc.putc(c);
        }
    } while(c != ' ');
}

void wait4join(){
    char c = 0;
    do {
        if (lora.readable()) {
            c = lora.getc();
            pc.putc(c);
        }
    } while(c != 'd');
}

void connect_to_lora() {
    modem_at_cmd("AT");
    pc.printf("Establishing connection to LoRa\r\n");
    wait(1);
    
    modem_at_cmd("AT+APPEUI=0000000000000001");
    pc.printf("Inviato EUI\r\n");
    wait(1);
    
    modem_at_cmd("AT+AK=00000000000000000000000000000001");
    pc.printf("Inviato AK\r\n");
    wait(1);
    
    modem_at_cmd("AT+JOIN=1");
    pc.printf("Inviato JOIN\r\n");
    wait4join();
    
    modem_at_cmd("AT+DC=0");
    pc.printf("Disabled DC\r\n");
    wait(1);
    
    modem_at_cmd("AT+ADR=1");
    pc.printf("Enabled ADR\r\n");
    wait(1);
}

void enable_all_sensors() {
    presSensor->enable();
    magSensor->enable();
    tempSensor->enable();
    iNemo->enable_g();
    iNemo->enable_x();
}

void get_pressure() {
    presSensor->get_pressure(&pressure);
    pc.printf("pressure [mbar]:\t%f\r\n", pressure);
}

void get_magnetic_field() {
    magSensor->get_m_axes(assi_magnetometro);
    pc.printf("mag field [mgauss]:\t%d\t%d\t%d\n\r", assi_magnetometro[0] , assi_magnetometro[1] , assi_magnetometro[2]);
}

void get_temperature() {
    tempSensor->get_temperature(&temperature);
    pc.printf("temperature ['C]:\t%f\n\r", temperature);
}

void get_humidity() {
    tempSensor->get_humidity(&humidity);
    pc.printf("humidity [%%]\t\t%f\r\n", humidity);
}

void get_acceleration() {
    iNemo->get_x_axes(assi_accelerazione);
    pc.printf("acceleration [mg]:\t%d\t%d\t%d\n\r", assi_accelerazione[0] , assi_accelerazione[1] , assi_accelerazione[2]);
}

void get_gyroscope() {
    iNemo->get_g_axes(assi_giroscopio);
    pc.printf("ang velocity [rad/s]:\t%d\t%d\t%d\n\r", assi_giroscopio[0] , assi_giroscopio[1] , assi_giroscopio[2]);
}
