#include "pinmap.h"
#include "parameters.h"
#include "pulsegenerator.h"
#include "rtos.h"
#include "motion.h"



modes currentMode=VOO;
modes nextMode=VOO;

void timer_thread ();
void pace_controller ();

/*=== RTOS Required Components ===*/
Thread paceThread(osPriorityNormal);
Thread timerThread(osPriorityNormal);

uint32_t pace_period_ms;
uint32_t next_pace_period_ms;


uint8_t initialized = 0;

/******************************************
 *
 *       Pacing Mode Functions
 *
 ******************************************/

modes set_pacing_mode (modes newMode) {
    nextMode = newMode;
    return currentMode;    
}

uint32_t set_pace_period_ms(uint32_t new_pace_period_ms){
    next_pace_period_ms = new_pace_period_ms;
    return pace_period_ms;
}

/*
Deactivates both pacing circuits
Cancels any existing pace event
*/
void pace_charge_shutdown () {
    /* Stage 1: Switch-OFF/Ground-ON */
    atr_pace_ctrl =  LOW;
    vent_pace_ctrl = LOW;
    atr_grnd_ctrl  = HIGH;
    vent_grnd_ctrl = HIGH;
    wait_ms(50);
    

    /* Stage 2: Ref PWM LOW */
    pacing_ref_pwm = LOW;
    wait_ms(100);

    /* Stage 3: Charge CTRL off */
    pace_charge_ctrl = LOW;
    wait_ms(10);
    
    /* Stage 4: Ground OFF */
    atr_grnd_ctrl  = LOW;
    vent_grnd_ctrl = LOW;
}

void pulsegenerator_initialize () {
    if (initialized) return;
    pace_charge_shutdown ();
    initialized = 1;
}

/*
  Prep the circuit for Ventricle Pace Event
*/
void vent_pace_prime () {
    /* Redundant code to close ground switches, 
     * prevents accidentally shorting the pacemaker
     */ 
    atr_grnd_ctrl  = LOW;
    vent_grnd_ctrl = LOW;
    
    /* Stage 1: Ref PWM On */
    pacing_ref_pwm = HIGH;
    wait_ms(10);

    /* Stage 2: Charge CTRL on */
    pace_charge_ctrl = HIGH;
    wait_ms(10);
}

/*
  Prep the circuit for Atrium Pace Event
*/
void atr_pace_prime () {
    /* Redundant code to close ground switches, 
     * prevents accidentally shorting the pacemaker
     */ 
    atr_grnd_ctrl  = LOW;
    vent_grnd_ctrl = LOW;
    
    /* Stage 4: Ref PWM On */
    pacing_ref_pwm = HIGH;
    wait_ms(10);

    /* Stage 5: Charge CTRL on */
    pace_charge_ctrl = HIGH;
    wait_ms(10);
}

void pace_vent (uint16_t pulse_width_us) {
    vent_pace_prime();

    // Begin Pace Event Output
    vent_pace_ctrl = HIGH;
    
    // Pace Duration
    wait_us(pulse_width_us);
    
    // Shut off Pace
    vent_pace_ctrl = LOW;
    
    pace_charge_shutdown (); 
}

void pace_atr (uint16_t pulse_width_us) {
    atr_pace_prime();

    // Begin Pace Event Output
    atr_pace_ctrl = HIGH;
    
    // Pace Duration
    wait_us(pulse_width_us);
    
    // Shut off Pace
    atr_pace_ctrl = LOW;
        
    pace_charge_shutdown (); 
}

void pace_VOO () {
    pace_vent (get_vent_pulse_width_us());
}

void pace_AOO () {
    pace_atr (get_atr_pulse_width_us());
}

void pace_VOOR() {
    if(isMotionThresholdExceeded()){
            //set_vent_pulse_width_us();//pace faster
            pace_VOO();
            resetMotionDetection();
        }
        else {
            //set_vent_pulse_width_us();//pace with previous value
        }
}

void begin_pace () {
    if (!initialized) pulsegenerator_initialize ();
    pace_period_ms = get_lower_rate_limit_period_ms();
    
    paceThread.start(pace_controller);
    timerThread.start(timer_thread);
}

void timer_thread () {
    while (true) {
        paceThread.signal_set(0x4);
        
        wait_ms(pace_period_ms - 180); //delay for every pace call
    }
}

void pace_controller () {
    while(true) {
        Thread::signal_wait(0x0, osWaitForever); //Sleep until next cycle
        
        switch (currentMode) {
            case VOO  : pace_VOO();  break;
            case VOOR : pace_VOOR(); break;
            case AOO  : pace_AOO();  break;
            default   : pace_VOO();  break;
        }
    }
} 