#include "mbed.h"
#include <iostream>
#include <math.h> 

//General   
    #define TI  .001f           //1kHz sample time
    #define LEDOFF 0.0f         //LED low value
    #define LEDON 1.0f          //LED high value
    #define DIHIGH 0.5f         //High value of a digital in pin
    #define BATTRATIO 3.9625f   //Battery Input ratio
    #define BATTCUTOFF 1.0f     //Battery Cutoff threshold
    #define LOOPTIME 0.75f      //Time(s) between loops
    #define ADCRATIO 0.30303f   //Ratio from port input to ADC value
    #define ZERO 0.0f           //Zero
    #ifndef M_PI
        #define M_PI 3.14f      //Defines Pi
    #endif

    #define BLUETOOTHBAUDRATE 115200 //Communication rate of the micro-controller to the Bluetooth module
    #define SERIALTXPORT PTE0   //Tx Pin (Sends information to serial device)
    #define SERIALRXPORT PTE1   //Rx Pin (Receives information from serial

//Driving
    #define dKP  5.1491f        //Proportional Gain
    #define dKI  119.3089f      //Integral Gain
    #define MOTORPWM 0.000034f  //PWM Frequency
    #define MOTORMAX 0.9f       //Max Value of Buck Converter
    #define MOTORMIN 0.0f       //Minimum Value of Buck Converter
    #define Speed 0.625f        //Speed Reference
    #define sSpeed 0.80f        //Straight Speed

//Steering
    #define SERVOPWM 0.02f      //Servo Frequency
    #define sKP 3.5f            //Proportional Gain
    #define sKD 0.003f          //Derivative Gain was .01
    #define sKI 0.025f          //Derivative Gain was .01
    #define ssKP 1.4f
    #define ssKD 0.00001f
    #define ssKI 0.0f
    #define FULLLEFT 0.1f       //PWM value of servo turning left
    #define NLANDMARKS 10       //Number of Landmarks
    #define DCINTERCEPT 0.064f  //Duty cycle intercept
    #define DCSLOPE 0.03f       //Duty cycle slope
    #define tDCINTERCEPT 0.06f
    #define tDCSLOPE 0.03f
    #define TOL 0.0005f         //Tolerance
    #define MIN -40.0*(M_PI/180)
    #define MAX -40.0*(M_PI/180)
    #define OFFTRACKRIGHT 0.401f//Right Off Track Threshold
    #define OFFTRACKLEFT 0.409f //Left Off Track Threshold
    #define STEERINGREF -0.1f   //Centerpoint reference for steering

//Pins
    //_p - pin
    //AI - Analog in
    //PWM - PWM out
    //DO - Digital out
    //IN - Interupt in
    //led - Onboard LED
//General Pins
    Serial bt(SERIALTXPORT, SERIALRXPORT); //(TX, RX) Serial 
    PwmOut led_red(LED1);
    PwmOut led_blue(LED3);
    PwmOut led_green(LED2);
    AnalogIn AI_pBattMon(A4);
    DigitalIn DI_pArrowheadLeft(PTD5);//PTA1
    DigitalIn DI_pArrowheadRight(PTA13);//PTA2
    //State Buttons
    DigitalIn _RunButton(PTA4);
    DigitalIn _StopButton(PTA5);
    DigitalIn _BumperStop(PTD3);

//Driving Pins
    AnalogIn AI_pSpeedReference(A0);
    AnalogIn AI_pTacho(A1);
    PwmOut PWM_pMotorControl(PTD0);
    DigitalOut DO_pBreakSignal(PTC9);//D5

//Steering Pins
    AnalogIn AI_pLeftSensors(A2);
    AnalogIn AI_pRightSensors(A3);
    PwmOut PWM_pServo(PTE20);//A13

//Binary LED
    DigitalOut DO_pFirstBinaryLED(A5);
    DigitalOut DO_pSecondBinaryLED(PTE21);
    DigitalOut DO_pThirdBinaryLED(PTD2);
    DigitalOut DO_pFourthBinaryLED(PTE31);
    
//Hardware Interupts
    Ticker DrivingUpdate;
    Ticker SteeringUpdate;
    Ticker LoopUpdate;


//Data Values
    //_f - Float
    //_n - Integer
    //_b - Boolean
    
//General Values
    int _nCounter;
    bool _bArrowHitRight;
    bool _bArrowHitLeft;

//Driving Values
    float _fDrivingErrorArea;
    float _fDrivingError;
    float _fDrivingControllerOutput;
    float _fMotorReference;
    
//Steering
    float _fSteeringControllerOut;
    float _fSteeringDutyCycle;
    float _fSteeringError;
    float _fSteeringErrorArea;
    float _fSteeringFeedbackPrev;

//States
    enum States {RUN, WAIT, STOP, EMPTY};
    States _oNextState;
    States _oCurrentState;

//General Methods
    void Init();
    void UpdateLoop();
    void UpdateLowBatteryStop();
    void StopMethod();
    void WaitMethod();
    void RunMethod();
    void RunButtonMethod();
    void UpdateLeft();
    void UpdateRight();
    void UpdateBinaryCounter();
    void CheckLocation();
    void OnboardLEDColor(float, float, float);

//Driving Methods
    void UpdateDriveControl();
    void SetMotor(float);
    void BreaksOn();
    void BreaksOff();

//Steering Methods
    void UpdateSteeringControl();
    void UpdateSteeringControlStraight();

//State Methods
    void TriggerSwitch();
    void StopMethod();
    void WaitMethod();
    void RunMethod();

int main() {
    cout << "\rCarStart" << endl;
    Init();
}

void Init() //Initializer
{
    //Variable Init
    _fDrivingError = ZERO;
    _fDrivingErrorArea = ZERO;
    _fMotorReference = ZERO;
    _fSteeringControllerOut = ZERO;
    _fSteeringDutyCycle= ZERO;
    _fSteeringErrorArea= ZERO;
    _fSteeringError = ZERO;
    _fSteeringFeedbackPrev = ZERO;
    _nCounter = ZERO;
    _oNextState = EMPTY;
    _oCurrentState = EMPTY;
    _bArrowHitRight = false;
    _bArrowHitLeft = false;
    
    //Motor Init
    PWM_pMotorControl.period(MOTORPWM);
    SetMotor(ZERO); //Turns off the motor
    StopMethod();
    TriggerSwitch();
    BreaksOn();
    
    //Steering Init
    PWM_pServo.period(SERVOPWM);
    
    //Interupt and Bluetooth Init
    bt.baud(BLUETOOTHBAUDRATE);
    DrivingUpdate.attach(&UpdateDriveControl, TI);
    SteeringUpdate.attach(&UpdateSteeringControl, TI);
    LoopUpdate.attach(&UpdateLoop, LOOPTIME);
}

void UpdateLoop() //
{
    //State Button Check
    if(_StopButton.read()>=DIHIGH)
        StopMethod();
    if(_RunButton.read()>=DIHIGH)
        TriggerSwitch();
    if(_BumperStop.read()<=DIHIGH)
        StopMethod();
    
    //Reseting Arrowhead Values
    _bArrowHitRight = false;
    _bArrowHitLeft = false;
    
    //Check Battery Voltage
    UpdateLowBatteryStop();
    
    //Bluetooth Print
    bt.printf("\rGroup 2: USC Football team\n Josh: %f %f %f     \033[F\033[F",DI_pArrowheadLeft.read(), DI_pArrowheadRight.read(), AI_pRightSensors.read());//\033[F
} 
    
void UpdateDriveControl()
{
    if(_oCurrentState == RUN )
    {
        //Localized speed value
        if(_nCounter == ZERO) _fMotorReference = sSpeed;
        else _fMotorReference = Speed;
    
        //Error and Area Calculation
        _fDrivingError = _fMotorReference - AI_pTacho.read();
        _fDrivingErrorArea += TI*_fDrivingError;
    
        //PI Value
        _fDrivingControllerOutput = dKP*_fDrivingError+ dKI*_fDrivingErrorArea;
    }
    SetMotor(_fDrivingControllerOutput);    
    CheckLocation();

}
void CheckLocation()
{
    if(DI_pArrowheadLeft.read() > DIHIGH && DI_pArrowheadRight < DIHIGH && !_bArrowHitLeft){
        if(_nCounter >=NLANDMARKS) {
           _nCounter = ZERO;
        } else {
            _nCounter++;
        }
        UpdateBinaryCounter();
        _bArrowHitLeft = true;
    }
   else if(DI_pArrowheadLeft.read() < DIHIGH && DI_pArrowheadRight > DIHIGH && !_bArrowHitRight){
        if(_nCounter >=NLANDMARKS) {
           _nCounter = ZERO;
        } else {
            _nCounter++;
        }
        UpdateBinaryCounter();
        _bArrowHitRight = true;
    }
    else if(DI_pArrowheadLeft.read() > DIHIGH && DI_pArrowheadRight > DIHIGH ){
        _nCounter = ZERO;
        _fSteeringErrorArea =ZERO;
        UpdateBinaryCounter();
        _bArrowHitLeft = true;
        _bArrowHitRight = true;
    }
    if( DI_pArrowheadRight < DIHIGH){
        _bArrowHitRight = false;
    }
    if( DI_pArrowheadLeft < DIHIGH){
        _bArrowHitLeft = false;
    }
}

void UpdateSteeringControl(void)
{
    if(_oCurrentState != STOP){
        float feedback = AI_pLeftSensors.read() - AI_pRightSensors.read();
        float reference = 0.1f;
        
        //Error, Slope, and Area Calculation
        _fSteeringError = reference - feedback;       
        float feedbackChange = (feedback - _fSteeringFeedbackPrev)/TI;
        _fSteeringErrorArea += TI*_fSteeringError;
    
        //Localized PID Calculation
        if(_nCounter <= 1){ 
            _fSteeringControllerOut = ssKP*_fSteeringError + ssKD*feedbackChange + ssKI*_fSteeringErrorArea;
            float controllerOutNorm = (_fSteeringControllerOut + M_PI/2.0f)/M_PI;
            _fSteeringDutyCycle = DCSLOPE*controllerOutNorm + DCINTERCEPT;
        } 
        else {
         _fSteeringControllerOut = sKP*_fSteeringError + sKD*feedbackChange + sKI*_fSteeringErrorArea;
         float controllerOutNorm = (_fSteeringControllerOut + M_PI/2.0f)/M_PI;
        _fSteeringDutyCycle = tDCSLOPE*controllerOutNorm + tDCINTERCEPT;
        }

        if (abs(_fSteeringDutyCycle-PWM_pServo.read()) > TOL)
         PWM_pServo.write(_fSteeringDutyCycle);
        
        _fSteeringFeedbackPrev = feedback;
    
        //Off Track Fault
        if (AI_pLeftSensors.read() < OFFTRACKLEFT && AI_pRightSensors.read() < OFFTRACKRIGHT){
            StopMethod();
        }
    } 
}



void UpdateLowBatteryStop()
{
    float val = ZERO;       // variable for the A/D value
    float pinVoltage = ZERO; // variable to hold the calculated voltage
    float batteryVoltage = ZERO; 
    val = AI_pBattMon.read();    // read the voltage on the divider  
  
    pinVoltage = val * ADCRATIO;       //  Calculate the voltage on the A/D pin                            
                                    
    batteryVoltage = pinVoltage * BATTRATIO;    //Ratio  for the voltage divider
    if(batteryVoltage < BATTCUTOFF){
      cout << "\n\rBattery Voltage is too low. Stop Method " << endl;
      StopMethod();
    } 
    else{
        cout << "\n\rBattery Voltage is: " << batteryVoltage << endl;
    }
  
}

void SetMotor(float speed)
{
    if(speed< MOTORMIN) //Speed is above floor
    speed = MOTORMIN;
    else if(speed> MOTORMAX) //Speed is above floor
    speed = MOTORMAX;
 
    PWM_pMotorControl.write(speed);
}

void BreaksOn()
{
    SetMotor(MOTORMIN); //Sets speed to Zero
    _fDrivingErrorArea = ZERO; //Resets Error area to zero
    _fSteeringErrorArea = ZERO;//Resets Error area to zero
    DO_pBreakSignal = 1; //Turns on breaks
    PWM_pServo= FULLLEFT; //Turns servo to left
}

void BreaksOff()
{
    SetMotor(MOTORMIN); //Sets speed to Zero
    _fSteeringErrorArea = ZERO; //Resets Error area to zero
    _fDrivingErrorArea = ZERO; //Resets Error area to zero
    DO_pBreakSignal = ZERO; //Turns off brakes
}

void WaitMethod()
{
    _oCurrentState = WAIT;
    BreaksOff();
    OnboardLEDColor(.125f, 1.0f, 0.125f); //Sets to Yellow
    _oNextState = RUN;
}

void RunMethod()
{
    _oCurrentState = RUN;
    BreaksOff();
    OnboardLEDColor(1.0f, 1.0f, 0.25f); //Sets to Green
    _oNextState = WAIT;
}

void StopMethod()
{   
    _oCurrentState = STOP;
    BreaksOn();
    OnboardLEDColor(0.25f, 1.0f, 1.0f); //Sets to Red
    _oNextState = WAIT;
}

void TriggerSwitch()
{
    switch(_oNextState){
    case STOP:
    StopMethod();
    break;
    
    case  WAIT:
    WaitMethod();
    break;
    
    case RUN:
    RunMethod();
    break;
    
    case EMPTY:
    OnboardLEDColor(0.25f, 0.25f, 0.25f); //Sets to White
    break;
    }
}

void UpdateBinaryCounter()
{
    switch(_nCounter){
        case 0:
            DO_pFirstBinaryLED = LEDOFF;
            DO_pSecondBinaryLED = LEDOFF;
            DO_pThirdBinaryLED = LEDOFF;
            DO_pFourthBinaryLED = LEDOFF;
        break;
        case 1:
            DO_pFirstBinaryLED = LEDON;
            DO_pSecondBinaryLED = LEDOFF;
            DO_pThirdBinaryLED = LEDOFF;
            DO_pFourthBinaryLED = LEDOFF;
        break; 
        case 2:
            DO_pFirstBinaryLED = LEDOFF;
            DO_pSecondBinaryLED = LEDON;
            DO_pThirdBinaryLED = LEDOFF;
            DO_pFourthBinaryLED = LEDOFF;
        break;
        case 3:
            DO_pFirstBinaryLED = LEDON;
            DO_pSecondBinaryLED = LEDON;
            DO_pThirdBinaryLED = LEDOFF;
            DO_pFourthBinaryLED = LEDOFF;
        break;    
        case 4:
            DO_pFirstBinaryLED = LEDOFF;
            DO_pSecondBinaryLED = LEDOFF;
            DO_pThirdBinaryLED = LEDON;
            DO_pFourthBinaryLED = LEDOFF;
        break;
        case 5:
            DO_pFirstBinaryLED = LEDON;
            DO_pSecondBinaryLED = LEDOFF;
            DO_pThirdBinaryLED = LEDON;
            DO_pFourthBinaryLED = LEDOFF;
        break;    
        case 6:
            DO_pFirstBinaryLED = LEDOFF;
            DO_pSecondBinaryLED = LEDON;
            DO_pThirdBinaryLED = LEDON;
            DO_pFourthBinaryLED = LEDOFF;
        break;
        case 7:
            DO_pFirstBinaryLED = LEDON;
            DO_pSecondBinaryLED = LEDON;
            DO_pThirdBinaryLED = LEDON;
            DO_pFourthBinaryLED = LEDOFF;
        break;
        case 8:
            DO_pFirstBinaryLED = LEDOFF;
            DO_pSecondBinaryLED = LEDOFF;
            DO_pThirdBinaryLED = LEDOFF;
            DO_pFourthBinaryLED = LEDON;
        break;
        case 9:
            DO_pFirstBinaryLED = LEDON;
            DO_pSecondBinaryLED = LEDOFF;
            DO_pThirdBinaryLED = LEDOFF;
            DO_pFourthBinaryLED = LEDON;
        break;
        case 10:
            DO_pFirstBinaryLED = LEDOFF;
            DO_pSecondBinaryLED = LEDON;
            DO_pThirdBinaryLED = LEDOFF;
            DO_pFourthBinaryLED = LEDON;
        break;
        case 11:
            DO_pFirstBinaryLED = LEDON;
            DO_pSecondBinaryLED = LEDON;
            DO_pThirdBinaryLED = LEDOFF;
            DO_pFourthBinaryLED = LEDON;
        break;
        case 12:
            DO_pFirstBinaryLED = LEDOFF;
            DO_pSecondBinaryLED = LEDOFF;
            DO_pThirdBinaryLED = LEDON;
            DO_pFourthBinaryLED = LEDON;
        break;
        case 13:
            DO_pFirstBinaryLED = LEDON;
            DO_pSecondBinaryLED = LEDOFF;
            DO_pThirdBinaryLED = LEDON;
            DO_pFourthBinaryLED = LEDON;
        break;
        case 14:
            DO_pFirstBinaryLED = LEDOFF;
            DO_pSecondBinaryLED = LEDON;
            DO_pThirdBinaryLED = LEDON;
            DO_pFourthBinaryLED = LEDON;
        break;
        case 15:
            DO_pFirstBinaryLED = LEDON;
            DO_pSecondBinaryLED = LEDON;
            DO_pThirdBinaryLED = LEDON;
            DO_pFourthBinaryLED = LEDON;
        break;
        default:
            DO_pFirstBinaryLED = LEDOFF;
            DO_pSecondBinaryLED = LEDOFF;
            DO_pThirdBinaryLED = LEDOFF;
            DO_pFourthBinaryLED = LEDOFF;
        break;               
    } 
}

void OnboardLEDColor(float fRed= 1.0f, float fGreen = 1.0f, float fBlue = 1.0f)
{
    led_red = fRed;
    led_blue = fGreen;
    led_green = fBlue;
}