/**
 * @file   : communication.h (1.0)
 * @brief  : manage communication
 * @author : Shinnosuke KOIKE
 * @date   : 2015/08/06
 */

#ifndef COMMUNICATION_H
#define COMMUNICATION_H

#include "mbed.h"

class CommuMaster {
public:
    CommuMaster(PinName sda, PinName scl);
    ~CommuMaster();
    void init(int correctFlequency, char addr);
    void receiveData(char data[]);

private:
    I2C* i2cmaster;
    char addr;
};

class CommuSlave {
public:
    CommuSlave(PinName sda, PinName scl);
    ~CommuSlave();
    void init(int correctFlequency, char addr);
    void sendData(char data[]);

private:
    I2CSlave* i2cslave;
};

// initialize
CommuMaster::CommuMaster(PinName sda, PinName scl) {
    i2cmaster = new I2C(sda, scl);
}

CommuMaster::~CommuMaster() {
    delete i2cmaster;
}

// set frequency and address
void CommuMaster::init(int correctFrequency, char addr) {
    i2cmaster->frequency(correctFrequency);
    this->addr = addr;
}

// receive some data
void CommuMaster::receiveData(char data[]) {
    i2cmaster->read(this->addr | 1, data, 8, false);
}

// initialize
CommuSlave::CommuSlave(PinName sda, PinName scl) {
    i2cslave = new I2CSlave(sda, scl);
}

CommuSlave::~CommuSlave() {
    delete i2cslave;
}

// set frequency and address
void CommuSlave::init(int correctFrequency, char addr) {
    i2cslave->frequency(correctFrequency);
    i2cslave->address(addr);
}

// send some data
void CommuSlave::sendData(char data[]) {
    i2cslave->write(data, 8);
}

#endif

/**
 * example program(master)

#include "mbed.h"
#include "communication.h"

#define DATA_NUM    8

int main(void) {
    CommuMaster master(D5, D6);
    int  freq = 2000;
    char addr = 0xA0;
    char data[DATA_NUM];
    master.init(freq, addr);

    while (1) {
        master.receiveData(data);
        for (int i = 0; i < DATA_NUM; i++) {
            pc.printf("%d ", data[i]);      // display "1 1 0 0 0 0 0 1 "
        }
        pc.putchar('\n');
    }
}

 * example program(slave)

#include "mbed.h"
#include "communication.h"

int main(void) {
    CommuSlave slave(D5, D6);
    int  freq = 2000;
    char addr = 0xA0;
    char data[8] = {1, 1, 0, 0, 0, 0, 0, 1};
    slave.init(freq, addr);

    while (1) {
        slave.sendData(data);
    }
}
*/
