#include "mbed.h"

//走行用プログラム
//モータードライバー＋回転数計測プログラム
//走るためのプログラム+道が白か黒かを判定するプログラム
//A入力に右のモーター、B入力に左のモーター
//58mm径タイヤの一周は約182.21mm

InterruptIn enc_R(D11);//フォトインタラプタ右
InterruptIn enc_L(D12);//フォトインタラプタ左

AnalogIn sensor_R(A0);//進行方向右のフォトリフレクタセンサー    
AnalogIn sensor_L(A2);//進行方向左のセンサー
AnalogIn sensor_C(A1);//中央配置のセンサー
int R_on_off =0;//左右の白黒
int L_on_off =0;
int C_on_off =0;
int pre_R_on_off =0;//1つ前の過程の、センサで決めた白黒（白なら0、黒なら1）
int pre_L_on_off =0;
int pre_C_on_off =1;//スタート時、中央センサーのみ黒の上

DigitalOut myled(A5);//200mmごとに光らせるLED用

PwmOut AIN1(D6);//右モーター
PwmOut AIN2(D5);
PwmOut BIN1(D9);//左モーター
PwmOut BIN2(D10);//D9とD10は回路上で逆にしないと、なぜか回転方向が逆転

//距離、時間関係の変数、定数
float distance_per_frame = 0.0f;//距離の初期値
float total_distance = 0.0f;//距離のトータル
float delta = 0.01;//計測、判断時間の周期

//デューティー比関連の変数、定数
const float duty_basic = 0.13f;//基本のデューティー比
const float duty_ad = 0.02f;//右左折時に作用
const float duty_ad_sharp = 0.03f;//急カーブ時に作用
const float duty_ignition = 0.20f;//急カーブ修了時にスピードアップ   
float R_duty =0.0f;
float L_duty =0.0f;

//回転数関連の変数、定数
float R_rot=0;//右タイヤの１秒間あたり回転数
float L_rot=0;
float pre_R_rot=0;//1つ前の過程の右モーターの累計回転数
float pre_L_rot=0;
float R_target_rot=0;//１秒間あたりの右目標タイヤ回転数
float L_target_rot=0;//１秒間あたりの左目標タイヤ回転数
float counter_R =0;//右モーターの累計回転数
float pre_counter_R = 0;//1つ前の過程の右のモーターの累計回転数
float counter_L =0;
float pre_counter_L = 0;

//PID制御関係の変数、定数
const float KP = 0;
const float KI = 0;
const float KD = 0;
float R_e=0;//右の偏差
float L_e=0;//左の偏差
float pre_R_e=0;//1つ前の過程の右の偏差
float pre_L_e=0;
float R_e_differential=0;//右の偏差の微分
float L_e_differential=0;
float R_e_integral=0;//右の偏差の積分
float L_e_integral=0;

//フォトリフレクタセンサ関連の定数

const float L_black = 2.6;//左センサの黒値
const float L_white = 0.3;//白値
const float R_black = 1.2;//右センサの黒値
const float R_white = 0.2;//
const float C_black = 1.5;//中心センサの黒値
const float C_white = 0.2;//


float L_target = (L_black + L_white) / 2;
float R_target = (R_black + R_white) / 2;
float C_target = (C_black + C_white) / 2;
/*
const float L_target = 1.30;//この値より大きければ黒と判定
const float R_target = 1.30;//この値より大きければ黒と判定
const float C_target = 1.30;//この値より大きければ黒と判定
*/


//PID制御で使う微分量と積分量の計算
void differential_integral_culculus(void){
    
    R_e_differential = (R_e-pre_R_e)/delta;
    L_e_differential = (L_e-pre_L_e)/delta;
    
    R_e_integral += (pre_R_e+R_e)/2.0f*delta;
    L_e_integral += (pre_L_e+L_e)/2.0f*delta;
    
}

//モーター回転数計測
void event_handler_R(void){//右モーター回転数計測
    counter_R++;
}

void event_handler_L(void){//左モーター
    counter_L++;
}


//ブレーキ
void motorStop(PwmOut IN1,PwmOut IN2) {
    IN1 = 1;
    IN2 = 1;
}

//正転
void motorForward(PwmOut IN1,PwmOut IN2,float duty) {
    IN1 = duty;
    IN2 = 0;
}

//逆転
void motorReverse(PwmOut IN1,PwmOut IN2,float duty) {
    IN1 = 0;
    IN2 = duty;
}
//空転
void motorIdling(PwmOut IN1,PwmOut IN2) {
    IN1 = 0;
    IN2 = 0;
}

//停止
void machine_Stop(PwmOut AIN1,PwmOut AIN2,PwmOut BIN1,PwmOut BIN2) {
    motorStop(AIN1,AIN2);
    motorStop(BIN1,BIN2);
}

//前進
void machine_Forward(PwmOut AIN1,PwmOut AIN2,PwmOut BIN1,PwmOut BIN2) {
    motorReverse(AIN1,AIN2,R_duty);//逆転のほうが制御しやすいので前進時は逆転を使用
    motorReverse(BIN1,BIN2,L_duty);
}

//後退
void machine_Back(PwmOut AIN1,PwmOut AIN2,PwmOut BIN1,PwmOut BIN2) {
    machine_Stop(AIN1,AIN2,BIN1,BIN2);
    wait(2);
    motorForward(AIN1,AIN2,R_duty);
    motorForward(BIN1,BIN2,L_duty);
}

//右カーブ
void Right_rotation(PwmOut AIN1,PwmOut AIN2,PwmOut BIN1,PwmOut BIN2) {

    motorReverse(AIN1,AIN2,R_duty);
    motorReverse(BIN1,BIN2,L_duty);
}

//左カーブ
void Left_rotation(PwmOut AIN1,PwmOut AIN2,PwmOut BIN1,PwmOut BIN2) {

    motorReverse(AIN1,AIN2,R_duty);
    motorReverse(BIN1,BIN2,L_duty);
}


//右急カーブ
void Right_high_rotation(PwmOut AIN1,PwmOut AIN2,PwmOut BIN1,PwmOut BIN2) {

    motorIdling(AIN1,AIN2);
    motorReverse(BIN1,BIN2,L_duty);
}

//左急カーブ
void Left_high_rotation(PwmOut AIN1,PwmOut AIN2,PwmOut BIN1,PwmOut BIN2) {
    
    motorReverse(AIN1,AIN2,R_duty);
    motorIdling(BIN1,BIN2);
}

enum branch_mode{//制御の状態記号
    STRAIGHT =0,
    RIGHT,
    LEFT,
    SHARP_R,
    SHARP_L,
    REVERSE
    };
    
int mode;//展開中の制御状態記号の格納
int pre_mode;//一つ前の状態記号の格納:線を外れた際に引用

        
enum restart_ignition{//減速後の復帰のための操作に使用する信号
    no=0,
    yes
    };
    
int re_ignition;//上restart_ignitionの格納

/*
restart_ignitionについて

[re_ignition = no]では、通常の走行状態
[re_ignition = ready]は急カーブ中の状態
この状態で中央のセンサーが黒を感知すると[re_ignition = yes]となり、
一時的にduty比を上げて、走行させる。その後[re_ignition = no]に戻る。
*/

//センサーの色判定と走行モード決め
void Check_reflecter(AnalogIn sensor_R, AnalogIn sensor_L,AnalogIn sensor_C){//リフレクタの黒白判定＆移動方向の決定    
    printf("sensor_Right:L432[%.3f]>\n",sensor_R*3.3F);//電圧測定
    printf("sensor_Left:L432[%.3f]>\n",sensor_L*3.3F);//電圧測定
    printf("sensor_Center:L432[%.3f]>\n\n",sensor_C*3.3F);//電圧測定  
    
    
    if(sensor_R*3.3F>R_target)
    {
        R_on_off=1;
    }else{
        R_on_off=0;
    }
    if(sensor_L*3.3F>L_target)
    {
        L_on_off=1;
    }else{
        L_on_off=0;
    }
    if(sensor_C*3.3F>C_target)
    {
        C_on_off=1;
    }else{
        C_on_off=0;
    }
    printf("Judgement of sensor L[%d] C[%d] R[%d]\n",L_on_off,C_on_off,R_on_off);//[1]なら黒判定
    
    
    if(L_on_off==0 && C_on_off==0 && R_on_off==0)//線の外
    {
        printf("I'm going out of line! Previous condition is %d \n",pre_mode);
        mode = pre_mode;
    }
    else if(L_on_off==1 && C_on_off==0 && R_on_off==0)//急左折
    {
        mode = SHARP_L;
        printf("SHARP_L\n");
    }
    else if(L_on_off==0 && C_on_off==0 && R_on_off==1)//急右折
    {
        mode = SHARP_R;
        printf("SHARP_R\n");
    }
    else if(L_on_off==1 && C_on_off==0 && R_on_off==1)//黒白黒、エラー、back
    {
        mode = REVERSE;
        printf("REVERSE\n");
    }
    else if(L_on_off==0 && C_on_off==1 && R_on_off==0)//前進
    {
        mode = STRAIGHT;
        printf("STRAIGHT\n");
    }
    else if(L_on_off==1 && C_on_off==1 && R_on_off==0)//左折
    {
        mode = LEFT;
        printf("LEFT\n");
    }
    else if(L_on_off==0 && C_on_off==1 && R_on_off==1)//右折
    {
        mode = RIGHT;
        printf("RIGHT\n");
    }
    else if(L_on_off==1 && C_on_off==1 && R_on_off==1)//all black（空中or十字部)よって直進
    {
        mode = STRAIGHT;
        printf("STRAIGHT\n");
    }
    
    if(pre_C_on_off==0 && C_on_off==1){//中央センサが白から黒に変わったとき
        re_ignition = yes;//急カーブが終了時のみ、一瞬加速
    }
       
}



//走行モードに応じて、右と左のタイヤの1秒間あたりの目標回転数,デューティー比を設定する
void set_target_rot_and_duty_value(void){

    if(mode==STRAIGHT){
        R_target_rot = 7;
        L_target_rot = 7;
        R_duty = duty_basic;
        L_duty = duty_basic;
        }
    else if(mode == RIGHT){
        R_target_rot = 8;
        L_target_rot = 8;
        R_duty = duty_basic-duty_ad;
        L_duty = duty_basic+duty_ad;
    }
    else if(mode == LEFT){
        R_target_rot = 7;
        L_target_rot = 7;
        R_duty = duty_basic+duty_ad;
        L_duty = duty_basic-duty_ad;
    }
    else if(mode == SHARP_R){
        R_target_rot = 7;
        L_target_rot = 7;
        R_duty = duty_basic-duty_ad_sharp;
        L_duty = duty_basic+duty_ad_sharp;
    }
    else if(mode == SHARP_L){
        R_target_rot = 8;
        L_target_rot = 8;
        R_duty = duty_basic+duty_ad_sharp;
        L_duty = duty_basic-duty_ad_sharp;
        
    }
    else if(mode == REVERSE){
        R_target_rot = 8;
        L_target_rot = 8;
        R_duty = duty_basic;
        L_duty = duty_basic;
    }
    
    printf("Base R_duty value:%f\n",R_duty);
    printf("Base L_duty value:%f\n",L_duty);
    
    pre_mode = mode;
    pre_C_on_off = C_on_off;
    pre_R_on_off = R_on_off;
    pre_L_on_off = L_on_off;
}    
    
    
//条件分岐、与えられた走行パターンとデューティー比を用いて走行
void Conditional_branch(void)//移動方向に対する操作
{
    /*if (re_ignition == yes){//急カーブ後のイグニッション
        machine_Forward(AIN1,AIN2,BIN1,BIN2,duty_ignition,duty_ignition);
        wait(0.001);
        re_ignition = no;
        printf("Fireeeeeeee!\n");
        }
    */    
    if (mode == STRAIGHT)
    {
        machine_Forward(AIN1,AIN2,BIN1,BIN2);
    }
    else if (mode == LEFT)
    {
        Left_rotation(AIN1,AIN2,BIN1,BIN2);
    }
    else if (mode ==RIGHT)
    {
        Right_rotation(AIN1,AIN2,BIN1,BIN2);
    }
    else if (mode == SHARP_L)
    {
        Left_high_rotation(AIN1,AIN2,BIN1,BIN2);
    }
    else if (mode == SHARP_R)
    {
        Right_high_rotation(AIN1,AIN2,BIN1,BIN2);
    }
    else if (mode == REVERSE)
    {
        machine_Back(AIN1,AIN2,BIN1,BIN2);
    }
    printf("ignition:%d\n",re_ignition);
}
        



//回転数によるPID制御
void PID_Control(void){

    
    R_rot = (counter_R-pre_counter_R)/24.0f/38.2f/delta;//一秒間当たりの右タイヤ回転数の計算
    L_rot = (counter_L-pre_counter_L)/24.0f/38.2f/delta;
    
    R_e = R_rot-R_target_rot;
    L_e = L_rot-L_target_rot;
    differential_integral_culculus();
   
    R_duty = R_duty + KP*R_e - KD*R_e_differential + KI*R_e_integral;
    L_duty = L_duty + KP*L_e - KD*L_e_differential + KI*L_e_integral;
    printf("After corrected R_duty value:%f\n",R_duty);
    printf("After corrected L_duty value:%f\n",L_duty);    
    
    Conditional_branch();
}


//距離計算関数
void distance_calculate (void)
{
    distance_per_frame = (R_rot+L_rot)/2.0f*182.12f;//右の回転数と左の回転数の平均に、一周の距離をかける
    total_distance += distance_per_frame;
    printf("Total Distance [%f]     Distance per Frame [%f]\n",total_distance,distance_per_frame);
    printf("Left and right rotation speed per frame R[%f]  L[%f]\n",R_rot,L_rot);
    
    pre_counter_R = counter_R;
    pre_counter_L = counter_L;   
    pre_R_rot=R_rot;
    pre_R_rot=R_rot;
}

//LED点灯プログラム
void led_lightning(float total_distance)
{
    int period_100mm =0;
    for (int i =0;i<500;i++)
    {
        if( total_distance >= (float)i*200 && total_distance < ((float)i+1)*200)
        {
            period_100mm = i;
            break;
        }
    }        
    if(period_100mm%2 == 1)
    {
        myled = 1;
        printf("LED IS [ON]\n");
        
    }else
    {
            myled = 0;
            printf("LED IS [OFF]\n");
    }
}
     

    
int main() {    
    AIN1.period(0.05);
    AIN2.period(0.05);
    BIN1.period(0.05);
    BIN2.period(0.05);
    
    enc_R.rise(&event_handler_R);
    enc_R.fall(&event_handler_R);
    enc_L.rise(&event_handler_L);
    enc_L.fall(&event_handler_L);
    
    machine_Stop(AIN1,AIN2,BIN1,BIN2);
    mode = STRAIGHT;
    re_ignition=no;
    
    while(1) {
        wait(delta);
        Check_reflecter(sensor_R,sensor_L,sensor_C);
        set_target_rot_and_duty_value();
        PID_Control();
        distance_calculate();
        led_lightning(total_distance);
    }
} 